<?php
/**
 * WPB Schedules
 *
 * Allows creation of monthly and weekly schedule tabs on admin and user pages
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpBSchedules' ) ) {

class WpBSchedules {

	/**
     * WP BASE Core + Front [+Admin] instance
     */
	protected $a = null;

	/**
     * Constructor
     */
	public function __construct() {
		$this->a = BASE();
	}

	/**
     * Add hooks
     */
	public function add_hooks(){
		add_action( 'app_menu_before_all', array( $this, 'add_menu' ), 20 ); 						// Called to add menu item
		add_action( 'wp_ajax_app_bookings_in_tooltip', array( $this, 'bookings_in_tooltip' ) );		// Show tooltips
		add_filter( 'app_weekly_calendar_cell_fill', array( $this, 'fill' ), 10, 2 );				// Fill weekly cell with booking info
		add_action( 'app_update_calendars', array( $this, 'update_cals' ) );
		add_filter( 'app_is_account_page', array( $this, 'is_account_page_filter' ) );
	}

	/**
     * Add admin menu
     */
	public function add_menu(){
		wpb_add_submenu_page('appointments', __('Schedules','wp-base'), __('Schedules','wp-base'), array(WPB_ADMIN_CAP,'manage_schedules'), "app_schedules", array($this,'render'));
	}

	/**
	 * Find requested worker id
	 * @param all: Force to select all workers instead of current user
	 * @since 2.0
	 */
	public function find_req_worker_id( ) {
		if ( $maybe_worker = $this->a->read_worker_id() )
			$user_id = $maybe_worker;
		else if ( ! empty( $_POST['app_worker_id'] ) && 'all' == $_POST['app_worker_id'] && ( empty($_GET['tab']) || 'daily' == $_GET['tab'] ) )
			$user_id = 'all';
		else if ( ! empty( $_POST['app_user_id'] ) )
			$user_id = $_POST['app_user_id'];
		else
			$user_id = get_current_user_id();

		return $user_id;
	}

	/**
	 * Dynamically show bookings in qtip content
	 * @since 2.0
	 */
	public function bookings_in_tooltip() {
		if ( empty( $_POST['app_val'] ) ) {
			wp_send_json( array( 'result' => __('Unexpected error','wp-base') ) );
		}

		if ( ! empty( $_POST['weekly'] ) ) {
			add_filter( 'app_time_slot_calculus_legacy', '__return_false' );
		}
		
		$slot	= new WpB_Slot( wpb_clean( $_POST['app_val'] ) );
		$start	= $slot->get_start();
		$end	= $slot->get_end();
		$worker = $slot->get_worker();
		$what	= apply_filters( 'app_list_tooltip_scope', ('all' === $slot->get_service() ? 'all' : 'worker'), $slot ); # For service=all show all bookings
		
		add_filter( 'app_list_skip_internal', '__return_false' );

		if ( ! empty( $_POST['weekly'] ) ) {
			$tooltip_status	= apply_filters( 'app_list_tooltip_status', 'completed,paid,confirmed,pending,running,reserved', 'weekly_schedule' );
			$columns		= apply_filters( 'app_list_tooltip_columns', 'id,service,client,worker,status', 'weekly_schedule' );
			$shortcode		= '[app_list _wide_coverage=1 user_id="'. $worker . '" status="'.$tooltip_status.'" what="'.$what.'" columns="'.$columns.'" order_by="start" _tablesorter="0" start="'.date("Y-m-d H:i:s",$start).'" end="'.date("Y-m-d H:i:s",$end).'" title="'.date_i18n($this->a->dt_format, $start).'"]';
		} else {
			$tooltip_status = apply_filters( 'app_list_tooltip_status', 'completed,paid,confirmed,pending,running,reserved', 'monthly_schedule' );
			$columns		= apply_filters( 'app_list_tooltip_columns', 'ID,service,client,worker,time,status', 'monthly_schedule' );
			$date			= date('Y-m-d', $start );
			$shortcode		= '[app_list _wide_coverage1 user_id="'.$worker.'" _as_tooltip="1" status="'.$tooltip_status.'" what="'.$what.'" columns="'.$columns.'" order_by="start" _tablesorter="0" start="'.$date.' 00:00" end="'.$date.' 23:59:59" title="'.date_i18n($this->a->date_format, $start).'"]';
		}
		
		$shortcode = apply_filters( 'app_list_tooltip_shortcode', $shortcode, $slot );

		wp_send_json( array( 'result' => do_shortcode( $shortcode ) ) );
	}

	/**
	 * Generate a set of default weekly shortcodes to be used in user pages
	 * @param count: How many weeks
	 * @param worker: Id of the worker. Can also be zero (unassigned provider)
	 * @param class: Class to add to the calendar
	 * @since 2.0
	 * @return array
	 */
	public function weekly_shortcodes( $count = 4, $worker = false, $class = '' ) {

		$worker		= false === $worker ? $this->find_req_worker_id() : $worker;
		$service 	= $this->a->get_min_max_service_by_worker( $worker );
		$class 		= $class . ( $count ? ' app-'.$count.'col' : '' );
		$add_class	= 1 == $count ? ' app-has-inline-cell' : '';
		$daily		= strpos( $class, 'app-schedule-daily' ) !== false ? 1 : 0;

		# Seasonal wh settings are not used for all case
		if ( 'all' === $worker ) {
			remove_filter( 'app_combine_alt', array( BASE('Annual'), 'combine_alt' ), 10 );
		}
		
		if ( $daily ) {
			$min_time = max( 15, $this->a->get_min_time() );
		} else {
			$min_time = 60;
		}		

		$weekly_shortcodes = array();

		for ( $add = 0; $add < $count; $add = $add + 1 ) {
			$weekly_shortcodes[] = $this->a->calendar_weekly(
				array(
				'title'				=> $daily ? "START" : "START_END",
				'logged'			=> 0,
				'worker'			=> $worker,
				'service'			=> isset($service->ID) ? $service->ID : 0,
				'add'				=> $add,
				'class'				=> $class. $add_class,
				'display'			=> 'with_break',
				'_inline'			=> 1 === $count ? 1 : '',
				'_force_min_time'	=> $min_time,
				'_admin'			=> 1,
				'_daily'			=> $daily,
				)
			);
		}

		return $weekly_shortcodes;
	}

	/**
	 * Generate a set of default monthly shortcodes to be used in user pages
	 * @param count: How many months
	 * @param worker: Id of the worker. Can also be zero (unassigned provider)
	 * @param class: Class to add to the calendar
	 * @since 2.0
	 * @return array
	 */
	public function monthly_shortcodes( $count = 3, $worker = false, $class = '' ) {

		$worker 	= false === $worker ? $this->find_req_worker_id() : $worker;
		$service 	= $this->a->get_min_max_service_by_worker( $worker );
		$service_id	= isset($service->ID) ? $service->ID : 0;
		$class 		= $class . ( $count ? ' app-'.$count.'col' : '' );

		if ( 'all' === $worker ) {
			remove_filter( 'app_combine_alt', array( BASE('Annual'), 'combine_alt' ), 10 );
		}
		
		$monthly_shortcodes = array();

		for ( $add = 0; $add < $count; $add = $add + 1 ) {
			$monthly_shortcodes[] = $this->a->calendar_monthly(
				array(
				'title'			=> "START",
				'logged'		=> 0,
				'service'		=> $service_id,
				'worker'		=> $worker,
				'add'			=> $add,
				'class'			=> $class,
				'_force_min_time'=> $this->a->is_daily( $service_id ) ? 720 : 60,
				'_no_timetable'	=> 1,
				'_preload'		=> 1,
				'_admin'		=> 1,
				)
			);
		}

		return $monthly_shortcodes;
	}

	/**
	 * Check if request is sent from an account page
	 * @since 3.4
	 * @return bool
	 */
	public function is_account_page(){
		if ( wpb_is_account_page() ) {
			return true;
		}
		
		if ( empty( $_POST['post_id'] ) ) {
			return false;
		}
		
		$post = get_post( $_POST['post_id'] );
		
		if ( empty( $post->post_content ) ) {
			return false;
		}
		
		return has_shortcode( $post->post_content, 'app_account' );
	}
	
	/**
	 * Check if request is sent from an account page
	 * @since 3.8
	 * @return bool
	 */
	public function is_account_page_filter( $is ){
		
		if ( empty( $_POST['post_id'] ) ) {
			return $is;
		}
		
		$post = get_post( $_POST['post_id'] );
		
		if ( empty( $post->post_content ) ) {
			return $is;
		}
		
		return has_shortcode( $post->post_content, 'app_account' ) ? true : $is;
	}
	
	/**
	 * Check if request is sent from BP Weekly Schedule page
	 * @since 3.4
	 * @return bool
	 */
	public function is_bp_weekly_page(){
		return ! empty( $_POST['bp_tab'] ) && strpos( $_POST['bp_tab'], 'weekly' ) !== false;
	}
	
	/**
	 * Check if request is sent from BP Schedule page
	 * @since 3.4
	 * @return bool
	 */
	public function is_bp_schedule_page(){
		return ! empty( $_POST['bp_tab'] ) && strpos( $_POST['bp_tab'], 'calendars' ) !== false;
	}

	/**
	 * Add reply for update_cals, i.e. update cals and next/prev
	 * @since 3.0
	 * @return array
	 */
	public function update_cals( $reply ) {
		
		if ( empty( $_POST['screen_base'] ) && !$this->is_bp_schedule_page() && !$this->is_bp_weekly_page() && !$this->is_account_page() ) {
			return;
		}			
		
		$step		= null;
		$result 	= $reply = array();
		$tab 		= $this->current_tab();
		$screen_base = isset( $_POST['screen_base'] ) ? $_POST['screen_base'] : '';
		
		if ( ( ! $tab || 'daily' == $tab ) && empty( $_REQUEST['app_worker_id'] ) ) {
			$_GET['app_worker_id'] = 'all';
		}
		
		if ( $screen_base == $this->a->app_name.'_page_app_schedules' || $screen_base == 'users_page_your_bookings' || $this->is_account_page() ) {
			#Schedules, user or account page
			
			switch ( $tab ) {
				case 'daily'	: 	$wscodes = $this->weekly_shortcodes(1, false, 'app-schedule-daily');
									$result[] = current( $wscodes );
									$step = 1; $unit = 'day';
									break;

				case 'weekly'	: 	$wscodes = $this->weekly_shortcodes(1);
									$result[] = current( $wscodes );
									$step = 1; $unit = 'week';
									break;

				case '4weeks'	: 	foreach( $this->weekly_shortcodes(4) as $scode ) {
										$result[] = $scode;
									}
									$step = 4; $unit = 'week';
									break;

				case 'monthly'	: 	$mscodes = $this->monthly_shortcodes(1);
									$result[] = current( $mscodes );
									$step = 1; $unit = 'month';
									break;

				case '3months'	: 	foreach( $this->monthly_shortcodes(3) as $scode ) {
										$result[] = $scode;
									}
									$step = 3; $unit = 'month';
									break;

				default	: 			$wscodes = $this->weekly_shortcodes(1, false, 'app-schedule-daily');
									$result[] = current( $wscodes );
									$step = 1; $unit = 'day';
									break;
			}
			
		} else if ( $this->is_bp_schedule_page() ) {
			// BP user page
			foreach ( $this->weekly_shortcodes(4, wpb_clean( $_POST['bp_displayed_user_id'] ), 'app_2column app-bp-profile') as $ws ) {
				$result[] = $ws;
			}

			foreach ( $this->monthly_shortcodes(2, wpb_clean( $_POST['bp_displayed_user_id'] ), 'app_2column app-monthly-daily app-bp-profile') as $ms ) {
				$result[] = $ms;
			}
			$step = 4; $unit = 'week';
			
		} else if ( $this->is_bp_weekly_page() ) {
			$wscodes = $this->weekly_shortcodes(1, wpb_clean( $_POST['bp_displayed_user_id'] ), 'app-bp-profile');
			$result[] = current( $wscodes );
			$step = 1; $unit = 'week';
		}
		
		if ( $step ) {
			$time = ! empty( $_REQUEST["app_timestamp"] ) ? intval( wpb_clean( $_REQUEST["app_timestamp"] ) ) : $this->a->_time;
			list( $prev, $prev_min, $ignore, $next, $next_max, $ignore ) = wpb_prev_next( $time, $step, $unit );

			if ( ('monthly' == $tab || '3months' == $tab) && $prev <= $prev_min ) {
				$prev = "hide";
			}

			if ( $next >= $next_max ) {
				$next = "hide";
			}

			$reply['prev'] = $prev;
			$reply['next'] = $next;
		}

		if ( ! empty( $result ) ) {
			$reply['html'] = $result;
		}

		die( json_encode( apply_filters( 'app_schedules_update_cals_reply', $reply ) ) );
	}
	
	/**
	 * Find tab to display
	 * @since 3.4
	 * @return string
	 */
	function current_tab(){
		return ! empty($_REQUEST['tab']) ? wpb_clean( $_REQUEST['tab'] ) : ( is_admin() ? 'daily' :'weekly' );
	}

	/**
	 * Fill the weekly calendar cell for daily and weekly
	 * @since 3.0
	 * @return array
	 */
	 function fill( $fill_arr, $slot ) {

		if ( ! $slot->calendar->is_inline() ) {
			return $fill_arr;
		}

		$fill		= '';
		$worker 	= $slot->get_worker();
		$service 	= $slot->get_service();
		
		add_filter( 'app_reserved_status_query', array( $this, 'reserved_status' ), 10, 2 );

		if ( 'all' === $service || ( !$this->a->get_nof_workers() && !$worker ) ) {
			$apps = $this->a->get_reserve_apps( date( "W", $slot->get_start() ) );
		} else if ( $worker ) {
			$apps = $this->a->get_daily_reserve_apps_by_worker( $slot->get_worker(), date( 'Y-m-d', $slot->get_start() ) );
		} else if ( $this->a->get_nof_workers() ) {
			$apps = $this->a->get_reserve_unassigned_apps( $slot->get_location(), $slot->get_service(), 0, date( "W", $slot->get_start() ) );
		} else {
			$apps = $this->a->get_daily_reserve_apps_by_service( $slot->get_service(), date( 'Y-m-d', $slot->get_start() ) );
		}
	
		if ( $apps ) {
			$unassigned_found	= false;
			$order			= null;
			$found 			= false;
			$short 			= null;
			$more 			= '';
			$status 		= '';
			$all			= 'all' === $service ? ' app-all' : '';
			$tab 			= $this->current_tab();
			$class_name_arr = array();
			$nof_workers	= $this->a->get_nof_workers();
			$worker_ids 	= array_filter( $this->a->get_worker_ids() );
			$href 			= admin_url( 'admin.php?page=app_bookings&status=all&stype=app_id&app_s=APP_ID' );
			$temp_part_short= 'daily' == $tab ? '<span class="app-cell-client-link">CLIENT</span>
								<span class="app-cell-dt">START_TIME-END_TIME</span>' :'<span class="app-cell-client-link">CLIENT</span>';
			$temp_part_long	=	$temp_part_short.				
								'<span class="app-cell-service">SERVICE</span>
								<span class="app-cell-status">STATUS</span>';
								
			$title			= preg_replace( '%<span(.*?)>(.*?)</span>%', '$2 ●', $temp_part_long );
			
			
			foreach ( $apps as $app ) {
				$slot_start	= $slot->get_start();
				$slot_end	= $slot->get_end();
				$start		= strtotime( $app->start );
				$end		= strtotime( $app->end );
				$status		= ' '. $app->status;
				$worker		= $app->worker;
							  
				if ( $end < $slot_start )	# Too early
					continue;

				if ( $start > $slot_end )	# Too late
					continue;
				
				if ( $nof_workers && ! $app->worker ) {				
					if ( $unassigned_found ) {
						continue;
					}
					
					$unassigned_found = true;
					$class_name_arr[] = 'app-unassigned';
					$temp_part_long   = $title = '';
				}				
				
				if ( 'daily' != $tab || ( wpb_strtotime( $app->end ) - wpb_strtotime( $app->start ) <= 15 * 60 ) ) {
					$short = true;
				} else {
					$short = false;
				}

				$template 	= ($all ? '<a href="'.$href.'" class="app-cell-edit" title="WORKER ●'.$title.'"><span class="dashicons dashicons-edit"></span><span class="app-cell-worker">WORKER_SHORT</span></a>' : ''). 
							  ($short ? $temp_part_short : $temp_part_long);
				$content 	= $this->a->_replace( wpb_setting( 'schedule_content', $template ), $app, 'subject' );
				$top 		= $bottom = 0;
				$height 	= 58; // In px

				if ( $start >= $slot_start && $end <= $slot_end ) {	# Completely inside
					$found			= true;
					$top			= $height * number_format( ($start - $slot_start)/36, 2 )/100;
					$bottom			= $height * number_format( ($slot_end - $end)/36, 2 )/100;
					$height			= $height - $top - $bottom;
					if ( isset( $this->order_arr[ $app->ID ] ) ) {
						unset( $this->order_arr[ $app->ID ] );
					}
				}

				if ( $start < $slot_start && $slot_start < $end && $end <= $slot_end ) {	# Started before, ends inside or at cce
					$found			= true;
					$bottom			= $height * number_format( ($slot_end - $end)/36, 2 )/100;
					$height			= $height - $bottom - 4;
					$more			= 'border-top:none;padding-top:0;';
					$content		= '';
					$class_name_arr[] = 'app_no_border_top';
					if ( isset( $this->order_arr[ $app->ID ] ) ) {
						unset( $this->order_arr[ $app->ID ] );
					}
				}

				if ( $slot_start <= $start && $start < $slot_end && $slot_end < $end ) {	# Starts inside or just at ccs and ends outside
					$found		= true;
					$top 		= $height * number_format( ($start - $slot_start)/36, 2 )/100;
					$height		= $height - $top - 4;
					$more		= 'border-bottom:none;padding-bottom:0px;';
					$class_name_arr[] = 'app_no_border_bottom';
				}

				if ( $start < $slot_start && $end > $slot_end ) {	# Starts outside and ends outside
					$found		= true;
					$more		= 'border-bottom:none;border-top:none;padding-bottom:0;padding-top:0;';
					$height		= 61;
					$content	= '';
					$class_name_arr[] = 'app_no_border_top';
					$class_name_arr[] = 'app_no_border_bottom';
				}
				
				if ( $found ) {
					$found = false;

					if ( $nof_workers ) {
						$order	= $app->worker && wpb_is_worker( $app->worker ) 
								  ? (int)array_search( $app->worker, $worker_ids ) 
								  : min( 7, max( $worker_ids ) + 1 );
					} else {
						if ( isset( $this->order_arr[ $app->ID ] ) )
							$order = $this->order_arr[ $app->ID ];
						else {	
							$order = $this->order_arr[ $app->ID ] = null !== $order ? ( $order + 1 ) : 0;
						}
					}

					$class_name_arr[] = 'has_inline_appointment';
					if ( $all ) {
						$class_name_arr[] = 'app-all app-no-title';
					}
					if ( $short ) {
						$class_name_arr[] = 'app-inline-cell-short';
					}
					$fill .= '<div data-order="'.$order.'" data-worker_id="'.$worker.'" class="app-cell-inline'.$status.$all.'" style="'.$more.'height:'.$height.'px;top:'.$top.'px;bottom:'.$bottom.'px;">' . $content . '</div>';
				}
				
			}

}

		if ( $fill ) {
			$fill_arr['fill'] = $fill;
			$fill_arr['class_name'] = ! empty( $class_name_arr ) ? implode( ' ', array_unique( $class_name_arr ) ) : '';
		}

		return $fill_arr;
	}
	
	/**
	 * Add completed to reserved status
	 * @since 3.4
	 */
	public function reserved_status( $query, $context ) {
		return $query . " OR status='completed' ";
	}

	/**
	 * Generate HTML for schedules
	 * @since 2.0
	 */
	public function render() {

		wpb_admin_access_check( 'manage_schedules' );

		$tab 				= $this->current_tab();
		
		if ( (!$tab || 'daily' == $tab) && empty($_GET['app_worker_id']) ) {
			$_GET['app_worker_id'] = 'all';
		}
		$worker_id 			= ! empty($_GET['app_worker_id']) ? wpb_clean( $_GET['app_worker_id'] ) : $this->find_req_worker_id( );
		$workers 			= $this->a->get_workers();
		$def_worker 		= $this->a->get_def_wid();
		$def_worker_name	= $this->a->get_worker_name( $def_worker, false );
		$href 				= wpb_add_query_arg( array( 'app_worker_id' => false, 'rand' => 1 ) ) .'&app_worker_id=';
?>
	<div class="wrap app-page">
	<h2 class="app-dashicons-before dashicons-calendar-alt"><?php echo __('Booking Schedules','wp-base'); ?></h2>
		<div class="app-submit">
			<span class="app-list-for">
			<?php _e('List for', 'wp-base')?>
			</span>
			<select class="app_ms" data-buttonwidth="300" onchange="if (this.value) window.location.href='<?php echo $href ?>'+this.value" class="app_select_workers_schedule" id="app_worker_id" name="app_worker_id">
				<option ><?php _e('Select provider', 'wp-base') ?></option>
				<?php if ( !$tab || 'daily' == $tab ) { ?>
					<option <?php selected( 'all', $worker_id ); ?> value="all"><?php _e('All', 'wp-base') ?></option>
				<?php } ?>
				<option <?php selected( $worker_id, $def_worker ); ?> value="<?php echo $def_worker ?>"><?php printf( __('Business Rep. (%s)', 'wp-base'), $def_worker_name ) ?></option>
				<?php
				foreach ( (array)$workers as $worker ) {
					# Do not show default worker a second time
					if ( $def_worker == $worker->ID )
						continue;

					echo '<option value="'.$worker->ID.'" '.selected( $worker_id, $worker->ID, false ).'>';
					echo $this->a->get_worker_name( $worker->ID, false );
					echo '</option>';
				}
				?>
			</select>
		</div>

		<h3 class="nav-tab-wrapper">
		<?php

		$tabs = array(
			'daily'      	=> __('Daily', 'wp-base'),
			'weekly'      	=> __('Weekly', 'wp-base'),
			'4weeks' 	    => __('4 Weeks', 'wp-base'),
			'monthly' 	    => __('Monthly', 'wp-base'),
			'3months'		=> __('3 Months', 'wp-base'),
		);

		$tabhtml = array();


		$tabs = apply_filters( 'app_schedules_tabs', $tabs );

		$class = ( 'daily' == $tab ) ? ' nav-tab-active' : '';

		foreach ( $tabs as $stub => $title ) {
			$class = ( $stub == $tab ) ? ' nav-tab-active' : '';
			if ( isset( $_GET['app_worker_id'] ) )
				$href = wpb_add_query_arg( array( 'app_worker_id' => wpb_clean( $_GET['app_worker_id'] ) ), admin_url( 'admin.php?page=app_schedules&amp;tab=' . $stub ) );
			else
				$href = admin_url( 'admin.php?page=app_schedules&amp;tab=' . $stub );
			$tabhtml[] = '	<a href="' . $href . '" class="nav-tab'.$class.'" id="app_tab_'.$stub.'">'.$title.'</a>';
		}

		echo implode( "\n", $tabhtml );
		?>
		</h3>
		<div class="clear"></div>
		<div id="poststuff" class="metabox-holder">
			<div class="postbox">
				<div class="inside">
				<?php
				# all is only allowed in daily tab
				if ( is_numeric( $worker_id ) || ( 'all' == $worker_id && 'daily' == $tab ) ) {

					switch( $tab ) {
						case 'daily'	:
							echo $this->a->pagination( array( 'select_date' => 1, 'disable_legend' => 1, 'unit' => 'day', 'step' => 1, 'history' => 1 ) );
							$wscodes = $this->weekly_shortcodes(1, false, 'app-schedule-daily');
							echo current( $wscodes );
							echo $this->a->pagination( array( 'select_date'=>0, 'unit'=>'day', 'step'=>1, 'history' => 1 ) );
							break;

						case 'weekly'	:
							echo $this->a->pagination( array( 'select_date'=>1, 'disable_legend'=>1, 'history' => 1 ) );
							$wscodes = $this->weekly_shortcodes(1);
							echo current( $wscodes );
							echo $this->a->pagination( array( 'select_date'=>0, 'history' => 1 ) );
							break;

						case '4weeks':
							echo $this->a->pagination( array( 'select_date'=>1, 'disable_legend' => 1, 'step' => 4, 'history' => 1 ) );
							foreach( $this->weekly_shortcodes(4) as $scode ) {
								echo $scode;
							}
							echo '<div style="clear:both"></div>';
							echo $this->a->pagination( array( 'select_date'=>0, 'step'=>4, 'history' => 1 ) );
							break;

						case 'monthly':
							echo $this->a->pagination( array( 'select_date'=>1, 'disable_legend' => 1, 'unit' => 'month', 'step' => 1 ) );
							$mscodes = $this->monthly_shortcodes(1);
							echo current( $mscodes );
							echo $this->a->pagination( array( 'select_date'=>0, 'unit'=>'month', 'step'=>1 ) );
							break;

						case '3months':
							echo $this->a->pagination( array( 'select_date'=>1, 'disable_legend'=>1, 'unit' => 'month', 'step' => 3 ) );
							foreach( $this->monthly_shortcodes(3) as $scode ) {
								echo $scode;
							}
							echo '<div style="clear:both"></div>';
							echo $this->a->pagination( array( 'select_date'=>0, 'unit' => 'month', 'step' => 3 ) );
							break;
					}
				} else {
					_e( 'Select provider','wp-base');
				}

				?>
				</div>
			</div>
		</div>
	</div>
<?php
	}

}
	BASE('Schedules')->add_hooks();
}