<?php
/**
 * WPB Notices
 *
 * Manages displaying notices
 *
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists('WpBNotices') ) {

class WpBNotices {

	/**
     * Page IDs for Make an Appointment and List of Bookings pages
	 * created by Quick Start
	 * @Integer
     */
	public $created_page_id			= 0;
	public $created_account_page_id = 0;

	/**
     * WP BASE Core + Front [+Admin] instance
     */
	protected $a = null;

	/**
     * Notice buffer
     */
	private $system_notice = array();

	/**
     * Class of the notice
     */
	private $notice_class = 'updated';

	/**
     * Delay of dialog to open (in msecs)
     */
	private $delay = 0;

	/**
     * Display alert or open dialog
     */
	private $alert = 0;

	/**
     * Dialog vars
     */
	private $dialog_text, $dialog_title, $refresh_url, $icon, $app_status;

	/**
     * Constructor
     */
	public function __construct() {
		$this->a = BASE();
	}

	/**
     * Add actions and filters
     */
	public function add_hooks() {
		add_action( 'wp_footer', array( $this, 'display_notice' ) );
		add_action( 'login_footer', array( $this, 'display_notice' ) );
		add_action( 'admin_notices', array( $this, 'admin_notices' ), 100 );
		add_action( 'admin_notices', array( $this, 'display_admin_notice' ), 120 );
		add_action( 'wp_ajax_dismiss', array( $this, 'dismiss' ) );
	}

	/**
	 * Set a message to class variable
	 * @since 3.0
	 */
	public function set_notice( $m, $class = 'updated', $delay = 0 ) {
		$this->system_notice[] = $m;
		$this->notice_class = $class;
		$this->delay = max( $this->delay, $delay );
	}

	/**
	 * Retrieve class message variable
	 * @since 3.0
	 */
	public function get_notice() {
		return ! empty( $this->system_notice ) ? $this->system_notice : array();
	}

	/**
	 * Set page id
	 * @since 3.0
	 */
	public function set_page_id( $id ) {
		$this->created_page_id = $id;
	}

	/**
	 * Set list page id
	 * @since 3.0
	 */
	public function set_account_page_id( $id ) {
		$this->created_account_page_id = $id;
	}

	/**
	 * Add js message for front end either with js or with jQuery dialog
	 * @since 2.0
	 */
	public function front( $context = false, $class = '', $delay = 0 ) {
		$title = $this->a->get_text('notice');
		$icon = $class == 'error' ? 'alert' :'info';
		if ( 'error' == $context ) {
			$title = $this->a->get_text('error_short');
			$icon = 'alert';
			$text = $this->a->get_text('error');
		} else if ( 'saved' == $context ) {
			$text = $this->a->get_text('saved');
		} else if ( 'updated' == $context ) {
			$text = $this->a->get_text('updated');
		} else if ( 'cancelled' == $context ) {
			$text = $this->a->get_text('cancelled');
		} else if ( 'too_late' == $context ) {
			$text = $this->a->get_text('too_late');
			$icon = 'alert';
		} else if ( 'not_possible' == $context ) {
			$text = $this->a->get_text('not_possible');
			$icon = 'alert';
		} else if ( 'edit_disabled' == $context ) {
			$text = $this->a->get_text('edit_disabled');
			$icon = 'alert';
		} else if ( $this->a->get_text( $context ) && $context != $this->a->get_text( $context ) ) {
			$text = $this->a->get_text( $context );
		} else if ( $context ) {
			$text = $context;
		} else {
			$text = $this->a->get_notice() ? implode( '\n', $this->a->get_notice() ) : '';
		}

		if ( ! wpb_is_admin() && ! wpb_is_login_page() && apply_filters( 'app_can_use_dialog', true ) ) {
			do_action( 'app_load_assets' );
			add_action( 'wp_footer', array( $this, 'footer' ) );

			$this->dialog_text	= $text;
			$this->dialog_title = $title;
			$this->delay 		= max( $this->delay, $delay );
			$this->icon 		= $icon;
			$this->refresh_url 	= -1;

		} else {
			$this->alert = 1;
			$this->dialog_text = $text;
			$this->set_notice( $text );
		}
	}

	/**
	 * Open the dialog with confirmation text as standalone, e.g. after gateway return or by email confirmation
	 * @since 3.0
	 * @return none
	 */
	public function open_confirmation_dialog( $app ) {
		do_action( 'app_load_assets' );
		add_action( 'wp_footer', array( $this, 'footer' ) );

		if ( 'paid' != $app->status && 'confirmed' != $app->status ) {
			$this->alert		= 1;
			$this->dialog_text	= wpb_dialog_title( $app ) ?: $this->a->get_text('appointment_received');
		} else {
			$this->dialog_text	= wpb_dialog_text( $app );
			$this->dialog_title	= wpb_dialog_title( $app );
			$this->refresh_url	= ('paid' == $app->status || 'confirmed' == $app->status)
								  ? wpb_refresh_url( home_url() )
								  : remove_query_arg( array( 'app_gw_success', 'app_hash' ) );
		}

		$this->app_status = $app->status;
	}

	/**
	 * Add html to the front to be displayed as dialog
	 * @uses wp_footer action
	 * @since 3.0
	 */
	public function footer(){
		if ( ! $this->dialog_text || is_admin() ) {
			return;
		}

		$h = '<script type="text/html" class="app-notice-template"';

		foreach( array( 'dialog_title', 'dialog_text', 'refresh_url', 'alert', 'icon', 'delay', 'app_status' ) as $var ) {
			$h .= ' data-'.$var.'="'.esc_attr( $this->{$var} ) .'"';
		}

		$h .= '>'. esc_html( html_entity_decode( $this->dialog_text ) ) .'</script>'."\r\n";

		echo $h;
	}

	/**
	 * Displays a message as javascript alert
	 * @since 2.0
	 */
	public function display_notice( ) {
		$m = $this->get_notice();

		if ( ! empty( $m ) ) {
			$text = implode( '\n', array_map( 'esc_js', $m ) );
			?>
			<script type="text/javascript">
				alert("<?php echo $text ?>");
			</script>
			<?php
		}
	}

	/**
	 * Prints an admin message
	 * @since 3.0
	 */
	public function display_admin_notice( ) {
		$m = $this->get_notice();
		$class = $this->notice_class && is_string( $this->notice_class ) ? esc_attr( $this->notice_class ) : 'updated';

		if ( ! empty( $m ) ) {
			$text = implode( '<br/>', array_map( 'trim', $m ) );
			echo '<div class="'.$class.' app-dismiss app-notice is-dismissable"><p><b>[WP BASE]</b> '. $text .
			'</p><a class="notice-dismiss" data-what="general" title="'.esc_attr( __('Dismiss this notice', 'wp-base' ) ).'" href="javascript:void(0)"></a></div>';
		}
	}

	/**
	 * Renders an infobox
	 * @param $visible		string		Visible content
	 * @param $hidden		string		Hidden content
	 * @param $class		string		Optional class name which button will open. If left empty: app-instructions
	 * @since 2.0
	 */
	public function infobox( $visible, $hidden, $class = '' ) {
		?>
		<div class="postbox app-infobox">
			<div class="inside">
				<div class="app-infobox-col1">
				<?php if ( $hidden ): ?>
				<a href="javascript:void(0)" data-boxname="<?php echo $class ?>" class="info-button" title="<?php _e('Click to toggle details', 'wp-base')?>">
					<img src="<?php echo WPB_PLUGIN_URL ."/images/help.png" ?>" alt="<?php _e('Information','wp-base')?>" />
				</a>
				<?php else: ?>
					<img src="<?php echo WPB_PLUGIN_URL ."/images/help.png" ?>" alt="<?php _e('Information','wp-base')?>" />
				<?php endif; ?>
				</div>

				<div class="app-infobox-col2">
					<span class="description">
					<?php echo $visible; ?>
					</span>
					<?php if ( $hidden ): ?>
						<span class="description">
						<?php _e( '<strong>For details please click the "i" icon.</strong>', 'wp-base'); ?>
						</span>
						<div class="app-instructions <?php echo $class ?>" style="display:none">
							<?php if ( is_array( $hidden ) ): ?>
							<ul class="app-info-list">
								<?php
									foreach ( $hidden as $key => $line ) {
										if ( preg_match( '%<b>(.*?)</b>%', $line, $match ) )
											$id = 'id="'. sanitize_file_name( strtolower($match[1]) ) .'"';
										else
											$id = '';

										echo '<li '.$id.'>'. $line . '</li>';
									}
								?>
							</ul>
							<?php else: ?>
								<span class="description">
								<?php echo $hidden; ?>
								</span>
							<?php endif; ?>
						</div>

					<?php endif; ?>
				</div>
				<div style="clear:both"></div>
			</div>
		</div>
		<?php

		do_action( 'app_admin_after_info' );
	}

	/**
	 * Check if there are more than one shortcodes for certain shortcode types
	 * @since 1.0.5
	 * @return bool
	 */
	public function has_duplicate_shortcode( $post_id ) {
		$post = get_post( $post_id );

		if ( empty( $post->post_content ) ) {
			return false;
		}

		$cont = $post->post_content;
		$desc = $post->post_excerpt;

		if ( strpos( $cont, '[app_' ) !== false ) {
			if ( substr_count( $cont, '[app_locations' ) > 1 || substr_count( $cont, '[app_services' ) > 1
				|| substr_count( $cont, '[app_workers' ) > 1 || substr_count( $cont, '[app_confirmation' ) > 1
				|| substr_count( $cont, '[app_login' ) > 1 || substr_count( $cont, '[app_manage' ) > 1 ) {
				return true;
			}

			if ( substr_count( $cont, '[app_book' ) > 0 && substr_count( $desc, '[app_book' ) > 0 ) {
				return true;
			}

		}

		return false;
	}

	/**
	 *	Warn admin if no services defined or duration is wrong or ...
	 */
	public function admin_notices() {

		if ( ! current_user_can( WPB_ADMIN_CAP ) ) {
			return;
		}

		global $current_user;
		$r = false;

		if ( ! $this->a->get_db_version() ) {
			_wpb_reactivate_required();
			$r = true;
		}
		
		// Warn if Zoom needs oauth setup
		if ( BASE('Zoom') && 'yes' == wpb_setting('zoom_enable') && 'individual' != wpb_setting('zoom_worker_account') && 'oauth' != wpb_setting('zoom_app_type') && ! $this->is_dismissed('zoom_oauth') ) {
			echo '<div class="app-notice notice is-dismissable"><p>' .
				__('<b>[WP BASE]</b> App type JWT has been deprecated by Zoom and it will not be available after 1st September 2023. Please reconfigure Online Meetings addon for "Server-to-Server Oauth"', 'wp-base' ) .
			'</p><a class="notice-dismiss" data-what="zoom_oauth" title="'.__('Dismiss this notice for this session', 'wp-base').'" href="javascript:void(0)"></a>'.
			'</div>';
			$r = true;
		}		

		// Warn if there is no Bus. Rep.
		if ( ! get_user_by( 'ID', wpb_setting('default_worker') ) ) {
			echo '<div class="error"><p>' . sprintf(
				__( '<b>[WP BASE]</b> Please assign a Business Representative on %s page.', 'wp-base' ),
				current_user_can( WPB_ADMIN_CAP )
				? '<a href="'. admin_url( 'admin.php?page=app_settings&tab=user' ).'">'. __( 'Users', 'wp-base' ) .'</a>'
				: __( 'Global Settings > Users', 'wp-base' )
			).'</p></div>';
		}

		// Warn if Cancel link is used, but cancellation is not enabled
		$msgs = wpb_setting('confirmation_message').' '. wpb_setting('reminder_message').' '. wpb_setting('pending_message');

		if ( 'yes' != wpb_setting('allow_cancel') && ! $this->is_dismissed('cancel') && false !== strpos( do_shortcode($msgs), 'CANCEL') ) {
			echo '<div class="app-notice notice is-dismissable"><p>' .
				__('<b>[WP BASE]</b> You have used CANCEL placeholder in at least one of the email message templates, but you did not enable cancellation. With these settings, cancel link will not be displayed in the emails. Please correct "Allow client cancel own appointments" setting on Global Settings tab or remove CANCEL placeholder from the email message template.', 'wp-base' ) .
			'</p><a class="notice-dismiss" data-what="cancel" title="'.__('Dismiss this notice for this session', 'wp-base').'" href="javascript:void(0)"></a>'.
			'</div>';
			$r = true;
		}

		// Warn if payment required, but no payment method is selected
		$active = wpb_active_gateways();
		if ( ! $this->is_dismissed('payment') && 'yes' == wpb_setting('payment_required') && empty( $active ) && !(BASE('WooCommerce') && 'yes' == wpb_setting('wc_enabled'))  ) {
			echo '<div class="app-notice notice is-dismissable"><p>' .
				__('<b>[WP BASE]</b> You have selected "Payment Required", but no payment method is active. With these settings, it may not be possible to complete a booking on the front end.', 'wp-base' ) .
			'</p><a class="notice-dismiss" data-what="payment" title="'.__('Dismiss this notice for this session', 'wp-base').'" href="javascript:void(0)"></a>'.
			'</div>';
			$r = true;
		}

		// Check for duplicate shortcodes for a visited page
		if ( ! empty( $_GET['post'] ) && !$this->is_dismissed('duplicate') && $this->has_duplicate_shortcode( $_GET['post'] ) ) {
			echo '<div class="app-notice notice is-dismissable"><p>' .
			__('<b>[WP BASE]</b> More than one instance of services, service providers, confirmation or login shortcodes on the same page may cause problems.</p>', 'wp-base' ).
			'</p><a class="notice-dismiss" data-what="duplicate" title="'.__('Dismiss this notice for this session', 'wp-base').'" href="javascript:void(0)"></a>'.
			'</div>';
		}

		// Make the rest of the checks only when a service OR time base have been changed
		if ( ! get_user_meta( $current_user->ID, 'app_service_check_needed' ) ) {
			return;
		}

		if ( ! $results = $this->a->get_services() ) {
			echo '<div class="error"><p>' .
				__('<b>[WP BASE]</b> You must define at least once service.', 'wp-base' ) .
			'</p></div>';
			$r = true;
		} else {
			// Check if any location is unassigned
			if ( $this->a->get_nof_locations() ) {

				$locations = $this->a->get_locations();
				foreach ( $locations as $location ) {
					if ( !$this->is_dismissed('location') && !$this->a->get_services_by_location( $location->ID ) ) {
						echo '<div class="app-notice notice is-dismissable"><p>' .
						__('<b>[WP BASE]</b> One of your locations does not have a service assigned. Delete locations you are not using.', 'wp-base' ) .
						'</p><a class="notice-dismiss" data-what="location" title="'.__('Dismiss this notice for this session', 'wp-base').'" href="javascript:void(0)"></a>'.
						'</div>';
						$r = true;
						break;
					}
				}
			}

			// Check services
			$max_duration = max( wpb_service_duration_selections() );

			foreach ( $results as $result ) {
				if ( $result->duration < $this->a->get_min_time() ) {
					echo '<div class="error"><p>' .
						__('<b>[WP BASE]</b> One of your services has a duration smaller than time base. Please visit Services tab and after making your corrections save new settings or try changing Time Base setting as "Auto".', 'wp-base' ) .
					'</p></div>';
					$r = true;
					break;
				}
				if ( $result->duration % $this->a->get_min_time() != 0 || $result->break_time % $this->a->get_min_time() != 0 || $result->padding % $this->a->get_min_time() != 0 ) {
					echo '<div class="error"><p>' .
						__('<b>[WP BASE]</b> One of your service duration or padding is not divisible by the time base. Please visit Services tab and after making your corrections save new settings or try changing Time Base setting as "Auto".', 'wp-base' ) .
					'</p></div>';
					$r = true;
					break;
				}

				if ( !$this->a->is_package($result->ID) && ( $result->padding + $result->duration + $result->break_time ) > $max_duration ) {
					echo '<div class="error"><p>' .
						sprintf( __('<b>[WP BASE]</b> One of your services has a duration (plus paddings) greater than 24 hours (without Advanced Features addon) or greater than %s (with Advanced Features addon).', 'wp-base' ), wpb_format_duration( $max_duration ) ) .
					'</p></div>';
					$r = true;
					break;
				}

				if ( !$this->is_dismissed('no_sp') && !$this->a->get_capacity( $result->ID ) && !$this->a->is_package( $result->ID ) ) {
					echo '<div class="app-notice notice is-dismissable"><p>' .
						sprintf( __('<b>[WP BASE]</b> One of your services does not have a service provider assigned (and capacity is not set). Delete %s you are not using.', 'wp-base'), '<a href="'.admin_url('admin.php?page=app_business&tab=services').'" title="'.__('Click to access Services settings','wp-base').'">'. __('services','wp-base').'</a>') .
						'</p><a class="notice-dismiss" data-what="no_sp" title="'.__('Dismiss this notice for this session', 'wp-base').'" href="javascript:void(0)"></a>'.
					'</p></div>';
					$r = true;
					break;
				}
				if ( BASE('Categories') && BASE('Categories')->get_nof_categories() && !$result->categories && !$this->is_dismissed('no_category') ) {
					echo '<div class="app-notice notice is-dismissable"><p>' .
						sprintf( __('<b>[WP BASE]</b> One of your services does not have a category assigned. Such services may not be selectable on the front end. %s', 'wp-base'), '<a href="'.admin_url('admin.php?page=app_business&tab=services#tabs-1').'" title="'.__('Click to access Services settings','wp-base').'">'. __('Click to fix.','wp-base').'</a>') .
						'</p><a class="notice-dismiss" data-what="no_category" title="'.__('Dismiss this notice for this session', 'wp-base').'" href="javascript:void(0)"></a>'.
					'</p></div>';
					$r = true;
					break;
				}

				if ( ! $r ) {
					delete_user_meta( $current_user->ID, 'app_service_check_needed' );
				}
			}
		}

		return $r;
	}

	/**
	 *	Dismiss warning messages for the current user for the session
	 *	@since 1.1.7
	 */
	public function dismiss() {
		global $current_user;
		$what = isset( $_POST['what'] ) ? wpb_clean( $_POST['what'] ) : false;

		if ( ! $what ) {
			return;
		}

		if ( update_user_meta( $current_user->ID, 'app_dismiss_'.$what, $this->a->_time ) ) {
			wp_send_json_success( $what );
		}
	}

	/**
	 *	Check if a message is dismissed and not expired
	 *	@since 3.0
	 */
	public function is_dismissed( $what ) {
		global $current_user;
		if ( ! $when = get_user_meta( $current_user->ID, 'app_dismiss_'.$what, true ) ) {
			return false;
		}

		if ( $this->a->_time - (int)$when > 60*60*24*7 ) {
			#expired
			delete_user_meta( $current_user->ID, 'app_dismiss_'.$what );
			return false;
		}

		return true;
	}

	/**
	 * Prints "A new page created" message on top of Admin page
	 * @since 2.0
	 */
	public function page_created( ) {
		echo '<div class="updated app-dismiss"><p><b>[WP BASE]</b> '. sprintf( __('A new booking page created. %s','wp-base'), '<a href="'. get_permalink( $this->created_page_id ). '">'.__('View Page','wp-base') .'</a>'  ).
		'</p><a class="notice-dismiss" data-what="general" title="'.__('Dismiss this notice', 'wp-base').'" href="javascript:void(0)"></a></div>';
	}

	/**
	 * Prints "A new page created" message on top of Admin page
	 * @since 2.0
	 */
	public function page_created_list( ) {
		echo '<div class="updated app-dismiss"><p><b>[WP BASE]</b> '. sprintf( __('A new My Account page created. %s','wp-base'), '<a href="'. get_permalink( $this->created_account_page_id ). '">'.__('View Page','wp-base') .'</a>'  ).
		'</p><a class="notice-dismiss" data-what="general" title="'.__('Dismiss this notice', 'wp-base').'" href="javascript:void(0)"></a></div>';
	}

	/**
	 * Prints "Demo" message on top of Admin page
	 * @since 2.0
	 */
	public function demo( ) {
		echo '<div class="error"><p><b>[WP BASE]</b> '. __('In Demo mode settings cannot be changed.','wp-base').'</p></div>';
	}

	/**
	 * Prints "Restored" message on top of Admin page
	 * @since 2.0
	 */
	public function restored( ) {
		echo '<div class="updated app-dismiss"><p><b>[WP BASE]</b> '. __('Settings have been set to their default values.','wp-base').
		'</p><a class="notice-dismiss" data-what="general" title="'.__('Dismiss this notice', 'wp-base').'" href="javascript:void(0)"></a></div>';
	}

	/**
	 * Prints DB reset message on top of Admin page
	 * @since 2.0
	 */
	public function reset_db( ) {
		echo '<div class="updated app-dismiss"><p><b>[WP BASE]</b> '. __('Database tables have been reset.','wp-base').
		'</p><a class="notice-dismiss" data-what="general" title="'.__('Dismiss this notice', 'wp-base').'" href="javascript:void(0)"></a></div>';
	}

	/**
	 * Prints "saved" message on top of Admin page
	 */
	public function saved( ) {
		echo '<div class="updated app-nag-saved app-notice is-dismissable"><p><b>[WP BASE]</b> '. __('Settings saved.','wp-base').
		'</p><a class="notice-dismiss" data-what="general" title="'.__('Dismiss this notice', 'wp-base').'" href="javascript:void(0)"></a></div>';
	}

	/**
	 * Prints "deleted" message on top of Admin page
	 */
	public function deleted( ) {
		echo '<div class="updated app-nag-saved app-notice is-dismissable"><p><b>[WP BASE]</b> '. __('Selected record(s) deleted.','wp-base').
		'</p><a class="notice-dismiss" data-what="general" title="'.__('Dismiss this notice', 'wp-base').'" href="javascript:void(0)"></a></div>';
	}

	/**
	 * Prints "updated" message on top of Admin page
	 */
	public function updated( ) {
		echo '<div class="updated app-nag-saved app-notice is-dismissable"><p><b>[WP BASE]</b> '. __('Selected record(s) updated.','wp-base').
		'</p><a class="notice-dismiss" data-what="general" title="'.__('Dismiss this notice', 'wp-base').'" href="javascript:void(0)"></a></div>';
	}

	/**
	 * Prints anauth message on top of Admin page
	 */
	public function unauthorised( ) {
		echo '<div class="error app-notice is-dismissable"><p><b>[WP BASE]</b> '. __('You are not authorised to do this.','wp-base').
		'</p><a class="notice-dismiss" data-what="general" title="'.__('Dismiss this notice', 'wp-base').'" href="javascript:void(0)"></a></div>';
	}

	/**
	 * Prints upload error message on top of Admin page
	 * @since 2.0
	 */
	public function reset_error( ) {
		echo '<div class="error app-notice is-dismissable"><p><b>[WP BASE]</b> '. __('An error occurred during resetting. Check log file to see if details of the error has been recorded.','wp-base').
		'</p><a class="notice-dismiss" data-what="general" title="'.__('Dismiss this notice', 'wp-base').'" href="javascript:void(0)"></a></div>';
	}

	/**
	 * Prints "Test email successful" message on top of Admin page
	 * @since 2.0
	 */
	public function test_email_success( ) {
		echo '<div class="updated app-dismiss"><p><b>[WP BASE]</b> '. __('Test email has been successfully sent. Please wait a few minutes and check first admin email account.','wp-base').
		'</p><a class="notice-dismiss" data-what="general" title="'.__('Dismiss this notice', 'wp-base').'" href="javascript:void(0)"></a></div>';
	}

	/**
	 * Prints "Test email failed" message on top of Admin page
	 * @since 2.0
	 */
	public function test_email_fail( ) {
		echo '<div class="error app-dismiss"><p><b>[WP BASE]</b> '. __('Sending the test email has failed. Please check if you can actually send any emails from this website. For example, logout and try to receive an email using "Lost your password?" link of WordPress admin login page.','wp-base').
		'</p><a class="notice-dismiss" data-what="general" title="'.__('Dismiss this notice', 'wp-base').'" href="javascript:void(0)"></a></div>';
	}

	/**
	 * wp_remote_get is not working message
	 * @since 2.0
	 */
	public function remote_get_error( ) {
		echo '<div class="error app-dismiss"><p><b>[WP BASE]</b> '. __('This website cannot use wp_remote_get function to retrieve data from itself. Preload function cannot be enabled.','wp-base').
		'</p><a class="notice-dismiss" data-what="general" title="'.__('Dismiss this notice', 'wp-base').'" href="javascript:void(0)"></a></div>';
	}

	/**
	 * Display Lite version notice
	 * @since 2.0
	 */
	public function lite() {
	?>
		<div id="poststuff" class="metabox-holder">
		<span class="description app-btm"><?php _e('This option is only available in the Developer version.', 'wp-base' ) ?></span>
		</div>
	<?php
	}

}

	BASE('Notices')->add_hooks();
}
