<?php
/**
 * WPB Holidays
 *
 * Handles holiday settings, definitions and methods for services and service providers
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( !class_exists( 'WpBHolidays' ) ) {

class WpBHolidays {

	/**
     * WP BASE Core + Front [+Admin] instance
     */
	protected $a = null;

	/**
     * Name of the tab/title
     */
	private static $name;

	/**
     * Constructor
     */
	public function __construct() {
		$this->a = BASE();
		self::$name = array( 'holidays' => __( 'Holidays', 'wp-base' ) );
	}

	/**
     * Add hooks
     */
	public function add_hooks(){
		add_filter( 'admin_title', array( $this, 'admin_title' ), 10, 2 );
		add_filter( 'appointments_business_tabs', array( $this, 'add_tab' ), 14 ); 		// Add tab
		add_action( 'app_business_holidays_tab', array( $this, 'render_tab' ) );		// Display HTML settings on Business Settings
		add_action( 'app_activated', array( $this, 'update_346' ), 10, 2 );
		add_action( 'app_installed', array( $this, 'update_346' ), 10, 2 );
		add_action( 'app_save_settings', array( $this, 'save_holidays' ), 12 );
		add_action( 'app_save_account_settings', array( $this, 'save_holidays' ), 12 );
	}
	
	/**
     * Update to support setting holidays per location: Save previous holiday settings
     */
	public function update_346( $installed_version, $previous_version ){
	
		if ( version_compare( $previous_version, '0.0', '>' ) && version_compare( $previous_version, '3.4.6', '<' ) ) {
			$ops = $this->a->get_business_options('holidays');
			if ( empty( $ops ) || !is_array( $ops ) ) {
				return;
			}
			
			if ( '0' === (string)key( $ops ) ) {
				return;
			}
			
			$new_ops[0] = $ops; # Add for location 0
			$this->a->update_business_options( $new_ops, 'holidays' );
		}
	}

	/**
     * Get current or selected location ID
	 * @since 3.0
	 * @return integer
     */
	public function get_location( $slot = null ) {
		$loc = apply_filters( 'app_wh_location', 0, $slot );
		return apply_filters( 'app_holidays_location', $loc, $slot );
	}
	
	/**
	 * Check if today (or a 24 hour time period between ccs-cce) is holiday for a worker or service
	 * @param $slot		WpB_Slot object
	 * @return bool
	 */
	public function is_holiday( $slot, $subject = 'worker' ) {
		if ( 'worker' == $subject && wpb_ignore_bis_rep_wh() ) {
			return $this->is_holiday( $slot, 'service' );
		}

		$who		= 'worker' === $subject ? $slot->get_worker() : $slot->get_service();
		$location	= $this->get_location( $slot );
		$h			= $this->a->get_business_options('holidays');
		$h_set		= isset( $h[$location][$subject][$who] ) ? $h[$location][$subject][$who] : false;
		if ( !$h_set )
			return false;

		$slot_start	= $slot->get_start();
		$slot_end	= $slot->get_end();
		$offset 	= 'worker' == $subject &&  $who ? $this->a->get_client_offset( $slot_start, $who ) : 0; // Service provider local time is taken into account

		$holidays = explode( ',', $h_set );
		sort( $holidays ); 
		foreach ( $holidays as $holiday ) {
			$h_start = strtotime( $holiday, $this->a->_time + $offset );
			$h_end  = $h_start + DAY_IN_SECONDS;
			if ( $slot_end > $h_start && $slot_start < $h_end ) {
				return true;
			}
			
			if ( $h_end > $slot_end + DAY_IN_SECONDS ) {
				break;
			}
		}

		return false;
	}

	/**
	 * Check if today is holiday for a worker/service in Wh domain
	 * @return bool
	 */
	public function is_holiday_wh( $start, $end, $who, $year_week_no, $subject = 'worker', $location = 0 ) {
		$year = substr( $year_week_no, 0, 4 );
		$week = substr( $year_week_no, 4 );
		$ts = wpb_week2time( $week, $year );
		// Shift 1 sec
		$slot_start	= BASE('WH')->from( $start + 0.004, $ts );
		$slot_end 	= BASE('WH')->from( $end - 0.004, $ts );
		$slot 		= new WpB_Slot( new WpB_Calendar( $location, $who, $who ), $slot_start, $slot_end );

		return $this->is_holiday( $slot, $subject );
	}

/****************************************************
* Methods for admin
*****************************************************
*/

	/**
	 * Change admin SEO title
	 * @since 3.8.0
	 */
	public function admin_title( $admin_title, $title ) {
		if ( ! empty( $_GET['tab'] ) && key( self::$name ) == $_GET['tab'] ) {
			return str_replace( $title, current( self::$name ), $admin_title );
		} else {
			return $admin_title;
		}
	}
	
	/**
	 *	Add tab
	 */
	public function add_tab( $tabs ) {
		if ( wpb_admin_access_check( 'manage_working_hours', false ) ) {
			$tabs = array_merge( $tabs, self::$name );
		}

		return $tabs;
	}

	/**
	 * Display HTML codes
	 * @param $profileuser_id: If this is called by a user from his profile
	 */
	public function render_tab( $profileuser_id = false, $bp = false ) {

		wpb_admin_access_check( ($profileuser_id ? 'manage_own_work_hours' : 'manage_working_hours') );
	?>
<div id="poststuff" class="metabox-holder">
		<?php

		wpb_infobox( __('Here you can define your holidays during which related service or provider will be unavailable. Note that every service and provider has own holidays. To set holidays, select related service or provider from List for pulldown, select days to mark as holiday on the yearly calendar and click Save. You can browse through years using &larr; and &rarr; buttons on the calendar.', 'wp-base') );

		$workers 			= $this->a->get_workers();
		$def_worker 		= $this->a->get_def_wid();
		$def_worker_name 	= $this->a->get_worker_name( $def_worker, false );
		$incl_def_worker	= current_user_can( WPB_ADMIN_CAP ) || ( $profileuser_id && $profileuser_id == $def_worker );
		$subj_sel 			= isset($_POST['app_select_subject']) 
							  ? wpb_clean( $_POST['app_select_subject'] ) 
							  : ($incl_def_worker && ( ! $profileuser_id || ! wpb_is_worker( $profileuser_id ) ) ? 'worker|'.$def_worker : 'worker|'.$profileuser_id);
		$subj_sel	 		= apply_filters( 'app_holidays_default_selected', $subj_sel, $profileuser_id );

	?>
	<div class="postbox app-prpl">
		<div class="app-submit app_2column">
			<div class="app-mt">
				<form class="app-flex" method="post">
					<span class="app-list-for"><?php _e('List for', 'wp-base')?></span>
					<select class="app_ms" data-buttonwidth="250" name="app_select_subject">
				<?php if ( class_exists( 'WpBSP' ) ) { ?>
						<optgroup label="<?php _e('Service Providers','wp-base') ?>" class="optgroup_worker">
				<?php }
					if ( !in_array( $def_worker, (array)$this->a->get_worker_ids() ) && $incl_def_worker ) {
				?>
						<option value="worker|<?php echo $def_worker ?>"><?php printf( __('Business Rep. (%s)', 'wp-base'), $def_worker_name) ?></option>
				<?php
					}
					if ( $workers ) {
						if ( $profileuser_id ) {
							$s = 'worker|'.$profileuser_id == $subj_sel ? " selected='selected'" : '';
							echo '<option value="worker|'.$profileuser_id.'"'.$s.'>' . $this->a->get_worker_name( $profileuser_id, false ) . '</option>';
						} else {
							foreach ( $workers as $worker ) {
								$s = 'worker|'.$worker->ID == $subj_sel ? " selected='selected'" : '';
								echo '<option value="worker|'.$worker->ID.'"'.$s.'>' . $this->a->get_worker_name( $worker->ID, false ) . '</option>';
							}
						}
					}
					if ( class_exists( 'WpBSP' ) ) { ?>
						</optgroup>
				<?php } ?>
						<optgroup label="<?php _e('Services','wp-base') ?>" class="optgroup_service">
				<?php
					if ( $profileuser_id ) {
						$services = $this->a->get_services_owned_by( $profileuser_id, -1 );
					} else {
						$services = $this->a->get_services();
					}

					if ( $services ) {
						foreach ( $services as $service ) {
							$s = 'service|'. $service->ID == $subj_sel ? " selected='selected'" : '';
							$disabled = $this->a->is_package( $service->ID ) ? ' disabled="disabled"' : '';
							echo '<option value="service|'.$service->ID.'"'. $s .$disabled.'>' . $this->a->get_service_name( $service->ID ) . '</option>';
						}
					}
					?>
						</optgroup>

					</select>
					<button class="ui-widget ui-button ui-state-default ui-corner-all ui-shadow app-ml10"><?php _e('Show','wp-base') ?></button>
				</form>
			</div>
		</div>
		<div class="app-submit app-4col">
		<?php do_action( 'app_wh_admin_submit_mid' ) ?>
		</div>
		<div class="app-submit app-4col">
		<?php do_action( 'app_wh_admin_submit' ) ?>
		</div>
		<div style="clear:both"></div>
	</div>

	<?php
		list( $subject, $who ) = explode( '|', $subj_sel );
		echo $this->draw( $who, $subject, $bp );
	?>
</div>
	<?php
	}

	/**
	 *	Render holidays
	 */
	 function draw( $who, $subject = 'worker', $bp = false ) {

		$year = date("Y", $this->a->_time );

		switch( $subject ) {
			case 'worker':	$pre_text = ($who == $this->a->get_def_wid()) ? __('Business Rep','wp-base') : $this->a->get_text('provider');
							$whose =  $pre_text.': '. $this->a->get_worker_name( $who );
							break;
			case 'service':	$whose = $this->a->get_text('service') .': '. $this->a->get_service_name( $who );
							break;
		}

		$options	= $this->a->get_business_options();
		$location	= $this->get_location();
		$holidays	= isset( $options['holidays'][$location][$subject][$who] ) ? $options['holidays'][$location][$subject][$who]: '';

		$r  = apply_filters( 'app_admin_holidays_before', '' );
		$r .= "<form class='app-form app_wh_annual_form' method='post' action='".wpb_add_query_arg( null, null )."'>";
		$r .= '<p class="submit">
			<input type="submit" class="button-primary" value="'. __('Save Holidays', 'wp-base') .'" />
			</p>';

		$r .= '<div class="postbox">';
		$r .= '<h3 class="hndle"><span>'. sprintf( __('Holidays of %s','wp-base'), $whose ). '</span></h3>';
		$r .= '<div class="inside">';

		$h = $holidays ? htmlspecialchars( wp_json_encode( explode( ',', $holidays ) ) ) : '';

		$r .= '<div id="full-year" class="'.($bp ? 'bp ': '').'box" data-add_dates="'.$h.'"></div>';

		$r .= '<input type="hidden" name="holidays" id="altField" value="'.esc_attr( $holidays ).'" />';

		$r .= '<div style="clear:both"></div>';

		if ( !$bp ) {
			$r .= '<fieldset class="app-mt"><label>';
			$r .= '<span class="title app-mr5 app-b">'.__('Days picked per click','wp-base').'</span>';
			$r .= '<span class="input-text-wrap app-mr5">';
			$r .= '<input type="text" class="app-days-picked app-no-save-alert app-50" value="1" />';
			$r .= '</span>';
			$r .= '<span class="description app-btm">';
			$r .= __('You can pick more than one day at once by entering desired number of days. There is no need to save. Then as you click a day, following days will also be selected/deselected accordingly. Note: This setting is not saved to the database and defaults to 1.', 'wp-base' );
			$r .= '</span>';
			$r .= "</label></fieldset>";
		}

		$r .= "</div></div>";

		if ( $bp )
			$r .='<input type="hidden" name="app_bp_settings_user" value="'.$who.'">';

		$r .= '<input type="hidden" name="app_location_id" value="'.$this->get_location().'" />
			<input type="hidden" name="action_app" value="save_holidays" />
			<input type="hidden" name="app_select_subject" value="'.$subject.'|'.$who.'" />
			<input type="hidden" name="who" value="'.$who.'" />
			<input type="hidden" name="subject" value="'.$subject.'" />
			<input type="hidden" name="app_nonce" value="'.wp_create_nonce( 'update_app_settings' ).'" />
			<p class="submit">
			<input type="submit" class="button-primary" value="'. __('Save Holidays', 'wp-base') .'" />
			</p>';

		$r .= '</form>';

		return $r;
	}

	/**
     * Save holidays settings
     */
	public function save_holidays( $profileuser_id = false ) {

		if ( empty( $_POST['action_app'] ) || 'save_holidays' != $_POST['action_app'] || ! isset( $_POST['holidays'] ) ) {
			return;
		}

		$who		= $profileuser_id ? $profileuser_id : wpb_clean( $_POST['who'] );
		$subject	= $profileuser_id ? 'worker' : wpb_clean( $_POST['subject'] );
		$holidays	= wpb_clean( $_POST['holidays'] );
		$location	= $this->get_location();

		$b_options = $this->a->get_business_options();
		$b_options['holidays'][$location][$subject][$who] = wpb_sanitize_commas( $holidays );

		if ( $this->a->update_business_options( $b_options ) ) {
			wpb_flush_cache();
			wpb_notice( 'saved' );
		}
	}

}

	BASE('Holidays')->add_hooks();
}