<?php
/**
 * Internal functions
 *
 * Manages internal affairs. Has little or no use for ordinary user.
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

 /**
 * Find current admin side screen ID
 * @param $add_tab	bool	Whether add $_GET['tab']
 * @uses get_current_screen()
 * @since 3.0
 * @return string
 */
function wpb_get_current_screen_id( $add_tab = false ) {
	if ( function_exists( 'get_current_screen' ) ) {
		$screen = get_current_screen();

		$tab = $add_tab && !empty( $_GET['tab'] ) ? '_' . wpb_clean( $_GET['tab'] ) : '';

		if ( isset( $screen->id ) ) {
			return $screen->id . $tab;
		}
	}

	return '';
}

/**
 * Find id of the product post being used
 * Includes ajaxed requests made from a page
 * @param $include_cookie	bool	Whether include cookiie
 * @return integer
 * @since 2.0
 */
function wpb_find_post_id( $include_cookie = true ) {
	global $wp_the_query;
	
	if ( is_callable( array( $wp_the_query, 'get_queried_object' ) ) ) {
		$current_object = $wp_the_query->get_queried_object();
	}
	
	if ( ! empty( $current_object ) ) {
		$post_id = ! empty( $current_object->ID ) ? $current_object->ID : 0;
	} else {
		$post_id = null;
	}	

	$post = get_post( $post_id );

	if ( ! empty( $_POST['post_id'] ) ) {
		$post_id = wpb_clean( $_POST['post_id'] );
	} else if ( isset( $post->ID ) ) {
		$post_id = $post->ID;
	} else if ( $include_cookie ) {
		$post_id = wpb_get_session_val( 'app_post_id', 0 );
	} else {
		$post_id = 0;
	}

	return $post_id;
}

/**
 * Check if cache shall be used - DEPRECATED
 * In addition to the setting, Quotas and Timezones prevent usage of cache
 * @param $calendar		object		WpB_Calendar object
 * @since 3.0
 * @until 3.9.1.2
 * @return bool
 */
function wpb_use_cache( $calendar = null ){
	return false;
}

/**
 * Flush WP BASE cache by invalidating cache prefix
 * Optionally also preload cache: Recreate a fresh cache
 * @param admin: make a flush cache only for admin, e.g do not delete Caching plugins
 * @since 3.0
 * @return none
 */
function wpb_flush_cache( $admin = false ){

	delete_transient( 'wpbase_revenue' );
	delete_transient( 'wpbase_sales' );

	# Increment prefix, so WP BASE caches using 'wp_base_cache_prefix' prefix are invalid
	# Idea from http://tollmanz.com/invalidation-schemes/
	wp_cache_incr( 'wp_base_cache_prefix' );

	if ( $admin ) {
		do_action( 'app_flush_cache_admin' );
	} else {
		do_action( 'app_flush_cache' );
	}
}

/**
 * Call pages remotely so cache can be preloaded - DEPRECATED
 * @param $pages	string		Optional comma delimited page IDs to call. If left empty, settings will be used
 * @since 3.0
 * @until 3.9.1.2
 * @return bool|null	null if settings do not allow, true on success, false if loopback fails
 */
function wpb_maybe_preload_cache( $pages = '' ){
	return;
}

/**
 * Get cache prefix
 * @since 3.0
 * @return string
 */
function wpb_cache_prefix(){
	$prefix = wp_cache_get( 'wp_base_cache_prefix' );

	if ( false === $prefix ) {
		$prefix = 1;
		wp_cache_set( 'wp_base_cache_prefix', $prefix );
	}

	return 'wp_base_cache_' . $prefix . '_';
}

/**
 * Get headers to read file data
 * @since 3.0
 * @return array
 */
function wpb_default_headers(){
	return array(
		'Name'			=> 'Plugin Name',
		'Author'		=> 'Author',
		'Description' 	=> 'Description',
		'Plugin URI' 	=> 'Plugin URI',
		'Version' 		=> 'Version',
		'Detail' 		=> 'Detail',
		'Class' 		=> 'Class Name',
		'ID' 			=> 'ID',
		'Category' 		=> 'Category',
		'Requires' 		=> 'WPB requires',
	);
}

/**
 * Instantiate and register an addon
 * Closures are intentional and prevent their related filters to be removed
 * @since 3.0
 *
 * @param $identifier	string		Part of name of the addon class (class name: WpB + $identifier)
 * @param $file			string		File name including absolute path
 * @return none
 */
function wpb_addon_run( $identifier, $file ) {
	if ( ! BASE( $identifier ) ) {
		return;
	}

	$data = get_file_data( $file, wpb_default_headers(), 'plugin' );

	if ( ! empty( $data['Requires'] ) && defined( 'WPB_VERSION' ) && version_compare( WPB_VERSION, $data['Requires'], '<' ) ) {
		if ( is_admin() && function_exists( 'wpb_notice' ) ) {
			wpb_notice( sprintf( __( '%s requires WP BASE version %s or higher', 'wp-base' ), $data['Name'], $data['Requires'] ), 'error' );
		}

		return;
	}

	add_filter( 'app_active_extensions',
		function( $result ) use ( $identifier, $file ){
			if ( is_array( $result ) ) {
				$result[$file] = $identifier;
			} else {
				$result = array( $file => $identifier );
			}

			return $result;
		},
	10, 1 );

	BASE( $identifier )->add_hooks();

	$addon_path = realpath( WPB_ADDON_DIR );

	if ( ! wpb_is_admin() || ( $addon_path && strpos( $file, $addon_path ) !== false ) ) {
		return;
	}

	add_action( 'admin_init', function() use ( $file, $data ){
		if ( ! class_exists( 'WPB_Plugin_Updater' ) ) {
			include_once( WPBASE_PLUGIN_DIR . "/includes/lib/plugin-updater.php" );
		}

		new WPB_Plugin_Updater( WPBASE_URL, $file, array(
			'version'   => $data['Version'],
			'license'   => trim( get_option( 'wpb_license_key_'. $data['ID'] ) ),
			'item_id' 	=> $data['ID'],
			'Category' 	=> $data['Category'],
			'author'    => $data['Author'],
		) );
	});

	add_filter( 'plugin_action_links_'. plugin_basename( $file ),
		function( $links ) use ( $file ){
			return array_merge( $links, array( '<a target="_blank" href="'.WPBASE_URL.'knowledge-base/'.str_replace( 'payment-gateway-', '', pathinfo($file, PATHINFO_FILENAME) ).'/">'. __('Help', 'wp-base' ) . '</a>' ) );
		},
	10, 1 );
}

/**
 * Disable identical addon
 * An addon in WP Plugin format will be loaded first and prevent same filename in includes/addons folder to be loaded again
 */
function wpb_disable_addon( $result, $addon, $file ) {
	if ( $result || strpos( str_replace( '\\', '/', dirname( $file ) ), 'includes/addons' ) !== false ) {
		return $result;
	}

	$filename = pathinfo( $file, PATHINFO_FILENAME );

	if ( $addon == $filename ) {
		return sprintf( __( 'Disabled by WP BASE %s plugin', 'wp-base' ), ucwords( str_replace('-', ' ', $filename ) ) );
	}

	return $result;
}

/**
 * Load and Run Main class (of a payment gateway addon)
 * @return bool		Success or failure
 */
function wpb_load_main( $file ) {
	$basename	= str_replace( 'payment-gateway-', '', basename( $file, '.php' ) );
	$main_file	= plugin_dir_path( $file ) . $basename . '/'. $basename . '.php';

	if ( file_exists( $main_file ) ) {
		include_once( $main_file );
	}

	global $app_gateway_plugins, $app_gateway_active_plugins;

	if ( empty( $app_gateway_plugins[$basename] ) ) {
		return false;
	}

	$options = BASE()->get_options();

	if ( isset( $_POST['mp']['gateways']['allowed'] ) ) {
		$options['gateways']['allowed'] = wpb_clean( $_POST['mp']['gateways']['allowed'] );
	} else if ( isset( $_POST['mp'] ) ) {
		# Blank array if no checkboxes
		$options['gateways']['allowed'] = array();
	}

	$allowed = !empty( $options['gateways']['allowed'] ) ? (array)$options['gateways']['allowed'] : array();

	if ( !in_array( $basename, $allowed ) )
		return false;

	$class_name = $app_gateway_plugins[$basename][0];

	if ( class_exists( $class_name ) ) {
		$app_gateway_active_plugins[] = new $class_name;
		return true;
	}

	return false;
}

/**
 * Check if a particular gateway plugin is active
 * @param $gateway: Name of the gateway to check, i.e. paypal-standard
 * @Since 3.5.10
 */
function wpb_is_gateway_active( $gateway ) {

	foreach ( wpb_active_gateways() as $code => $addon ) {
		if ( $gateway == $addon->plugin_name ) {
			return true;
		}
	}

	return false;
}

/**
 * Get a list of active gateways
 * @return array
 */
function wpb_active_gateways(){
	global $app_gateway_active_plugins;
	return apply_filters( 'app_active_gateways', !empty( $app_gateway_active_plugins ) ? (array)$app_gateway_active_plugins : array() );
}

/**
 * Register gateway plugin/addon class
 *
 * @param string $class_name - the case sensitive name of plugin class
 * @param string $plugin_name - the sanitized private name for plugin
 * @param string $admin_name - pretty name of gateway, for the admin side.
 */
function wpb_register_gateway_plugin( $class_name, $plugin_name, $admin_name ) {
	global $app_gateway_plugins;

	if ( ! is_array( $app_gateway_plugins ) ) {
		$app_gateway_plugins = array();
	}

	if ( class_exists( $class_name ) ) {
		$app_gateway_plugins[ $plugin_name ] = array( $class_name, $admin_name );
	} else {
		return false;
	}
}

/**
 * Check if any Credit Card processor is active
 * PayPal is counted as well
 * @return bool
 */
function wpb_can_use_cc(){
	if ( ! $gateways = wp_list_pluck( wpb_active_gateways(), 'admin_name', 'plugin_name' ) ) {
		return false;
	}

	unset( $gateways['pay-later'] );
	unset( $gateways['manual-payments'] );

	return (bool)array_filter( $gateways );
}

/**
 * Sanitizes url against XSS attacks
 * @param $a, $b, $c	mixed		Same as add_query_arg
 * @since 2.0
 * @return string
 */
function wpb_add_query_arg( $a, $b = false, $c = false ) {
	if ( is_array( $a ) ) {
		if ( false === $b && false === $c )
			$b = esc_url_raw( $_SERVER['REQUEST_URI'] );

		return add_query_arg( $a, $b );
	} else {
		if ( $c === false )
			$c = esc_url_raw( $_SERVER['REQUEST_URI'] );

		return add_query_arg( $a, $b, $c );
	}
}

/**
 * Find difference of two associated arrays recursively
 * http://nl3.php.net/manual/en/function.array-diff-assoc.php#111675
 * @return array
 */
function wpb_array_diff_assoc_recursive($array1, $array2) {
	$difference = array();

	foreach ( $array1 as $key => $value ) {
		if( is_array($value) ) {
			if( ! isset($array2[$key]) || !is_array($array2[$key]) ) {
				$difference[$key] = $value;
			} else {
				$new_diff = wpb_array_diff_assoc_recursive($value, $array2[$key]);
				if( !empty($new_diff) )
					$difference[$key] = $new_diff;
			}
		} else if( !array_key_exists($key,$array2) || $array2[$key] !== $value ) {
			$difference[$key] = $value;
		}
	}

	return $difference;
}

/**
 * Remove duplicate email messages by app ID
 * @param $messages		array	array of objects or array of arrays
 * @return array
 */
function wpb_array_unique_by_ID( $messages ) {
	if ( !is_array( $messages ) || empty( $messages ) )
		return false;
	$idlist = array();

	$result = $messages;
	foreach ( $messages as $key => $message ) {
		if ( isset( $message['ID'] ) ) {
			if ( in_array( $message['ID'], $idlist ) )
				unset( $result[$key] );
			else
				$idlist[] = $message['ID'];
		} else if ( isset( $message->ID ) ) {
			if ( in_array( $message->ID, $idlist ) )
				unset( $result[$key] );
			else
				$idlist[] = $message->ID;
		}
	}
	return $result;
}

/**
 * Arranges days array acc. to start of week, e.g 1234567 (Week starting with Monday)
 * @param days: input array
 * @param prepend: What to add as first element
 * @pram nod: If number of days (true) or name of days (false)
 * @return array
 */
function wpb_arrange( $days, $prepend, $nod = false ) {
	if ( BASE()->start_of_week ) {
		for ( $n = 1; $n <= BASE()->start_of_week; $n++ ) {
			array_push( $days, array_shift( $days ) );
		}
		// Fix for displaying past days; apply only for number of days
		if ( $nod ) {
			$first = false;
			$temp = array();
			foreach ( $days as $key => $day ) {
				if ( !$first )
					$first = $day; // Save the first day
				if ( $day < $first )
					$temp[$key] = $day + 7; // Latter days should be higher than the first day
				else
					$temp[$key] = $day;
			}
			$days = $temp;
		}
	}
	if ( false !== $prepend )
		array_unshift( $days, $prepend );

	return $days;
}

/**
 * Find timestamp of the first moment of a given week defined with wpb_time2week function
 * @since 2.0
 * @return string
 */
function wpb_week2time( $week_no, $year = false ) {
	if ( false === $year ) {
		$year = date( "Y", BASE()->_time );
	}

	// http://www.php.net/manual/en/function.strftime.php#77489
	$first_day = 1 + ( 7 + BASE()->start_of_week - wpb_strftime( "%w", mktime( 0, 0, 0, 1, 1, $year ) ) )%7;
	$first_date = mktime( 0, 0, 0, 1, $first_day, $year );

	return $first_date + ($week_no-1)*7*24*3600;
}

/**
 * Find week number for a given timestamp
 * This number is used only internally: It may not comply with ISO8601
 * @since 2.0
 * @return string
 */
function wpb_time2week( $timestamp = false ) {

	$start_of_week = BASE()->start_of_week;

	if ( false === $timestamp ) {
		$timestamp = BASE()->_time;
	}

	if ( 1 == $start_of_week ) {
		return intval( wpb_strftime( "%W", $timestamp ) );
	} else if ( "0" === (string)$start_of_week ) {
		return intval( wpb_strftime( "%U", $timestamp ) );
	} else {
		$year 		= date( "Y", $timestamp );
		$first_day 	= 1 + ( 7 + $start_of_week - wpb_strftime( "%w", mktime( 0, 0, 0, 1, 1, $year ) ) )%7;
		$first_date = mktime( 0, 0, 0, 1, $first_day, $year );

		$first_day_sunday 	= 1 + ( 7 - wpb_strftime( "%w", mktime( 0, 0, 0, 1, 1, $year ) ) )%7;
		$first_date_sunday 	= mktime( 0, 0, 0, 1, $first_day_sunday, $year );

		$add = $first_date_sunday > $first_date ? 1 : 0;

		return wpb_strftime( "%U", $timestamp ) + $add;
	}
}

/**
 * Find number of weeks in a year
 * This number is used only internally: It does not comply with ISO8601
 * @since 2.0
 * @return integer
 */
function wpb_nof_weeks( $year ) {

	$start_of_week = BASE()->start_of_week;

	$first_day	= 1 + ( 7 + $start_of_week - wpb_strftime( "%w", mktime( 0, 0, 0, 1, 1, $year ) ) )%7;
	$first_date = mktime( 0, 0, 0, 1, $first_day, $year );

	$first_day_ny 	= 1 + ( 7 + $start_of_week - wpb_strftime( "%w", mktime( 0, 0, 0, 1, 1, $year + 1) ) )%7;
	$first_date_ny	= mktime( 0, 0, 0, 1, $first_day_ny, $year+1 );

	return intval( ( $first_date_ny - $first_date )/(7*24*3600) );
}

/**
 * Locale-formatted strftime using \IntlDateFormatter (PHP 8.1 compatible)
 * This provides a cross-platform alternative to strftime() for when it will be removed from PHP.
 * Note that output can be slightly different between libc sprintf and this function as it is using ICU.
 *
 * Usage:
 * use function \PHP81_BC\strftime;
 * echo strftime('%A %e %B %Y %X', new \DateTime('2021-09-28 00:00:00'), 'fr_FR');
 *
 * Original use:
 * \setlocale('fr_FR.UTF-8', LC_TIME);
 * echo \strftime('%A %e %B %Y %X', strtotime('2021-09-28 00:00:00'));
 *
 * https://gist.github.com/bohwaz/42fc223031e2b2dd2585aab159a20f30
 *
 * @param  string $format Date format
 * @param  integer|string|DateTime $timestamp Timestamp
 * @return string
 * @author BohwaZ <https://bohwaz.net/>
 * @since 3.8.6
 */
function wpb_strftime($format, $timestamp = null, $locale = null)
{

	if ( version_compare( PHP_VERSION, '8.1.0' ) < 0 ) {
		return strftime( $format, $timestamp );
	}

	if (null === $timestamp) {
		$timestamp = new \DateTime;
	}
	elseif (is_numeric($timestamp)) {
		$timestamp = date_create('@' . $timestamp);

		if ($timestamp) {
			$timestamp->setTimezone(new \DateTimezone(date_default_timezone_get()));
		}
	}
	elseif (is_string($timestamp)) {
		$timestamp = date_create($timestamp);
	}

	if (!($timestamp instanceof \DateTimeInterface)) {
		throw new \InvalidArgumentException('$timestamp argument is neither a valid UNIX timestamp, a valid date-time string or a DateTime object.');
	}

	$locale = substr((string) $locale, 0, 5);

	$intl_formats = [
		'%a' => 'EEE',	// An abbreviated textual representation of the day	Sun through Sat
		'%A' => 'EEEE',	// A full textual representation of the day	Sunday through Saturday
		'%b' => 'MMM',	// Abbreviated month name, based on the locale	Jan through Dec
		'%B' => 'MMMM',	// Full month name, based on the locale	January through December
		'%h' => 'MMM',	// Abbreviated month name, based on the locale (an alias of %b)	Jan through Dec
	];

	$intl_formatter = function (\DateTimeInterface $timestamp, string $format) use ($intl_formats, $locale) {
		$tz = $timestamp->getTimezone();
		$date_type = \IntlDateFormatter::FULL;
		$time_type = \IntlDateFormatter::FULL;
		$pattern = '';

		// %c = Preferred date and time stamp based on locale
		// Example: Tue Feb 5 00:45:10 2009 for February 5, 2009 at 12:45:10 AM
		if ($format == '%c') {
			$date_type = \IntlDateFormatter::LONG;
			$time_type = \IntlDateFormatter::SHORT;
		}
		// %x = Preferred date representation based on locale, without the time
		// Example: 02/05/09 for February 5, 2009
		elseif ($format == '%x') {
			$date_type = \IntlDateFormatter::SHORT;
			$time_type = \IntlDateFormatter::NONE;
		}
		// Localized time format
		elseif ($format == '%X') {
			$date_type = \IntlDateFormatter::NONE;
			$time_type = \IntlDateFormatter::MEDIUM;
		}
		else {
			$pattern = $intl_formats[$format];
		}

		return (new \IntlDateFormatter($locale, $date_type, $time_type, $tz, null, $pattern))->format($timestamp);
	};

	// Same order as https://www.php.net/manual/en/function.strftime.php
	$translation_table = [
		// Day
		'%a' => $intl_formatter,
		'%A' => $intl_formatter,
		'%d' => 'd',
		'%e' => function ($timestamp) {
			return sprintf('% 2u', $timestamp->format('j'));
		},
		'%j' => function ($timestamp) {
			// Day number in year, 001 to 366
			return sprintf('%03d', $timestamp->format('z')+1);
		},
		'%u' => 'N',
		'%w' => 'w',

		// Week
		'%U' => function ($timestamp) {
			// Number of weeks between date and first Sunday of year
			$day = new \DateTime(sprintf('%d-01 Sunday', $timestamp->format('Y')));
			return sprintf('%02u', 1 + ($timestamp->format('z') - $day->format('z')) / 7);
		},
		'%V' => 'W',
		'%W' => function ($timestamp) {
			// Number of weeks between date and first Monday of year
			$day = new \DateTime(sprintf('%d-01 Monday', $timestamp->format('Y')));
			return sprintf('%02u', 1 + ($timestamp->format('z') - $day->format('z')) / 7);
		},

		// Month
		'%b' => $intl_formatter,
		'%B' => $intl_formatter,
		'%h' => $intl_formatter,
		'%m' => 'm',

		// Year
		'%C' => function ($timestamp) {
			// Century (-1): 19 for 20th century
			return floor($timestamp->format('Y') / 100);
		},
		'%g' => function ($timestamp) {
			return substr($timestamp->format('o'), -2);
		},
		'%G' => 'o',
		'%y' => 'y',
		'%Y' => 'Y',

		// Time
		'%H' => 'H',
		'%k' => function ($timestamp) {
			return sprintf('% 2u', $timestamp->format('G'));
		},
		'%I' => 'h',
		'%l' => function ($timestamp) {
			return sprintf('% 2u', $timestamp->format('g'));
		},
		'%M' => 'i',
		'%p' => 'A', // AM PM (this is reversed on purpose!)
		'%P' => 'a', // am pm
		'%r' => 'h:i:s A', // %I:%M:%S %p
		'%R' => 'H:i', // %H:%M
		'%S' => 's',
		'%T' => 'H:i:s', // %H:%M:%S
		'%X' => $intl_formatter, // Preferred time representation based on locale, without the date

		// Timezone
		'%z' => 'O',
		'%Z' => 'T',

		// Time and Date Stamps
		'%c' => $intl_formatter,
		'%D' => 'm/d/Y',
		'%F' => 'Y-m-d',
		'%s' => 'U',
		'%x' => $intl_formatter,
	];

	$out = preg_replace_callback('/(?<!%)(%[a-zA-Z])/', function ($match) use ($translation_table, $timestamp) {
		if ($match[1] == '%n') {
			return "\n";
		}
		elseif ($match[1] == '%t') {
			return "\t";
		}

		if (!isset($translation_table[$match[1]])) {
			throw new \InvalidArgumentException(sprintf('Format "%s" is unknown in time format', $match[1]));
		}

		$replace = $translation_table[$match[1]];

		if (is_string($replace)) {
			return $timestamp->format($replace);
		}
		else {
			return $replace($timestamp, $match[1]);
		}
	}, $format);

	$out = str_replace('%%', '%', $out);
	return $out;
}

/*
 * Matches each symbol of PHP date format standard
 * with jQuery equivalent codeword
 * http://stackoverflow.com/a/16725290
 * @author Tristan Jahier
 */
function wpb_convert_dt_format( $php_format )
{
	$SYMBOLS_MATCHING = array(
		// Day
		'd' => 'dd',
		'D' => 'D',
		'j' => 'd',
		'l' => 'DD',
		'N' => '',
		'S' => '',
		'w' => '',
		'z' => 'o',
		// Week
		'W' => '',
		// Month
		'F' => 'MM',
		'm' => 'mm',
		'M' => 'M',
		'n' => 'm',
		't' => '',
		// Year
		'L' => '',
		'o' => '',
		'Y' => 'yy',
		'y' => 'y',
		// Time
		'a' => '',
		'A' => '',
		'B' => '',
		'g' => '',
		'G' => '',
		'h' => '',
		'H' => '',
		'i' => '',
		's' => '',
		'u' => ''
	);
	$jqueryui_format = "";
	$escaping = false;

	for ( $i = 0; $i < strlen($php_format); $i++ ) {
		$char = $php_format[$i];
		if($char === '\\') {
			$i++;
			if($escaping) $jqueryui_format .= $php_format[$i];
			else $jqueryui_format .= '\'' . $php_format[$i];
			$escaping = true;
		} else {
			if($escaping) { $jqueryui_format .= "'"; $escaping = false; }
			if(isset($SYMBOLS_MATCHING[$char]))
				$jqueryui_format .= $SYMBOLS_MATCHING[$char];
			else
				$jqueryui_format .= $char;
		}
	}

	return $jqueryui_format;
}

/**
 * Return set date time format in Moment Format
 * http://stackoverflow.com/questions/30186611/php-dateformat-to-moment-js-format
 * @param $time_only: No date, just time
 * @since 2.0
 * @return string
 */
function wpb_moment_format( $time_only = false ) {
	$replacements = array(
		'd' => 'DD',
		'D' => 'ddd',
		'j' => 'D',
		'l' => 'dddd',
		'N' => 'E',
		'S' => 'o',
		'w' => 'e',
		'z' => 'DDD',
		'W' => 'W',
		'F' => 'MMMM',
		'm' => 'MM',
		'M' => 'MMM',
		'n' => 'M',
		't' => '', // no equivalent
		'L' => '', // no equivalent
		'o' => 'YYYY',
		'Y' => 'YYYY',
		'y' => 'YY',
		'a' => 'a',
		'A' => 'A',
		'B' => '', // no equivalent
		'g' => 'h',
		'G' => 'H',
		'h' => 'hh',
		'H' => 'HH',
		'i' => 'mm',
		's' => 'ss',
		'u' => 'SSS',
		'e' => 'zz', // deprecated since version 1.6.0 of moment.js
		'I' => '', // no equivalent
		'O' => '', // no equivalent
		'P' => '', // no equivalent
		'T' => '', // no equivalent
		'Z' => '', // no equivalent
		'c' => '', // no equivalent
		'r' => '', // no equivalent
		'U' => 'X',
	);
	if ( $time_only )
		$momentFormat = strtr(BASE()->time_format, $replacements);
	else
		$momentFormat = strtr(BASE()->dt_format, $replacements);

	return $momentFormat;
}

/**
 * Return a default color for a selected box class
 * @return string|array
 */
function wpb_get_preset( $class = null, $set = null ) {
	$colors = array(
		'legacy1' => array (
					'free'				=> '48c048',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'ffffff',
					'notpossible'		=> 'ffffff',
					),
		'legacy2' => array (
					'free'				=> '73ac39',
					'has_appointment'	=> 'ffa500',
					'busy'				=> '616b6b',
					'notpossible'		=> '8f99a3',
					),
		'legacy3' => array (
					'free'				=> '40BF40',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'ff0000',
					'notpossible'		=> 'c0c0c0',
					),
		'base' => array (
					'free'				=> 'f5f5f5',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'c0c0c0',
					'notpossible'		=> 'c0c0c0',
					),
		'ui-lightness' => array (
					'free'				=> '1c94c4',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'f6a828',
					'notpossible'		=> 'dddddd',
					),
		'ui-darkness' => array (
					'free'				=> 'ffffff',
					'has_appointment'	=> 'ffa500',
					'busy'				=> '000000',
					'notpossible'		=> 'dddddd',
					),
		'smoothness' => array (
					'free'				=> 'b3d4fc',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'cd0a0a',
					'notpossible'		=> 'fcefa1',
					),
		'start' => array (
					'free'				=> '6eac2c',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'f8da4e',
					'notpossible'		=> 'aaaaaa',
					),
		'redmond' => array (
					'free'				=> '79b7e7',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'fcefa1',
					'notpossible'		=> 'aaaaaa',
					),
		'sunny' => array (
					'free'				=> 'feeebd',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'd34d17',
					'notpossible'		=> '817865',
					),
		'overcast' => array (
					'free'				=> '3383bb',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'c0402a',
					'notpossible'		=> 'eeeeee',
					),
		'le-frog' => array (
					'free'				=> '4ce90b',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'fcefa1',
					'notpossible'		=> 'e6e6e6',
					),
		'flick' => array (
					'free'				=> '0073ea',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'ff0084',
					'notpossible'		=> 'dddddd',
					),
		'pepper-grinder' => array (
					'free'				=> 'f7f3de',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'b83400',
					'notpossible'		=> '6e4f1c',
					),
		'eggplant' => array (
					'free'				=> '734d99',
					'has_appointment'	=> 'ffa500',
					'busy'				=> '994d53',
					'notpossible'		=> 'dcd9de',
					),
		'dark-hive' => array (
					'free'				=> '0972a5',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'cd0a0a',
					'notpossible'		=> 'aaaaaa',
					),
		'cupertino' => array (
					'free'				=> '2779aa',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'ffef8f',
					'notpossible'		=> 'dddddd',
					),
		'south-street' => array (
					'free'				=> '45c800',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'fcefa1',
					'notpossible'		=> 'aaaaaa',
					),
		'blitzer' => array (
					'free'				=> 'ebefeb',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'cc0000',
					'notpossible'		=> 'aaaaaa',
					),
		'humanity' => array (
					'free'				=> 'e6e6e6',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'cb842e',
					'notpossible'		=> 'aaaaaa',
					),
		'hot-sneaks' => array (
					'free'				=> 'ffff38',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'ff3853',
					'notpossible'		=> 'f7f7ba',
					),
		'excite-bike' => array (
					'free'				=> 'c5ddfc',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'e69700',
					'notpossible'		=> 'e6b900',
					),
		'vader' => array (
					'free'				=> 'cccccc',
					'has_appointment'	=> 'ffa500',
					'busy'				=> '121212',
					'notpossible'		=> '888888',
					),
		'dot-luv' => array (
					'free'				=> '0b58a2',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'a32d00',
					'notpossible'		=> '333333',
					),
		'mint-choc' => array (
					'free'				=> 'add978',
					'has_appointment'	=> 'ffa500',
					'busy'				=> '5f391b',
					'notpossible'		=> 'aaaaaa',
					),
		'black-tie' => array (
					'free'				=> 'ffeb80',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'cd0a0a',
					'notpossible'		=> 'aaaaaa',
					),
		'trontastic' => array (
					'free'				=> '8cce3b',
					'has_appointment'	=> 'ffa500',
					'busy'				=> 'f1ac88',
					'notpossible'		=> 'f6ecd5',
					),
		'swanky-purse' => array (
					'free'				=> 'f8eec9',
					'has_appointment'	=> 'ffa500',
					'busy'				=> '4f4221',
					'notpossible'		=> 'aaaaaa',
					),
		);

		if ( null === $set )
			return $colors;

		return isset( $colors[$set][$class] ) ? $colors[$set][$class] : 'aaaaaa';
}

/**
 * Determine if this is a demo installation where saving of settings are not allowed
 * @since 2.0
 */
function wpb_is_demo(){
	if ( defined( 'WPB_DEMO_MODE' ) && WPB_DEMO_MODE && !(function_exists('is_super_admin') && is_super_admin()) ) {
		return true;
	}

	return false;
}

/**
 * Add submenu page with multiple capability option
 * @param $capability	array|string	Comma delimited string or array with required caps to check (any of them is sufficient)
 * @see add_submenu_page WP function for other parameters: https://developer.wordpress.org/reference/functions/add_submenu_page/
 * @since 3.0
 * @return string
 */
function wpb_add_submenu_page( $parent_slug, $page_title, $menu_title, $capability, $menu_slug, $function = '' ) {
	$caps = is_array( $capability ) ? $capability : array_filter( explode( ',', $capability ) );

	foreach ( $caps as $cap ) {
		if ( current_user_can( $cap ) )
			return add_submenu_page( $parent_slug, $page_title, $menu_title, $cap, $menu_slug, $function );
	}
}

/**
 * On admin side, print an admin notice. On front end open javascript message box or jQuery dialog
 * @param 	$msg	string 		Message to be displayed or WpBNotices method to call
 * @param	$class	string		Optional class name for admin nag
 * @param	$delay	integer		Optional delay time for dialog open (in msec)
 * @since 3.0
 * @return none
 */
function wpb_notice( $msg, $class = 'updated', $delay = 0 ) {
	if ( is_admin() && class_exists( 'WpBNotices' ) ) {
		if ( is_callable( array( BASE('Notices'), $msg ) ) ) {
			add_action( 'admin_notices', array( BASE('Notices'), $msg ) );
		} else {
			BASE('Notices')->set_notice( $msg, $class, $delay );
		}
	} else {
		BASE('Notices')->front( $msg, $class, $delay );
	}

	if ( 'saved' == $msg ) {
		global $wpb_setting_saved;
		$wpb_setting_saved = true;
	}
}

/**
 * Rebuild WP BASE admin menu items
 * @return none
 */
function wpb_rebuild_menu(){
	include_once( WPBASE_PLUGIN_DIR . "/includes/admin/toolbar.php" );
	BASE('Toolbar')->rebuild_menu();
}

/**
 * Outputs an infobox on admin side
 * @param	$visible	string		Visible content
 * @param	$hidden		string		Optional hidden content
 * @since 3.0
 * @return none
 */
function wpb_infobox( $visible, $hidden = false ) {
	include_once( WPBASE_PLUGIN_DIR . "/includes/notices.php" );
	BASE('Notices')->infobox( $visible, $hidden );
}

/**
 * Outputs an infobox with description on admin side
 * @param	$context	string		What to display
 * @since 3.7.7
 * @return none
 */
function wpb_desc_infobox( $context ) {
	include_once( WPBASE_PLUGIN_DIR . '/includes/constant-data.php' );
	switch ( $context ) {
		case 'location':		$desc = WpBConstant::location_desc(); break;
		case 'service':			$desc = WpBConstant::service_desc(); break;
		case 'worker':			$desc = WpBConstant::worker_desc(); break;
		case 'wh':				$desc = WpBConstant::wh_desc(); break;
		case 'email':			$desc = WpBConstant::email_desc(); break;
		case 'addon':			$desc = WpBConstant::addon_desc(); break;
		case 'annual':			$desc = WpBConstant::annual_desc(); break;
		case 'woocommerce':		$desc = WpBConstant::woocommerce_desc(); break;
		case 'udf':				$desc = WpBConstant::udf_desc(); break;
		case 'zoom':			$desc = WpBConstant::zoom_desc(); break;
		case 'jitsi':			$desc = WpBConstant::jitsi_desc(); break;
		case 'extra':			$desc = WpBConstant::extra_desc(); break;
		case 'coupon':			$desc = WpBConstant::coupon_desc(); break;
		case 'ep':				$desc = WpBConstant::ep_desc(); break;
		case 'edd':				$desc = WpBConstant::edd_desc(); break;
		case 'impex':			$desc = WpBConstant::impex_desc(); break;
		case 'import_a_plus':	$desc = WpBConstant::import_a_plus_desc(); break;
	}

	wpb_infobox( array_shift( $desc ), $desc );
}

/**
 * Unobstructive spinner panel
 * @since 3.0
 */
function wpb_info_panel(){
	$spinner = wpb_setting( 'spinner' );
	if ( 'none' == $spinner && !is_admin() ) {
		return;
	}

	?><div id="app-updating-panel" class="app-updating-panel" style="display:none">
	<?php if ( 'clock' == $spinner ) { ?>
	<div class="app-spinner clock" id="clock">
		<div class="shadow"></div>
		<div class="dial">
			<div class="hour hand"></div>
			<div class="minute hand"></div>
		</div>
	</div>
	<?php } else if ( 'cell' == $spinner ) { ?>
	<div class="app-cssload-loader2">
		<div class="cssload-side"></div>
		<div class="cssload-side"></div>
		<div class="cssload-side"></div>
		<div class="cssload-side"></div>
		<div class="cssload-side"></div>
		<div class="cssload-side"></div>
		<div class="cssload-side"></div>
		<div class="cssload-side"></div>
	</div>
	<?php } else if ( 'window' == $spinner ) { ?>
	<div id="xLoader">
		<div class="glistening-window">
		<span></span>
		<span></span>
		<span></span>
		<span></span>
		</div>
	</div>
	<?php } else { ?>
	<div id="app-cssload-loader">
		<div class="cssload-dot"></div>
		<div class="cssload-dot"></div>
		<div class="cssload-dot"></div>
		<div class="cssload-dot"></div>
		<div class="cssload-dot"></div>
		<div class="cssload-dot"></div>
		<div class="cssload-dot"></div>
		<div class="cssload-dot"></div>
	</div>
	<?php } ?>
	<div class="app-updating">WP BASE</div>
</div>
<?php
}

/**
 * Turns an unavail code into readable
 * @param $no: Code number
 * @since 3.0
 * @return string
 */
function wpb_code2reason( $no ) {
	$codes = array(
		1	=> 'busy',
		2	=> 'no_workers',
		3	=> 'no_time',
		4	=> 'break',
		5	=> 'complete_brk',
		6	=> 'lateral',
		7	=> 'location_capacity_full',
		8	=> 'service_not_working',
		9	=> 'all_day_off',
		10	=> 'holiday',
		11	=> 'blocked',
		12	=> 'past_today',
		13	=> 'past_yesterday_or_before',
		14	=> 'upper_limit',
		15	=> 'not_published',
		16	=> 'app_interim',
		17	=> 'unknown',
		18	=> 'nesting_level_exceeded',
		19	=> 'custom',
	);

	$codes = apply_filters( 'app_notavailable_codes', $codes, $no );

	return isset( $codes[$no] ) ? $codes[$no] : '';
}

/**
 * Return a selected field name
 * @return string (name of the field)
 */
function wpb_get_field_name( $key ) {

	$field_name = BASE()->get_text( $key );
	$field_name = $field_name ? $field_name : ucwords( str_replace('_',' ', $key) );

	return apply_filters( 'app_get_field_name', $field_name, $key );
}

/**
 * Return an array of login methods
 * @since 2.0
 * @return array
 */
function wpb_login_methods(){
	return apply_filters( 'app_login_methods', array('Facebook', 'Twitter', 'Google+', 'WordPress') );
}

/**
 * Die showing which field has a problem
 * @param $goto		string	jQuery selector on the document to scroll to
 * @return json object
 */
function wpb_json_die( $field_name, $goto = '' ) {
	die( json_encode( array( 'error' => sprintf(  BASE()->get_text('wrong_value'), wpb_get_field_name($field_name)), 'goTo' => $goto ) ) );
}

/**
 * Check if user can access an admin page
 * @param $cap	array|string	Comma delimited string or array with required caps to check (any one of them is sufficient)
 * @param $die	bool			If true die, if false return result
 * @since 3.0
 * @return none|bool
 */
function wpb_admin_access_check( $cap = '', $die = true ) {

	$one_of_these = array( WPB_ADMIN_CAP ); # Admin capability can access everything

	if ( function_exists( 'is_multisite' ) && is_multisite() )
		$one_of_these[] = 'super_admin'; # Not a capability, but it is valid

	if ( $cap ) {
		$caps_arr = is_array( $cap ) ? $cap : explode( ',', $cap );
		$one_of_these = array_unique( array_merge( $one_of_these, $caps_arr ) );
	}

	if ( !wpb_current_user_can( $one_of_these ) ) {
		if ( $die )
			wp_die( __('You do not have sufficient permissions to access this page.','wp-base'), __('Unauthorised','wp-base'), array("back_link"=>true));
		else
			return false;
	}

	return true;
}

/**
* Whether a field is hidden in confirmation form
* @param $field		string		Name of the confirmation form field to be checked
* @since 3.0
* @return bool
*/
function wpb_is_hidden( $field ) {
	return in_array( $field, explode( ',', wpb_setting("conf_form_hidden_fields") ) );
}

/**
 * Return all fields in the confirmation form
 * @since 3.0
 * @return array
 */
function wpb_conf_form_fields(){
	return apply_filters( 'app_conf_form_fields', array('service', 'provider', 'date_time', 'end_date_time', 'lasts', 'details', 'price', 'deposit', 'down_payment' ) );
}

/**
 * Cart error getter
 * @since 3.0
 * @return string
 */
function wpb_get_cart_error(){
	return BASE()->checkout_error;
}

/**
 * Cart error setter
 * @since 3.0
 */
function wpb_set_cart_error( $msg, $context = 'checkout' ){
	BASE()->checkout_error = $msg;
}

/**
 * Return a human readable payment method name
 * @param $method	string	'plugin-name' of gateway given during registering itself
 * @since 3.0
 * @return string
 */
function wpb_payment_method_name( $method ) {
	global $app_gateway_plugins;

	if ( $maybe_name = wpb_gateway_setting( $method, 'name' ) ) {
		return $maybe_name;
	} else if ( isset( $app_gateway_plugins[ $method ][1] ) ) {
		return $app_gateway_plugins[ $method ][1];
	} else if ( 'woocommerce' == $method ) {
		return 'WooCommerce';
	} else if ( 'edd' == $method ) {
		return 'EDD';
	} else if ( 'pay-later' == $method ) {
		return __( 'Pay Later', 'wp-base' );
	} else {
		return __( 'None', 'wp-base' );
	}
}

/**
 * Increase memory limit
 * @uses wp_raise_memory_limit
 * @since 3.0
 */
function wpb_raise_memory_limit( $context = 'admin' ) {
	if ( function_exists( 'wp_raise_memory_limit' ) ) {
		wp_raise_memory_limit( $context );
	}
}

/**
 * Format HTML output
 * @param $html		string		HTML to be formatted
 * @since 3.0
 * @return string
 */
function wpb_beautify( $html ) {
	if ( ! ( defined( 'WPB_ENABLE_BEAUTIFY' ) && WPB_ENABLE_BEAUTIFY ) ) {
		return $html;
	}

	if ( ! class_exists( 'DOMDocument' ) || !WpBDebug::is_debug() )
		return $html;

	$dom = new DOMDocument();

	$dom->preserveWhiteSpace = false;
	$dom->loadHTML( '<html><body>' . $html ."\n\n" .'</body></html>', LIBXML_HTML_NOIMPLIED );
	$dom->formatOutput = true;

	return str_replace( array( '<html>', '</html>', '<body>', '</body>' ), '', $dom->saveXML( $dom->documentElement, LIBXML_NOEMPTYTAG ) );
}

/**
 * Get pages/post to be used as description or bio
 * @param $post_type	string|array		Post type, e.g. 'post', 'page', 'product'. Comma delimited values are allowed
 * @since 3.0
 * @return array
 */
function wpb_get_description_pages( $post_type ) {
	$post_types = is_array( $post_type ) ? $post_type : array_filter( explode( ',', $post_type ) );
	$post_types = apply_filters( 'app_pages_post_types', $post_types );

	if ( 'page' === $post_type ) {
		$pages = get_pages( array('post_type' => 'page', 'post_status' => 'publish,private') );
	} else {
		$pages = get_posts( array('post_type' => $post_types, 'post_status' => 'publish,private','numberposts' => -1, 'orderby' => 'title', 'order' => 'ASC') );
	}

	return apply_filters( 'app_pages_filter', $pages, $post_type );
}

/**
 * Create the options of a select menu for time of day selection
 * @param $time		string		Selected time of day in military format (H:i), or 'all_day'
 * @since 3.0
 * @return string
 */
function wpb_time_selection( $time ) {
	$step		= 60 * BASE()->get_min_time();
	$is_all_day = (bool)( 'all_day' === $time );
	$fmt_time 	= !$time || $is_all_day ? '' : wpb_from_military( $time );
	$found 		= $is_all_day;

	$html = $html_top = '';

	for ( $t = 0; $t < DAY_IN_SECONDS; $t = $t + $step ) {
		$t_disp = wpb_secs2hours( $t );
		if ( $t_disp == $fmt_time ) {
			$s = ' selected="selected"';
			$found = true;
		} else {
			$s = '';
		}

		$html .= '<option value="'.wpb_to_military( $t_disp) .'" '.$s.'>';
		$html .= $t_disp;
		$html .= '</option>';
	}

	$html .= '<option value="all_day" '.selected( (int)$is_all_day, 1, false ).'>'. esc_html( __( 'All day', 'wp-base' ) ).'</option>';

	# If set time is not in the list, lets add it to the top of options
	if ( !$found && $time) {
		$html_top .= '<option value="'.$time.'" selected="selected">';
		$html_top .= $fmt_time;
		$html_top .= '</option>';
	}

	return $html . $html_top;
}

/**
 * Create the options of a select menu for role selection
 * @param $roles		array		Array of selected roles each as keys in $wp_roles->roles
 * @since 3.0
 * @return string
 */
function wpb_role_selection( $roles ) {
	global $wp_roles;

	$html = '';

	if ( !empty( $wp_roles->roles ) ) {
		foreach (  $wp_roles->roles as $key => $role ) {
			$s = in_array( $key, (array)$roles ) ? " selected='selected'" : '';
			$html .= '<option value="'.$key.'"'. $s .'>'. esc_html( $role['name'] ) . '</option>';
		}
	} else {
		$html .= '<option>'. __( 'No roles defined', 'wp-base' ) . '</option>';
	}

	return $html;
}

/**
 * Create the options of a select menu for frequency (recurrence) selection
 * @param $freq		string		Frequency: Either a week day name, or 'always', 'once', 'everyday', 'weekend', 'weekday'
 * @since 3.0
 * @return string
 */
function wpb_freq_selection( $freq ) {
	$html  = '';
	$html .= '<option '.selected( $freq, 'always', false ).' value="always">'. __( 'Anytime', 'wp-base' ) .'</option>';
	$html .= '<option '.selected( $freq, 'once', false ).' value="once">'. __( 'Sel. range', 'wp-base' ) .'</option>';
	$html .= '<option '.selected( $freq, 'everyday', false ).' value="everyday">'. __( 'Every day', 'wp-base' ) .'</option>';
	$html .= '<option '.selected( $freq, 'weekday', false ).' value="weekday">'. __( 'Week day', 'wp-base' ) .'</option>';
	$html .= '<option '.selected( $freq, 'weekend', false ).' value="weekend">'. __( 'Week end', 'wp-base' ) .'</option>';
	foreach ( BASE()->weekdays() as $day => $day_name ) {
		$html .= '<option '.selected( $freq, $day, false ).' value="'.$day.'">'.
				esc_html( sprintf( __( 'Every %s', 'wp-base' ), $day_name ) ) .'</option>';
	}
	$html .= '<option '.selected( $freq, 'special_days', false ).' value="special_days">'. __( 'Special Days', 'wp-base' ) .'</option>';

	return $html;
}

/**
 * Create the options of a select menu for "applies_to" selection
 * @param $applies_to	array	An array of | delimited strings where left part is lsw, right side is ID, e.g. ['service|3','location|2']
 * @since 3.0
 * @return string
 */
function wpb_applies_to_selection( $applies_to ) {
	$html = '';
	$html .= '<optgroup label="'. esc_attr(__('Locations','wp-base')) . '" class="optgroup_location">';
	if ( $locations = BASE()->get_locations() ) {
		foreach ( $locations as $location ) {
			$s = in_array( 'location|'. $location->ID, $applies_to ) ? " selected='selected'" : '';
			$html .= '<option value="location|'.$location->ID.'"'. $s .'>' . esc_html( BASE()->get_location_name( $location->ID ) ) . '</option>';
		}
	}
	$html .= '</optgroup>';

	$html .= '<optgroup label="'. esc_attr(__('Services','wp-base')) . '" class="optgroup_service">';
	$services = BASE()->get_services();
	foreach ( (array)$services as $service ) {
		$s = in_array( 'service|'. $service->ID, $applies_to ) ? " selected='selected'" : '';
		$html .= '<option value="service|'.$service->ID.'"'. $s .'>' . esc_html( BASE()->get_service_name( $service->ID ) ) . '</option>';
	}
	$html .= '</optgroup>';

	$html .= '<optgroup label="'. esc_attr(__('Service Providers','wp-base')) .'" class="optgroup_worker">';
	if ( $workers = BASE()->get_workers() ) {
		foreach ( $workers as $worker ) {
			$s = in_array( 'worker|'.$worker->ID, $applies_to ) ? " selected='selected'" : '';
			$html .= '<option value="worker|'.$worker->ID.'"'.$s.'>' . esc_html( BASE()->get_worker_name( $worker->ID, false ) ) . '</option>';
		}
	}
	$html .= '</optgroup>';

	return apply_filters( 'app_applies_to_selection', $html, $applies_to );
}

/**
 * Displays an info button for which upon click description text displayed/hidden
 * @param $instance		object	Instance of the class of addon where it is called from
 * @param $text			string	Tooltip text for the title
 * @since 3.0
 * @return string
 */
function wpb_info_button( $instance, $text = '' ) {
	$classname = strtolower( get_class( $instance ) ).'-desc';
	$title = $text ? '<abbr title="'.$text.'">'. $instance->public_name .'</abbr>' : $instance->public_name;

	$html  = '<div class="addon-set-info">';
	$html .= '<span class="addon-set-title">'. $title . '</span>';
	$html .= '<a href="javascript:void(0)" data-boxname="'.$classname.'" class="addon-info-button info-button" title="'. __('Click to toggle details', 'wp-base' ) .'">';
	$html .= '<img style="width:24px;height:24px" src="'. WPB_PLUGIN_URL ."/images/help.png" .'" alt="'. __('Information','wp-base').'" /></a>';
	$html .= '</div>';

	return $html;
}

/**
 * Render special days box
 * @param $by_worker	integer		Id of worker who is calling the function from his user page
 */
function wpb_render_special_days( $by_worker ) {

	if ( $by_worker ) {
		return;
	}
?>
<div class="postbox app-special-days-box">
	<div class="inside">
		<p>
		<?php _e('It looks like some of your WP BASE Addons have been outdated. Please update them to the latest version.','wp-base') ?>
		</p>
	</div>
</div>
<?php
}

/**
 * Render yes/no selection for admin settings
 * @param $set		string		Name of the setting
 * @param $third	array		Optional third option where key is option value, array value is option text
 */
function wpb_setting_yn( $set, $third = array() ) {
	$val = wpb_setting( $set );
?>
<tr id="<?php echo str_replace( '_', '-', $set ) ?>">
	<th scope="row" ><?php WpBConstant::echo_setting_name( $set ) ?></th>
	<td>
		<select name="<?php echo $set ?>">
			<option value="no" <?php selected( $val, 'no' ) ?>><?php _e('No', 'wp-base')?></option>
			<option value="yes" <?php selected( $val, 'yes' ) ?>><?php _e('Yes', 'wp-base')?></option>
<?php if ( $third ) { ?>
			<option value="<?php echo key( $third ) ?>" <?php selected( $val, key( $third ) ) ?>><?php echo current( $third ) ?></option>
<?php } ?>
		</select>
		<span class="description app-btm"><?php WpBConstant::echo_setting_desc( $set ) ?></span>
	</td>
</tr>
<?php
}

/**
 * Render input text field for admin settings
 * @param $set			string		Name of the setting
 * @param $class		string		Css class name
 * @param $fallback		mix			Return value when item is not set
 */
function wpb_setting_text( $set, $class = 'app-50', $fallback = null ) {
	$val = wpb_setting( $set, $fallback );
?>
<tr id="<?php echo str_replace( '_', '-', $set ) ?>">
	<th scope="row" ><?php WpBConstant::echo_setting_name( $set ) ?></th>
	<td>
		<input type="text" class="<?php echo $class ?>" value="<?php echo esc_attr( $val ) ?>" name="<?php echo $set ?>" />
		<span class="description app-btm"><?php WpBConstant::echo_setting_desc( $set ) ?></span>
	</td>
</tr>
<?php
}

/**
 * Render textarea field for admin settings
 * @param $set		string		Name of the setting
 * @param $class	string		Class name
 */
function wpb_setting_textarea( $set, $class = 'app-1' ) {
	$val = wpb_setting( $set );
?>
<tr id="<?php echo str_replace( '_', '-', $set ) ?>">
	<th scope="row"><?php WpBConstant::echo_setting_name( $set ) ?></th>
	<td>
		<textarea class="<?php echo $class ?>" name="<?php echo $set ?>"><?php echo esc_textarea( $val ) ?></textarea>
		<span class="description"><?php WpBConstant::echo_setting_desc( $set ) ?></span>
	</td>
</tr>
<?php
}

/**
 * Calls 'footer' method for instance once
 * @param $instance		object|string		Class instance footer method belongs to or function name to call
 * @param $pri			integer				Priority
 */
function wpb_add_action_footer( $instance, $pri = 10 ) {
	$action = is_admin() ? 'admin_footer' : 'wp_footer';
	$callable = is_object( $instance ) ? array( $instance, 'footer' ) : $instance;

	if ( ! has_action( $action, $callable ) ) {
		add_action( $action, $callable, $pri );
	}
}

/**
 * Sanitizes search filter/order parameters
 * @return array	key is the searched parameter, value is the searched value
 */
function wpb_sanitize_search(){
	$filter = array();
	foreach ( array( 'location_id', 'service_id', 'category_id', 'worker_id', 'order_by', 'balance', 'monweek', 'owner', 'duration' ) as $i ) {
		if ( isset( $_GET['app_'.$i] ) ) {
			$filter[$i] = sanitize_text_field( $_GET['app_'.$i] );
		} else {
			$filter[$i] =  '';
		}
	}

	return $filter;
}

/**
 * Helper to print month options in filter by month
 * @param $where 	string 		WHERE clause for mysql. Should have been sanitized before
 * @param $context	string		Where it is called from and which DB table: transactions, created, paid (default empty means 'start' in bookings table)
 * @return none
 */
function wpb_month_options( $where, $context = '' ) {
	global $wp_locale;

	$monweek = !empty( $_GET['app_monweek'] ) ? sanitize_text_field( $_GET['app_monweek'] ) : '';

	if ( 'transactions' == $context ) {
		$months = BASE()->db->get_results( "
			SELECT DISTINCT
				MONTH(FROM_UNIXTIME(transaction_stamp)) AS month,
				YEAR(FROM_UNIXTIME(transaction_stamp)) AS year
			FROM ".BASE()->transaction_table."
			{$where}
			GROUP BY month, year
			ORDER BY transaction_stamp"
		);
	} else {
		$table = $context ? BASE()->com_table : BASE()->app_table;
		$field = $context ?: 'start';

		$months = BASE()->db->get_results(
			"SELECT DISTINCT YEAR( {$field} ) AS year, MONTH( {$field} ) AS month
			FROM {$table}
			{$where}
			ORDER BY {$field}"
		 );
	}

	foreach ( (array)$months as $arc_row ) {
		if ( 0 == $arc_row->year )
			continue;

		$month = zeroise( $arc_row->month, 2 );
		$year = $arc_row->year;

		printf( "<option %s value='%s'>%s</option>\n",
			selected( $monweek, $year . $month, false ),
			esc_attr( $arc_row->year . $month ),
			sprintf( __( '%1$s %2$d', 'wp-base' ), $wp_locale->get_month( $month ), $year )
		);
	}
}

/**
 * Helper to print week options in filter by week
 * @param $where 	string 		WHERE clause for mysql. Should have been sanitized before
 * @param $context	string		Where it is called from and which DB table: transactions, created, paid (default empty means 'start' in bookings table)
 * @return none
 */
function wpb_week_options( $where, $context = '' ) {
	$monweek	= !empty( $_GET['app_monweek'] ) ? sanitize_text_field( $_GET['app_monweek'] ) : '';
	$mode 		= BASE()->start_of_week ? 7 : 4;

	if ( 'transactions' == $context ) {
		$weeks = BASE()->db->get_results( "
			SELECT DISTINCT YEARWEEK(FROM_UNIXTIME(transaction_stamp), {$mode}) AS yearweek
			FROM ".BASE()->transaction_table."
			{$where}
			ORDER BY transaction_stamp"
		);
	} else {
		$table = $context ? BASE()->com_table : BASE()->app_table;
		$field = $context ?: 'start';

		$weeks = BASE()->db->get_results(
			"SELECT DISTINCT YEARWEEK( {$field}, {$mode} ) AS yearweek
			FROM {$table}
			{$where}
			ORDER BY {$field}"
		);
	}

	foreach ( (array)$weeks as $arc_row ) {
		if ( !$arc_row->yearweek )
			continue;

		$yearweek	= 'w'. $arc_row->yearweek;
		$year		= substr( $yearweek, 1, 4 );
		$week_no	= substr( $yearweek, -2 );
		$first_day	= 1 + (7 + BASE()->start_of_week - wpb_strftime( "%w", mktime( 0, 0, 0, 1, 1, $year ) ) )%7;
		$start_ts	= mktime( 0, 0, 0, 1, $first_day, $year ) + ($week_no-1)*7*24*3600;
		$end_ts		= $start_ts + 6*24*3600;
		$start_date = date_i18n( "j M Y", (int)$start_ts );
		$end_date	= date_i18n( "j M Y", (int)$end_ts );

		printf( "<option %s value='%s'>%s</option>\n",
			selected( $monweek, $yearweek, false ),
			esc_attr( $yearweek ),
			sprintf( __( '%1$s - %2$s', 'wp-base' ), $start_date, $end_date )
		);
	}
}

/**
 * Determine if this is a WP BASE page on admin
 * This should be called later than admin_init:
 * https://codex.wordpress.org/Function_Reference/get_current_screen#Usage_Restrictions
 * @param $page: Query for a specific page: bookings, transactions, calendars, business, settings, addons, tools, help
 * @return bool
 */
function wpb_is_admin_page( $page = '' ) {
	if ( !$screen_id = wpb_get_current_screen_id( ) ) {
		$r = false;
	} else if ( !$page ) {
		if ( 'toplevel_page_appointments' == $screen_id ) {
			$r = true;
		} else if ( strpos( $screen_id, BASE()->app_name.'_page' ) !== false
				|| strpos( $screen_id, 'users_page_your_bookings' ) !== false
				|| strpos( $screen_id, 'profile_page_your_bookings' ) !== false ) {

			$r = true;
		} else {
			$r = false;
		}
	} else if ( 'bookings' == $page ) {
		if ( strpos( $screen_id, 'app_bookings' ) !== false ) {
			$r = true;
		} else {
			$r = false;
		}
	} else if ( strpos( $screen_id, BASE()->app_name.'_page_app_'.$page ) !== false ) {
		$r = true;
	} else {
		$r = false;
	}

	return apply_filters( 'app_is_admin_page', $r, $page );
}

/**
 * Determine if current admin page is similar to a WP BASE front page
 * It means, an admin page that requires front end css and js files (e.g. users page, calendars page)
 * Note: This should be called later than admin_init:
 * https://codex.wordpress.org/Function_Reference/get_current_screen#Usage_Restrictions
 * @return bool
 */
function wpb_requires_front() {
	$screen_id =  wpb_get_current_screen_id( );
	if ( BASE()->app_name.'_page_app_schedules' == $screen_id || BASE()->app_name.'_page_app_business' == $screen_id ||
		'users_page_your_bookings' == $screen_id || 'profile_page_your_bookings' == $screen_id ) {
		$r = true;
	} else {
		$r = false;
	}

	return apply_filters( 'app_requires_front', $r );
}

/**
 * Fix font colors for certain themes on wp header
 * @param $echo		bool	Whether print or return
 * @return string|none
 */
function wpb_fix_font_colors( $echo = true ){
	$theme 		= BASE()->selected_theme( wpb_is_admin() );
	$is_enabled = wpb_is_admin() ? 'yes' != wpb_setting('disable_css_admin') : 'yes' != wpb_setting('disable_css');
	$themes 	= array( 'start', 'ui-darkness', 'dark-hive', 'south-street', 'excite-bike', 'vader',
					'dot-luv', 'le-frog', 'mint-choc', 'black-tie', 'trontastic', 'swanky-purse' );

	if ( ! ( $is_enabled && in_array( $theme, $themes ) ) )
		return;

	$style = '.dataTables_info,.dataTables_wrapper .dataTables_paginate .fg-button,.app-account-page .dataTables_wrapper .dataTables_paginate .fg-button,.app-weekly-hours-mins,.app-event-daynum.ui-state-default,.ui-datepicker-calendar .ui-state-default {color: #fff;}';

	if ( $echo ) {
		echo '<style type="text/css">'.$style.'</style>';
	} else {
		return $style;
	}
}

/**
 * DEPRECATED - Use wpb_sanitize_user_fields instead
 * @param	$fields		string	Comma delimited values for required field names
 * @return array
 */
function wpb_sorted_user_fields( $fields, $_app_id = 0, $_editing = false ) {
	return wpb_sanitize_user_fields( $fields, $_app_id, $_editing );
}

/**
 * Return a sanitized list of user fields sorted according to attributes in conf shortcode
 * @param	$fields		string	Comma delimited values for required field names
 * @return array
 */
function wpb_sanitize_user_fields( $fields, $_app_id = 0, $_editing = false ) {
	$allowed_fields  = apply_filters( 'app_confirmation_allowed_fields', BASE()->get_user_fields(), $_app_id, $_editing, $fields );

	$u_fields = array();
	if ( trim( $fields ) ) {
		$u_fields = array_filter( explode( ",", wpb_sanitize_commas( $fields ) ) );
		$u_fields = array_map( 'strtolower', $u_fields );

		foreach ( $u_fields as $key => $f ) {
			if ( !in_array( $f, $allowed_fields ) )
				unset( $u_fields[$key] );
		}
		$u_fields = array_filter( array_unique( $u_fields ) );
	}

	# If no special sorting set or nothing left, use defaults instead
	if ( empty( $u_fields ) ) {
		$u_fields = $allowed_fields;
		foreach( $u_fields as $key => $f ) {
			if ( in_array( strtolower( $f ), BASE()->get_user_fields() ) && !wpb_setting("ask_".$f) )
				unset( $u_fields[$key] );
		}
	}

	return $u_fields;
}

/**
 * Return final confirmation dialog title
 * @param $context	string	confirmation, pending
 * @since 3.0
 */
function wpb_dialog_title( $app, $context = 'confirmation' ) {
	return BASE()->_replace( apply_filters( 'app_'.$context.'_title', wpb_setting($context.'_title'), $app, $context.'_title' ), $app, $context.'_title' );
}

/**
 * Return final confirmation dialog text
 * @param $context	string	confirmation, pending
 * @since 3.0
 */
function wpb_dialog_text( $app, $context = 'confirmation' ) {
	return BASE()->_replace( apply_filters( 'app_'.$context.'_text', wpb_setting($context.'_text'), $app, $context.'_text' ), $app, $context.'_text' );
}

/**
 * Check if current request originated from BuddyPress "Book me" page
 * @return bool
 */
function wpb_is_bp_book_me(){
	return (isset( $_POST['bp_tab'] ) && !empty( $_POST['bp_displayed_user_id'] )
			&& strpos( $_POST['bp_tab'], WPB_BP_BOOK_ME_SLUG ) !== false && BASE('BuddyPress'));
}

/**
 * Check if current request originated from BuddyPress Manage Bookings tab
 * @since 3.7.1.5
 * @return bool
 */
function wpb_is_bp_manage(){
	return (isset( $_POST['bp_tab'] ) && !empty( $_POST['bp_displayed_user_id'] ) && strpos( $_POST['bp_tab'], 'manage' ) !== false && BASE('BuddyPress'));
}

/**
 * Calculate previous/next timestamps and max/min values for browse buttons
 * @param $time		integer		Timestamp of the first step
 * @param $step		integer		Number of steps of units to browse to
 * @param $unit		string		Unit of time: month, week, day
 *
 * @return array
 */
function wpb_prev_next( $time, $step, $unit ) {
	$plural = $step > 1 ? 's' : '';

	if ( 'week' === $unit ) {
		$prev 		= $time - ($step*WEEK_IN_SECONDS);
		$next 		= $time + ($step*WEEK_IN_SECONDS);
		$prev_min 	= BASE()->_time - $step*WEEK_IN_SECONDS;
		$next_max 	= BASE()->_time + (BASE()->get_app_limit() + 7*$step ) *DAY_IN_SECONDS;
		$next_text 	= BASE()->get_text( 'next_week'. $plural );
		$prev_text 	= BASE()->get_text( 'previous_week'. $plural );
	} else if ( 'month' === $unit ) {
		$prev 		= wpb_first_of_month( $time, -1 * $step );
		$next 		= wpb_first_of_month( $time, $step );
		$prev_min 	= wpb_first_of_month( BASE()->_time, -1 * $step );
		$next_max 	= wpb_first_of_month( BASE()->_time, $step ) + BASE()->get_app_limit() * DAY_IN_SECONDS;
		$next_text	= BASE()->get_text( 'next_month'. $plural );
		$prev_text	= BASE()->get_text( 'previous_month'. $plural );
	} else if ( 'day' === $unit || 'number' === $unit ) {
		$prev 		= $time - ($step*DAY_IN_SECONDS);
		$next 		= $time + ($step*DAY_IN_SECONDS);
		$prev_min 	= BASE()->_time - $step*DAY_IN_SECONDS;
		$next_max 	= BASE()->_time + (BASE()->get_app_limit() + $step ) *DAY_IN_SECONDS;
		$next_text 	= BASE()->get_text( 'next' );
		$prev_text 	= BASE()->get_text( 'previous' );
	}

	return array( $prev, $prev_min, $prev_text, $next, $next_max, $next_text );
}

/**
 * Create html of flexslider for location/service/worker selection for one slide
 * DEPRECATED - Moved to Controller class
 * @param $lsw			object		Location/service/worker object
 * @param $controller	object		WpB_Controller object
 * @param $context		string		location, service or worker
 *
 * @return string
 */
function wpb_flexslider_item( $lsw, $controller, $context ) {
	return;
}

/**
 * Check if this is a WC BASE installation
 *
 * @return bool
 */
function wpb_wc() {
	return defined( 'WPB_WC' ) && WPB_WC;
}

/**
 * Check if this is a EB BASE installation
 *
 * @return bool
 */
function wpb_eb() {
	return defined( 'WPB_EB' ) ? WPB_EB : false;
}

/**
 * Find recurring increment in seconds based on repeat unit and timestamp
 * @param $unit			string		daily, weekly, etc
 * @param $timestamp	integer		Time to count from
 * @return integer	In seconds
 */
function wpb_recurring_increment( $unit, $timestamp = 0 ) {
	if ( ! $timestamp )
		$timestamp = BASE()->_time;

	switch( $unit ) {

		case 'daily':			$d = DAY_IN_SECONDS; break;
		case 'weekday':			if ( wpb_is_weekend( $timestamp ) )
									$d = WPB_HUGE_NUMBER;
								else if ( 'Friday' == date( 'l', $timestamp ) )
									$d = 3*DAY_IN_SECONDS;
								else
									$d = DAY_IN_SECONDS;
								break;

		case 'weekend':			if ( 'Saturday' == date( 'l', $timestamp ) )
									$d = DAY_IN_SECONDS;
								else if ( 'Sunday' == date( 'l', $timestamp ) )
									$d = 6*DAY_IN_SECONDS;
								else
									$d = WPB_HUGE_NUMBER;
								break;

		case 'eod':				$d = 2*DAY_IN_SECONDS; break;
		case 'eod_except_sunday':
								if ( 'Sunday' == date( 'l', $timestamp ) )
									$d = WPB_HUGE_NUMBER;
								else if ( 'Friday' == date( 'l', $timestamp ) )
									$d = 3*DAY_IN_SECONDS;
								else
									$d = 2*DAY_IN_SECONDS;
								break;

		case 'weekly':			$d = WEEK_IN_SECONDS; break;
		case 'biweekly':		$d = 2*WEEK_IN_SECONDS; break;

		# Same day of week
		case 'monthly_same_day':$new_day = $timestamp + 5*WEEK_IN_SECONDS;
								if ( date( 'n', $new_day ) != date( 'n', $timestamp ) + 1 )
									$new_day = $timestamp + 4*WEEK_IN_SECONDS;

								$d = $new_day - $timestamp;
								break;

		# Same day of month - Fallback to last day of next month when not possible
		case 'monthly':			$this_day 	= strtotime( date('Y-m-d', $timestamp ) );
								$day		= date( "j", $this_day );
								$month		= date( "n", $this_day );
								$year		= date ( "Y", $this_day );
								$start		= mktime( 0, 0, 0, $month, $day, $year );
								$end		= strtotime( '+1 months', $start );

								if ( date( 'd', $start ) != date( 'd', $end ) ) {
									$end = strtotime( '- ' . date('d', $end ) . ' days', $end );
								}
								$d = $end - $start;
								break;

		# Same day of month - Strict: Do not allow fallback
		case 'monthly_strict':	$this_day = strtotime( date( 'Y-m-d', $timestamp ) );
								$d = strtotime( '+1 months', $this_day ) == $this_day + 30*DAY_IN_SECONDS ? 30*DAY_IN_SECONDS : WPB_HUGE_NUMBER;
								break;

		default:				$d = WPB_HUGE_NUMBER; break;
	}

	return $d;
}

/**
 * Pick a random color
 * @return string (hex code of the color including # sign)
 */
function wpb_random_color() {
	include_once( WPBASE_PLUGIN_DIR . '/includes/constant-data.php' );
	$colors = array_flip( WpBConstant::colors() );
	shuffle( $colors );
	return '#'. current( $colors );
}

/**
 * Find a suitable text color based on background color
 * @return string (black or white)
 */
function wpb_text_color( $hexcolor ) {
	$hexcolor = ltrim( $hexcolor, '#' );
	$r = hexdec(substr($hexcolor, 0, 2));
	$g = hexdec(substr($hexcolor, 2, 2));
	$b = hexdec(substr($hexcolor, 4, 2));
	$yiq = (($r * 299) + ($g * 587) + ($b * 114)) / 1000;
	return ($yiq >= 128) ? 'black' : 'white';
}

/**
 * Check if current page is an "account page", i.e. has [app_account] shortcode
 * @return bool
 */
function wpb_is_account_page() {
	return apply_filters( 'app_is_account_page', false );
}

/**
 * Check if business representative's working hours will be ignored
 * @return bool
 */
function wpb_ignore_bis_rep_wh() {
	return ! BASE()->get_nof_workers() && 'yes' == wpb_setting( 'ignore_bis_rep_wh' );
}

/**
 * Check if current page is login page
 * @return bool
 */
function wpb_is_login_page(){
	return function_exists( 'wp_login_url' ) && stripos($_SERVER["SCRIPT_NAME"], strrchr(wp_login_url(), '/')) !== false;
}

/**
 * Whether user is the owner of the booking
 * @param $app		object		Appointment StdClass object to check against
 * @since 3.5.3
 * @return bool
 */
function wpb_is_owner( $app ) {
	if ( $app->user == get_current_user_id() ) {
		return true;
	}

	# Check from cookie
	if ( $apps = BASE()->get_apps_from_cookie() ) {
		if ( is_array( $apps ) && in_array( $app->ID, $apps ) ) {
			return true;
		}
	}

	return false;
}

/**
 * WP Editor settings
 * @param $context		string		Name of the class function is called from
 * @since 3.5.3
 * @return array
 */
function wpb_editor_settings( $context = '' ) {
	return apply_filters( 'app_editor_settings', array(
		'editor_height'	=> WPB_EDITOR_HEIGHT,
		'editor_class'	=> 'app-wp-editor',
		'editor_css'	=> '',
		'tinymce'		=> array( 'body_class' => 'app-editor-body' ),
	), $context );
}

/**
 * Get a list of service duration selection values
 * @since 3.5.4
 * @return array (in minutes)
 */
function wpb_service_duration_selections(){
	$min_time = BASE()->get_min_time();
	return apply_filters( 'app_service_duration_selection', array_merge( range( $min_time, 720, $min_time ), array( 1440 ), range( 2 * 60 * 24, WPB_MAX_SERVICE_DURATION_SELECTION * 60 * 24, 60 * 24 ) ) );
}

/**
 * Add script for Language Selection widget and Shortcode
 * @since 3.5.6
 * @return none
 */
function wpb_multilang_script(){
?>
<script type="text/javascript">
jQuery(document).ready(function($){
	$(".app_select_lang").change( function(){
		var lang = $(this).val();
		var form = $(this).parents("form.app_select_lang_form");
		var action = form.attr("action");
		var new_action = action.replace("SELECTED_LANGUAGE",lang);
		form.attr("action",new_action).submit();
	});
});
</script>
<?php
}

/**
 * Return number of days in selected calendar
 * @since 3.5.7
 * @return integer
 */
function wpb_cal_days_in_month( $calendar, $month, $year ){
	if ( ! function_exists( 'cal_days_in_month' ) ) {
		return date( 't', mktime( 0, 0, 0, $month, 1, $year ) );
	}

	return cal_days_in_month( CAL_GREGORIAN, $month, $year );
}

/**
 * Return admin side tabs
 * @param	$context	string		Name of the page tab will be used on
 * @since 3.6.0
 * @return array
 */
function wpb_admin_tabs( $context ) {
	switch ( $context ) {
		case 'monetary':	$tabs = array( 'general' => __('General', 'wp-base' ), 'gateways' => __('Payment Methods', 'wp-base') ); break;
		case 'display':		$tabs = array( 'general' => __('General', 'wp-base') ); break;
		case 'business':	$tabs = array(); break;
		case 'settings':
		case 'global':		$tabs = array( 'general' => __('General', 'wp-base' ) );

							if ( !array_key_exists( 'advanced', $tabs ) && apply_filters( 'app_add_advanced_tab', false ) ) {
								$tabs['advanced'] = __('Addons', 'wp-base' );
							}

							$tabs['email'] = __('Email', 'wp-base');

							$tabs = apply_filters( 'appointments_tabs', $tabs ); # For backwards compat
							break;
		case 'tools':		$tabs = array( 'log' => __('Logs', 'wp-base'), 'reset' => __('Reset', 'wp-base') ); break;
		case 'help':		$tabs = array(
								'general'    	=> __('General', 'wp-base'),
								'settings'    	=> __('Global Settings', 'wp-base'),
								'shortcodes'    => __('Shortcodes', 'wp-base'),
								'support'    	=> __('Support', 'wp-base'),
								'about'    		=> __('About', 'wp-base' )
							);
							break;
		default:			$tabs = array(); break;
	}

	return apply_filters( 'appointments_'.$context.'_tabs', $tabs );
}

/**
 * Find currently requested url without parameters
 * @since 3.6.0
 * @return string
 */
function wpb_requested_url(){
	$scheme	= is_ssl() ? 'https://' : 'http://';
	$full	= strtolower( $scheme . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'] );
	$parts	= explode( '?', $full );

	return !empty( $parts[0] ) ? $parts[0] : $full;
}

/**
 * Whether user can view payments
 * @param	$user_id	integer		User whose capability is being checked
 * @since 3.6.0
 * @return bool
 */
function wpb_user_can_view_payments( $user_id ) {
	if ( ! $user_id ) {
		return false;
	}

	if ( user_can( $user_id, WPB_ADMIN_CAP ) || user_can( $user_id, 'manage_transactions' ) ) {
		return true;
	}

	if ( wpb_is_worker( $user_id ) ) {
		if ( ! user_can( $user_id, 'manage_own_transactions' ) ) {
			return false;
		}

		if ( 'yes' == wpb_setting( 'allow_worker_view_payments' ) ) {
			return true;
		}
	} else if ( user_can( $user_id, 'wpb_client' ) ) {
		return true; # Client has always access
	}

	return false;
}

/**
 * Log email result
 * @param	$text	string		Text to log
 * @since 3.6.2
 * @return none
 */
function wpb_log_email( $text ) {
	if ( 'yes' == wpb_setting( 'log_emails' ) ) {
		BASE()->log( $text );
	}
}

/**
 * Check if website is a Multi Vendor Marketplace
 * @since 3.6.4
 * @return bool
 */
function wpb_is_multi_vendor(){
	return apply_filters( 'app_is_multi_vendor', false );
}

/**
 * Find or calculate time zone setting in string form
 * @since 3.6.5
 * @return string
 */
function wpb_find_timezone_string(){

	$tz = get_option( 'timezone_string' );
	if ( ! empty( $tz ) ) {
		return $tz;
	}

	$offset  = get_option( 'gmt_offset' );
	$hours   = (int)$offset;
	$minutes = abs( ( $offset - (int) $offset ) * 60 );
	$offset  = sprintf( '%+03d:%02d', $hours, $minutes );

	// Calculate seconds from offset
	if ( function_exists( 'timezone_name_from_abbr' ) ) {

		list( $hours, $minutes ) = explode( ':', $offset );
		$seconds = $hours * 60 * 60 + $minutes * 60;
		$tz      = timezone_name_from_abbr( '', $seconds, 1 );

		if ( $tz === false ) {
			$tz = timezone_name_from_abbr( '', $seconds, 0 );
		}
	}

	return $tz;
}

/**
 * Similar to array_merge_recursive but do not duplicate values
 * @since 3.7.0.1
 * @return array
 */
function wpb_array_merge( $arr1, $arr2 ) {
	$keys = array_keys( array_merge( $arr1, $arr2 ) );
	$out = array();

	foreach ( $keys as $key ) {
		if ( ! array_key_exists( $key, $arr1 ) ) {
			$out[ $key ] = $arr2[ $key ];
		} else if ( ! array_key_exists( $key, $arr2 ) ) {
			$out[ $key ] = $arr1[ $key ];
		} else {
			if ( is_array( $arr1[ $key ] ) && is_array( $arr2[ $key ] ) ) {
				$out[ $key ] = array_merge( $arr1[ $key ], $arr2[ $key ] );
			} else {
				$out[ $key ] = $arr2[ $key ];
			}
		}
	}

	return $out;
}

/**
 * Check if user has manage capability
 * @since 3.7.0.4
 * @return bool
*/
function wpb_can_manage( $user_id = null ) {
	if ( null === $user_id ) {
		$user_id = get_current_user_id();
	}

	if ( ! user_can( $user_id, WPB_ADMIN_CAP ) && ! wpb_is_worker( $user_id ) ) {
		return false;
	}

	if ( user_can( $user_id, WPB_ADMIN_CAP ) && ! wpb_admin_access_check( 'manage_bookings', false ) ) {
		if ( wpb_is_worker( $user_id ) && ! wpb_admin_access_check( 'manage_own_bookings', false ) ) {
			return false;
		}
	}

	if ( ! user_can( $user_id, WPB_ADMIN_CAP ) && wpb_is_worker( $user_id ) && 'yes' != wpb_setting('allow_worker_edit') ) {
		return false;
	}

	return true;
}

/**
 * Url that client will be returned after checkout/booking
 * Usually a thank you page, or account page
 * @param	$fallback	string		Url that will be used when setting is empty
 * @since 3.7.4.1
 * @return string
*/
function wpb_refresh_url( $fallback = null ){
	if ( $maybe_url = wpb_setting( 'refresh_url' ) ) {
		$url = is_numeric( $maybe_url ) ? get_permalink( $maybe_url ) : $maybe_url;
	} else if ( null !== $fallback ) {
		$url = $fallback;
	} else {
		$url = get_permalink( wpb_find_post_id() );
	}

	return apply_filters( 'app_refresh_url', $url );
}

/**
 * For WP versions 4.7<
 * @since 3.7.4.1
 * @return bool
*/
function wpb_doing_ajax(){
	if ( function_exists( 'wp_doing_ajax' ) ) {
		return wp_doing_ajax();
	}

	return apply_filters( 'wp_doing_ajax', defined( 'DOING_AJAX' ) && DOING_AJAX );
}

/**
 * Escape JSON for use on HTML or attribute text nodes.
 *
 * @since 3.7.5.1
 * @param string $json JSON to escape.
 * @param bool   $html True if escaping for HTML text node, false for attributes. Determines how quotes are handled.
 * @return string Escaped JSON.
 */
function wpb_esc_json( $json, $html = false ) {
	return _wp_specialchars(
		$json,
		$html ? ENT_NOQUOTES : ENT_QUOTES, // Escape quotes in attribute nodes only.
		'UTF-8',                           // json_encode() outputs UTF-8 (really just ASCII), not the blog's charset.
		true                               // Double escape entities: `&amp;` -> `&amp;amp;`.
	);
}

/**
 * Wraps field with label/span
 * @since 3.7.7
 * @return string
 */
function wpb_wrap_field( $name, $title, $field, $helptip = '', $style = '' ) {
	$tip = '';
	# Backwards compat
	if ( strpos( $helptip , 'style=' ) !== false ) {
		$style = $helptip;
		$helptip = '';
	} else if ( $helptip ) {
		$tip = '<em class="app-helptip" title="'.esc_attr( $helptip ).'"></em>';
	}

	$html  = '<label class="app_iedit_'.$name.' wp-clearfix"'. ($style ? ' '. $style : '') .'>';
	$html .= '<span class="title">'.$title. $tip. '</span>';
	$html .= '<span class="input-text-wrap">';
	$html .= $field;
	$html .= '</span>';
	$html .= '</label>';

	return apply_filters( 'app_wrap_field', $html, $name, $title, $field, $helptip, $style );
}

/**
 * Whether a page/post is CPT of WC or EDD and it has WP BASE shortcode
 * @since 3.7.7
 * @return bool
 */
function wpb_is_product_page( $post = null ) {
	return apply_filters( 'app_is_product_page', false, $post );
}

/**
 * Replace common placeholders in title texts and create HTML
 * @param $title		string		Title template with placeholders
 * @param $controller	object		WpB_Controller object
 * @param $design		string		Compact/auto/legacy. If empty, don't use flexslider next/prev
 * @since 3.7.7.1
 * @return string		Escaped html string
 */
function wpb_title_html( $title, $controller, $design = '' ) {
	if ( '0' === (string)$title ) {
		return '';
	}

	if ( 'auto' == $design ) {
		$design = wpb_setting( 'calendar_design' );
	}

	$design = apply_filters( 'app_html_title_design', $design, $controller );

	$t = str_replace(
		array( "LOCATION", "WORKER", "SERVICE", "CATEGORY", ),
		array(
			BASE()->get_location_name( $controller->get_location() ),
			BASE()->get_worker_name( $controller->get_worker() ),
			BASE()->get_service_name( $controller->get_service() ),
			BASE()->guess_category_name( $controller->get_service() ),
			),
		$title
	);

	if ( 'compact' != $design ) {
		return '<div class="app-title">' . esc_html( $t ) . '</div>';
	}

	return
	'<div class="app-flex">
	<div class="app-nextprev app-prv"><a class="flex-prev prev" href="#"><em class="app-icon icon-left-open"></em></a></div>
	<div class="app-title">' . esc_html( $t ) . '</div>
	<div class="app-nextprev app-nxt"><a class="flex-next next" href="#"><em class="app-icon icon-right-open"></em></a></div>
	</div>';
}

/**
 * Add paginate links
 * @param $total_pages	integer		Total number of pages
 * @param $paged		integer		Current page number
 * @param $rpp			integer		Records per page
 * @since 3.7.8
 * @return none
 */
function wpb_paginate_links( $total, $paged, $rpp = 20 ) {
	$rpp			= $rpp ?:  wpb_setting( 'records_per_page', 20 );
	$total_pages	= ceil( $total/$rpp );
	$save			= $_SERVER['REQUEST_URI'];

	$_SERVER['REQUEST_URI'] = remove_query_arg( array( 'add_new', 'cpy_from' ) );

	if ( wpb_is_admin() || ! get_option( 'permalink_structure' ) ) {
		$navigation = paginate_links( array(
			'base'		=> add_query_arg( 'paged', '%#%' ),
			'format'	=> '',
			'total'		=> $total_pages,
			'current'	=> max( 1, $paged ),
			'prev_text'	=> '<span aria-label="' . esc_attr__( 'Previous page' ) . '">' . __( '&laquo;' ) . '</span>',
			'next_text'	=> '<span aria-label="' . esc_attr__( 'Next page' ) . '">' . __( '&raquo;' ) . '</span>',
		) );
	} else {
		$navigation =  paginate_links( array(
			'base'		=> str_replace( WPB_HUGE_NUMBER, '%#%', get_pagenum_link( WPB_HUGE_NUMBER ) ),
			'format'	=> '/page/%#%',
			'current'	=> max( 1, $paged ),
			'total'		=> $total_pages,
			'add_args'	=> array( 'page' => false,),
			'prev_text'	=> '<span aria-label="' . esc_attr__( 'Previous page' ) . '">' . __( '&laquo;' ) . '</span>',
			'next_text'	=> '<span aria-label="' . esc_attr__( 'Next page' ) . '">' . __( '&raquo;' ) . '</span>',
		) );
	}

	$_SERVER['REQUEST_URI'] = $save;

	$total_html = '<span class="displaying-num">' . sprintf(
		/* translators: %s: Number of items. */
		_n( '%s item', '%s items', $total ),
		number_format_i18n( $total )
	) . '</span>';

	if ( $total_pages ) {
		$page_class = $total_pages < 2 ? ' one-page' : '';
	} else {
		$page_class = ' no-pages';
	}
?>
	<div class="app-manage-third-column tablenav">
		<div class="tablenav-pages<?php echo $page_class ?>">
			<?php echo $total_html; ?>
			<span class="pagination-links">
			<?php echo $navigation; ?>
			</span>
		</div>
	</div>
<?php
}

/**
 * Whether WPB_EXTRA_SECURITY constant is set
 * @since 3.7.8
 * @return bool
 */
function wpb_extra_security() {
	return defined( 'WPB_EXTRA_SECURITY' ) && WPB_EXTRA_SECURITY;

}

/**
 * Do upgrade to V3.8.0 - Upgrade DB
 * @since 3.7.6
 * @return none
 */
function wpb_upgrade_376() {
	require_once( WPBASE_PLUGIN_DIR . '/includes/install.php' );
	WpBInstall::install();
}

/**
 * Check if advanced features addon version is 3.5.0 or higher
 * @uses static		$wpb_advanced_features_notice
 * @since 3.8.0
 * @return bool		true if required version is installed or addon is not active, false if addon is active and has an earlier version
 */
function wpb_check_advanced_features() {
	static $wpb_advanced_features_notice;

	if ( ! WPB_DEV && class_exists( 'WpBPro' ) ) {
		include_once( ABSPATH . 'wp-admin/includes/plugin.php' );
		$path = 'wp-base-addon-advanced-features/advanced-features.php';
		$data = get_file_data( WP_PLUGIN_DIR.'/'.$path, wpb_default_headers(), 'plugin' );

		if ( is_plugin_active( $path ) && ! empty( $data['Version'] ) && version_compare( $data['Version'], '3.5.0', '<' ) ) {
			if ( ! $wpb_advanced_features_notice && is_admin() ) {
				$wpb_advanced_features_notice = true;
				wpb_notice( sprintf( __( 'Please update Advanced Features addon to the latest version. Minimum Required: 3.5.0 - Installed: %s', 'wp-base' ), $data['Version'] ), 'error' );
			}

			return false;
		}
	}

	return true;
}

/**
 * Prepare a timestamp to be written to the database
 * @param $timestamp	int		Unix timestamp. If left empty, current time
 * @param $timezone		obj		DateTimeZone object
 * @since 3.8.0
 * @return string	Date/time formatted in MySQL format
 */
function wpb_prepare_date( $timestamp = null, $timezone = null ) {

    if ( null === $timestamp ) {
        $timestamp = time();
    } elseif ( ! is_numeric( $timestamp ) ) {
        return false;
    }

	if ( ! function_exists( 'date_create' ) ) {
		return date( 'Y-m-d H:i:s', $timestamp + (int)( get_option( 'gmt_offset' ) * HOUR_IN_SECONDS ) );
	}

    $datetime = date_create( '@' . $timestamp );

    if ( ! $timezone && function_exists( 'wp_timezone' ) ) {
		$timezone = wp_timezone();
    }

	if ( $timezone ) {
		$datetime->setTimezone( $timezone );
	}

	return $datetime->format( 'Y-m-d H:i:s' );
}
	
/**
 * Filter function to select provider by default
 * @since 3.8.9.2
 * @return array
 */
function wpb_wh_default_selected( $wh_selected ) {
	if ( empty( $_POST['app_select_wh'] ) && wpb_is_worker() ) {
		$wh_selected[] = 'worker|'. get_current_user_id();
		$wh_selected = array_unique( $wh_selected );
	}

	return $wh_selected;
}

/**
 * Filter function to select provider by default
 * @since 3.8.9.2
 * @return string
 */
function wpb_wh_annual_default_selected( $wh_selected ) {
	if ( empty( $_POST['app_select_wh_annual'] ) && empty( $_POST['app_select_subject'] ) && wpb_is_worker() ) {
		$wh_selected = 'worker|'. get_current_user_id();
	}

	return $wh_selected;
}

/**
 * Get a cached result of pending bookings count
 * @since 3.8.10.3
 */
function wpb_pending_count(){
	$id		= wpb_cache_prefix() .'pending_count';
	$count	= wp_cache_get( $id );

	if ( false === $count ) {
		global $wpdb;
		$count = (int)$wpdb->get_var("SELECT COUNT(ID) FROM " . BASE()->app_table . " WHERE status='pending' " );
		wp_cache_set( $id, $count );
	}

	return $count;
}
