<?php
/**
 * General functions
 *
 * Functions about time, data handling and other stuff
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

 /**
 * Add meta data field to a booking
 *
 * @since 3.0
 *
 * @param int    $app_id	 App ID.
 * @param string $meta_key   Metadata name.
 * @param mixed  $meta_value Metadata value. Must be serializable if non-scalar.
 * @param bool   $unique     Optional. Whether the same key should not be added.
 *                           Default true.
 * @return int|false Meta ID on success, false on failure.
 */
function wpb_add_app_meta( $app_id, $meta_key, $meta_value, $unique = true ) {
	return WpBMeta::add_metadata( 'app', $app_id, $meta_key, $meta_value, $unique );
}

/**
 * Retrieve app meta field for a booking.
 *
 * @since 3.0
 *
 * @param int    $post_id Post ID.
 * @param string $key     Optional. The meta key to retrieve. By default, returns
 *                        data for all keys. Default empty.
 * @param bool   $single  Optional. Whether to return a single value.
 * @return mixed Will be an array if $single is false. Will be value of meta data
 *               field if $single is true.
 */
function wpb_get_app_meta( $app_id, $key = '', $single = true ) {
	return WpBMeta::get_metadata('app', $app_id, $key, $single);
}

/**
 * Update app meta field based on app ID.
 *
 * Use the $prev_value parameter to differentiate between meta fields with the
 * same key and app ID.
 *
 * If the meta field for the post does not exist, it will be added.
 *
 * @since 3.0
 *
 * @param int    $app_id     Post ID.
 * @param string $meta_key   Metadata key.
 * @param mixed  $meta_value Metadata value. Must be serializable if non-scalar.
 * @param mixed  $prev_value Optional. Previous value to check before removing.
 *                           Default empty.
 * @return int|bool Meta ID if the key didn't exist, true on successful update,
 *                  false on failure.
 */
function wpb_update_app_meta( $app_id, $meta_key, $meta_value, $prev_value = '' ) {
	return WpBMeta::update_metadata( 'app', $app_id, $meta_key, $meta_value, $prev_value );
}

/**
 * Update the metadata cache for the given app IDs.
 * Useful to update cache for several ID before preparing a list, e.g. [app_list]
 *
 * @since 3.0
 *
 * @param int|array $object_ids Array or comma delimited list of object IDs to update cache for
 * @return array|false Metadata cache for the specified objects, or false on failure.
 */
function wpb_update_app_meta_cache($object_ids) {
	return WpBMeta::update_meta_cache( 'app', $object_ids );
}

/**
 * Remove metadata matching criteria from an appt record.
 *
 * You can match based on the key, or key and value. Removing based on key and
 * value, will keep from removing duplicate metadata with the same key. It also
 * allows removing all metadata matching key, if needed.
 *
 * @since 3.0
 *
 * @param int    $app_id 	 Booking ID.
 * @param string $meta_key   Metadata name.
 * @param mixed  $meta_value Optional. Metadata value. Must be serializable if
 *                           non-scalar. Default empty.
 * @return bool True on success, false on failure.
 */
function wpb_delete_app_meta( $app_id, $meta_key, $meta_value = '' ) {
	return WpBMeta::delete_metadata('app', $app_id, $meta_key, $meta_value);
}

 /**
 * Add meta data field to a service
 *
 * @since 3.0
 *
 * @param int    $service_id	 Service ID.
 * @param string $meta_key   Metadata name.
 * @param mixed  $meta_value Metadata value. Must be serializable if non-scalar.
 * @param bool   $unique     Optional. Whether the same key should not be added.
 *                           Default true.
 * @return int|false Meta ID on success, false on failure.
 */
function wpb_add_service_meta( $service_id, $meta_key, $meta_value, $unique = true ) {
	return WpBMeta::add_metadata( 'service', $service_id, $meta_key, $meta_value, $unique );
}

/**
 * Retrieve service meta field for a booking.
 *
 * @since 3.0
 *
 * @param int    $service_id	 Service ID.
 * @param string $key     Optional. The meta key to retrieve. By default, returns
 *                        data for all keys. Default empty.
 * @param bool   $single  Optional. Whether to return a single value.
 * @return mixed Will be an array if $single is false. Will be value of meta data
 *               field if $single is true.
 */
function wpb_get_service_meta( $service_id, $key = '', $single = true ) {
	return WpBMeta::get_metadata('service', $service_id, $key, $single);
}

/**
 * Update service meta field based on service ID.
 *
 * Use the $prev_value parameter to differentiate between meta fields with the
 * same key and service ID.
 *
 * If the meta field for the post does not exist, it will be added.
 *
 * @since 3.0
 *
 * @param int    $service_id     service ID.
 * @param string $meta_key   Metadata key.
 * @param mixed  $meta_value Metadata value. Must be serializable if non-scalar.
 * @param mixed  $prev_value Optional. Previous value to check before removing.
 *                           Default empty.
 * @return int|bool Meta ID if the key didn't exist, true on successful update,
 *                  false on failure.
 */
function wpb_update_service_meta( $service_id, $meta_key, $meta_value, $prev_value = '' ) {
	return WpBMeta::update_metadata( 'service', $service_id, $meta_key, $meta_value, $prev_value );
}

/**
 * Remove metadata matching criteria from a service record.
 *
 * You can match based on the key, or key and value. Removing based on key and
 * value, will keep from removing duplicate metadata with the same key. It also
 * allows removing all metadata matching key, if needed.
 *
 * @since 3.0
 *
 * @param int    $service_id Service ID.
 * @param string $meta_key   Metadata name.
 * @param mixed  $meta_value Optional. Metadata value. Must be serializable if
 *                           non-scalar. Default empty.
 * @return bool True on success, false on failure.
 */
function wpb_delete_service_meta( $service_id, $meta_key, $meta_value = '' ) {
	return WpBMeta::delete_metadata('service', $service_id, $meta_key, $meta_value);
}

 /**
 * Add meta data field to a location
 *
 * @since 3.0
 *
 * @param int    $loc_id	 Location ID.
 * @param string $meta_key   Metadata name.
 * @param mixed  $meta_value Metadata value. Must be serializable if non-scalar.
 * @param bool   $unique     Optional. Whether the same key should not be added.
 *                           Default true.
 * @return int|false Meta ID on success, false on failure.
 */
function wpb_add_location_meta( $loc_id, $meta_key, $meta_value, $unique = true ) {
	return WpBMeta::add_metadata( 'location', $loc_id, $meta_key, $meta_value, $unique );
}

/**
 * Retrieve location meta field for a booking.
 *
 * @since 3.0
 *
 * @param int    $loc_id  Location ID.
 * @param string $key     Optional. The meta key to retrieve. By default, returns
 *                        data for all keys. Default empty.
 * @param bool   $single  Optional. Whether to return a single value.
 * @return mixed Will be an array if $single is false. Will be value of meta data
 *               field if $single is true.
 */
function wpb_get_location_meta( $loc_id, $key = '', $single = true ) {
	return WpBMeta::get_metadata('location', $loc_id, $key, $single);
}

/**
 * Update location meta field based on location ID.
 *
 * Use the $prev_value parameter to differentiate between meta fields with the
 * same key and location ID.
 *
 * If the meta field for the post does not exist, it will be added.
 *
 * @since 3.0
 *
 * @param int    $loc_id     location ID.
 * @param string $meta_key   Metadata key.
 * @param mixed  $meta_value Metadata value. Must be serializable if non-scalar.
 * @param mixed  $prev_value Optional. Previous value to check before removing.
 *                           Default empty.
 * @return int|bool Meta ID if the key didn't exist, true on successful update,
 *                  false on failure.
 */
function wpb_update_location_meta( $loc_id, $meta_key, $meta_value, $prev_value = '' ) {
	return WpBMeta::update_metadata( 'location', $loc_id, $meta_key, $meta_value, $prev_value );
}

/**
 * Remove metadata matching criteria from a location record.
 *
 * You can match based on the key, or key and value. Removing based on key and
 * value, will keep from removing duplicate metadata with the same key. It also
 * allows removing all metadata matching key, if needed.
 *
 * @since 3.0
 *
 * @param int    $loc_id     Location ID.
 * @param string $meta_key   Metadata name.
 * @param mixed  $meta_value Optional. Metadata value. Must be serializable if
 *                           non-scalar. Default empty.
 * @return bool True on success, false on failure.
 */
function wpb_delete_location_meta( $loc_id, $meta_key, $meta_value = '' ) {
	return WpBMeta::delete_metadata('location', $loc_id, $meta_key, $meta_value);
}

/**
 * Remove all metadata of a location record.
 *
 * @since 3.0
 *
 * @param int    $loc_id     Location ID.
 * @return bool True on success, false on failure.
 */
function wpb_delete_location_metadata( $loc_id ) {
	return WpBMeta::delete_metadata_by_oid( 'location', $loc_id );
}


 /**
 * Add meta data field to a payment
 *
 * @since 3.6.3
 *
 * @param int    $pid	 Payment ID.
 * @param string $meta_key   Metadata name.
 * @param mixed  $meta_value Metadata value. Must be serializable if non-scalar.
 * @param bool   $unique     Optional. Whether the same key should not be added.
 *                           Default true.
 * @return int|false Meta ID on success, false on failure.
 */
function wpb_add_payment_meta( $pid, $meta_key, $meta_value, $unique = true ) {
	return WpBMeta::add_metadata( 'payment', $pid, $meta_key, $meta_value, $unique );
}

/**
 * Retrieve payment meta field for a booking.
 *
 * @since 3.6.3
 *
 * @param int    $pid  	  Payment ID.
 * @param string $key     Optional. The meta key to retrieve. By default, returns
 *                        data for all keys. Default empty.
 * @param bool   $single  Optional. Whether to return a single value.
 * @return mixed Will be an array if $single is false. Will be value of meta data
 *               field if $single is true.
 */
function wpb_get_payment_meta( $pid, $key = '', $single = true ) {
	return WpBMeta::get_metadata( 'payment', $pid, $key, $single );
}

/**
 * Update payment meta field based on payment ID.
 *
 * Use the $prev_value parameter to differentiate between meta fields with the
 * same key and payment ID.
 *
 * If the meta field for the post does not exist, it will be added.
 *
 * @since 3.6.3
 *
 * @param int    $pid     	 payment ID.
 * @param string $meta_key   Metadata key.
 * @param mixed  $meta_value Metadata value. Must be serializable if non-scalar.
 * @param mixed  $prev_value Optional. Previous value to check before removing.
 *                           Default empty.
 * @return int|bool Meta ID if the key didn't exist, true on successful update,
 *                  false on failure.
 */
function wpb_update_payment_meta( $pid, $meta_key, $meta_value, $prev_value = '' ) {
	return WpBMeta::update_metadata( 'payment', $pid, $meta_key, $meta_value, $prev_value );
}

/**
 * Remove metadata matching criteria from a payment record.
 *
 * You can match based on the key, or key and value. Removing based on key and
 * value, will keep from removing duplicate metadata with the same key. It also
 * allows removing all metadata matching key, if needed.
 *
 * @since 3.6.3
 *
 * @param int    $pid     Payment ID.
 * @param string $meta_key   Metadata name.
 * @param mixed  $meta_value Optional. Metadata value. Must be serializable if
 *                           non-scalar. Default empty.
 * @return bool True on success, false on failure.
 */
function wpb_delete_payment_meta( $pid, $meta_key, $meta_value = '' ) {
	return WpBMeta::delete_metadata( 'payment', $pid, $meta_key, $meta_value );
}

/**
 * Remove all metadata of a payment record.
 *
 * @since 3.6.3
 *
 * @param int    $pid     Payment ID.
 * @return bool True on success, false on failure.
 */
function wpb_delete_payment_metadata( $pid ) {
	return WpBMeta::delete_metadata_by_oid( 'payment', $pid );
}

/**
 * Get salt. If it does not exist, create it
 * @since 2.0
 */
function wpb_get_salt(){
	# Create a salt, if it doesn't exist from the previous installation
	if ( !$salt = get_option( "appointments_salt" ) ) {
		$salt = mt_rand();
		add_option( "appointments_salt", $salt ); # Save it to be used until it is cleared manually
	}
	return $salt;
}

/**
 * Limit length of a text
 * @param $limit		integer		Character/word limit for text
 * @param $word			bool		Limit is for number of words
 * @since 2.0
 * @return string
 */
function wpb_cut( $text, $limit = 22, $word = false ) {
	$limit		= apply_filters( 'app_chr_limit', $limit );
	$full_text	= $text;

	if ( $word ) {
		$text= wp_trim_words( $text, $limit, '...' );
	} else if ( $limit && strlen( $text ) > $limit + 3 ) {
		$text = mb_substr( $text, 0, $limit, 'UTF-8' ) . '...';
	}

	if ( strlen( $full_text ) > strlen( $text ) ) {
		$text = '<abbr title="'.$full_text.'">'. $text . '</abbr>';
	}

	return $text;
}

/**
 * Fixes modulus operator of php against negative results
 * @since 2.0
 */
function wpb_mod( $a, $n ) {
	return ($a % $n) + ($a < 0 ? $n : 0);
}

/**
 * Packs an array into a string with : as glue
 * @param $sep	string	Separator
 * @return string
 */

function wpb_implode( $input, $sep = ':' ) {
	if ( !is_array( $input ) || empty( $input ) ) {
		return '';
	}

	return $sep. implode( $sep, array_filter( array_unique( $input ) ) ) . $sep;
}

/**
 * Packs a string into an array assuming : as glue
 * @param $sep	string	Separator
 * @return array
 */
function wpb_explode( $input, $sep = ':' ){
	if ( ! is_string( $input ) ) {
		return array();
	}

	return array_filter( array_unique( array_map( 'trim', explode( $sep, ltrim( $input, $sep ) ) ) ) );
}

/**
 * Remove spaces and make comma delimited options unique
 * @return string
 */
function wpb_sanitize_commas( $options, $sort = false ) {
	$arr = explode( ',', $options );
	$arr = array_map( 'trim', $arr );
	$arr = array_unique( $arr );
	if ( 'reverse' == $sort )
		rsort( $arr );
	else if ( $sort )
		sort( $arr );

	return implode( ',', array_filter( $arr ) );
}

/**
* Get timestamp in seconds to measure execution time
* @since 2.0
* @return float
*/
function wpb_microtime( ) {
	list( $usec, $sec ) = explode( " ", microtime() );
	return ( (float)$usec + (float)$sec );
}

/**
 * Converts number of seconds to hours:mins acc to the WP time format setting
 * @param $show_2400	bool	If this is an 'end' time, allow 24:00
 * @return string
 */
function wpb_secs2hours( $secs, $show_2400 = false ) {
	$min = (int)($secs / 60);
	$hours = "00";
	if ( $min < 60 ) {
		$hours_min = $hours . ":" . $min;
	} else {
		$hours = (int)($min / 60);
		if ( $hours < 10 )
			$hours = "0" . $hours;
		$mins = $min - $hours * 60;
		if ( $mins < 10 )
			$mins = "0" . $mins;
		$hours_min = $hours . ":" . $mins;
	}
	if ( BASE()->time_format )
		$hours_min = date( BASE()->time_format, strtotime( $hours_min . ":00" ) );

	# Exception - show 24:00 at midnight only if selected
	if ( $show_2400 && '00:00' === $hours_min && $secs )
		$hours_min = '24:00';

	return $hours_min;
}

/**
 * Convert any time format (incl. most custom formats) to military format
 * @param $time				string	Time value to be converted
 * @param $correct_2400		bool	If this is an 'end' time, behave 24:00 as 00:00
 * @since 1.0.3
 * @return string
 */
function wpb_to_military( $time, $correct_2400 = false ) {

	$time_format = BASE()->time_format;

	# Already in military format
	if ( 'H:i' == $time_format ) {
		if ( $correct_2400 && '24:00' === $time )
			$time = '00:00';

		return $time;
	}

	# In one of the default formats
	if ( 'g:i a' == $time_format || 'g:i A' == $time_format )
		return date( 'H:i', strtotime( $time ) );

	# Custom format. Use a reference time
	# ref is expected to be something like 23saat45dakika
	$ref = date_i18n( $time_format, strtotime( "23:45" ) );
	if ( strpos( $ref, "23" ) !== false )
		$twentyfour = true;
	else
		$twentyfour = false;

	# Now ref could be something like saat,dakika
	$ref = ltrim( str_replace( array( '23', '45' ), ',', $ref ), ',' );
	$ref_arr = explode( ',', $ref );

	if ( isset( $ref_arr[0] ) ) {
		$s = $ref_arr[0]; # separator. We will replace it by :
		if ( ! empty($ref_arr[1]) )
			$e = $ref_arr[1];
		else {
			$e = 'PLACEHOLDER';
			$time = $time. $e; # Add placeholder at the back
		}
		if ( $twentyfour )
			$new_e = '';
		else
			$new_e = ' a';
	} else if ( false === $ref_arr ) {
		# Only possible without separators, e.g. 23saat45dakika becomes 23
		if ( "23" === (string)$ref )
			return date( 'H:i', strtotime( $time .':00' ) );
		else
			return $time; # We do not have an idea what format this can be
	} else {
		return $time; # We do not support this format
	}

	return date( 'H:i', strtotime( str_replace( array($s,$e), array(':',$new_e), $time ) ) );
}

/**
 * Convert military formatted time (H:i) value to current time format
 * @since 2.0
 * @return integer
 */
function wpb_from_military( $time, $only_secs = false ) {
	list( $hours, $mins ) = explode( ':', $time );
	$secs = $hours *60*60 + $mins *60;
	return $only_secs ? $secs: wpb_secs2hours( $secs );
}

/**
 * Find timestamp of first day of month for a given time
 * @param $time		integer		Timestamp for which first day will be found
 * @param $add		integer		How many months to add
 * @return integer (timestamp)
 * @since 1.0.4
 */
function wpb_first_of_month( $time, $add = 0 ) {
	$year = date( "Y", $time );
	$month = date( "n",  $time );

	return mktime( 0, 0, 0, $month+$add, 1, $year );
}

/**
 * Find timestamp of midnight of last day of month for a given time
 * @param $time		integer		Timestamp for which first day will be found
 * @param $add		integer		How many months to add
 * @return integer (timestamp)
 * @since 2.0
 */
function wpb_last_of_month( $time, $add = 0 ) {
	$sign = $add > 0 ? '+' : '-';
	$time = $add ? strtotime( $sign.abs($add).' month', wpb_first_of_month( $time ) ) : $time;
	$year = date( "Y", $time );
	$month = date( "n",  $time );
	$last_day = date( "t",  $time );

	return mktime( 23, 59, 59, $month, $last_day, $year );
}

/**
 * Returns the timestamp of Monday of the current week or selected date
 * This is used for calendars using Monday as week start
 * @param timestamp: Timestamp of the selected date or false for current time
 * @since 3.5.0
 * @return integer (timestamp)
 */
function wpb_monday( $timestamp = false ) {

	$date = $timestamp ? $timestamp : BASE()->_time;

	if ( "Monday" == date( "l", $date )  )
		return strtotime("today", $date );
	else
		return strtotime("last Monday", $date );
}

/**
 * Returns the timestamp of Sunday of the current week or selected date
 * This is used for calendars using Sunday (and sometimes Monday too) as week start
 * @param timestamp: Timestamp of the selected date or false for current time
 * @return integer (timestamp)
 */
function wpb_sunday( $timestamp = false ) {

	$date = $timestamp ? $timestamp : BASE()->_time;

	if ( "Sunday" == date( "l", $date )  )
		return strtotime("today", $date );
	else
		return strtotime("last Sunday", $date );
}

/**
 * Returns the timestamp of Saturday of the current week or selected date
 * This is used for calendars using Saturday as week start
 * @param timestamp: Timestamp of the selected date or false for current time
 * @since 3.0
 * @return integer (timestamp)
 */
function wpb_saturday( $timestamp = false ) {

	$date = $timestamp ? $timestamp : BASE()->_time;

	if ( "Saturday" == date( "l", $date ) && 6 == BASE()->start_of_week ) {
		return strtotime("today", $date );
	} else {
		return strtotime("last Saturday", $date );
	}
}

/**
 * Check if a time is on Saturday or Sunday
 * @param $timestamp	integer		Timestamp to be checked
 * @return bool
 */
function wpb_is_weekend( $timestamp ) {
	$day_name = date( 'l', $timestamp );

	return apply_filters( 'app_is_weekend', ('Saturday' === $day_name || 'Sunday' === $day_name), $timestamp );
}

/**
 * Format duration for front end
 * @param duration: Minutes to be formatted
 * @return string
 * @since 2.0
 */
function wpb_format_duration( $duration ) {
	if ( !$duration )
		return '';

	# First, cases from 24h+
	# Case exact 24h
	if ( 1440 == $duration )
		return '1 '. BASE()->get_text('day');

	# Case 24h - 48h
	if ( $duration > 1440 && $duration < 2880 )
		return intval( ceil( $duration/60 ) ) . ' '. BASE()->get_text('hours');

	# Case 48h+
	$days = intval( ceil( $duration/1440 ) );
	if ( $days >= 2 )
		return $days .' '. BASE()->get_text('days');

	$format = BASE()->get_options('duration_format');

	if ( $duration < 60 || 'minutes' == $format )
		return $duration . ' ' . BASE()->get_text('minutes');

	if ( !$format || 'hours_minutes' == $format ) {
		$hours = floor( $duration/60 );
		if ( $hours > 1 )
			$hour_text = $hours . ' ' . BASE()->get_text('hours');
		else
			$hour_text = $hours . ' ' . BASE()->get_text('hour');

		$mins = $duration - $hours *60;
		if ( $mins ) {
			if ( $mins > 1 )
				$min_text = ' ' . $mins . ' ' . BASE()->get_text('minutes');
			else
				$min_text = ' ' . $mins . ' ' . BASE()->get_text('minute');
		} else {
			$min_text = '';
		}

		return $hour_text . $min_text;
	}

	$hours = intval( ceil( $duration/60 ) );

	if ( 'top_hours' == $format ) {
		if ( $hours > 1 )
			return $hours . ' ' . BASE()->get_text('hours');
		else
			return $hours . ' ' . BASE()->get_text('hour');
	}

	# Fallback
	return number_format_i18n( $duration/60, 1 ) . ' ' . BASE()->get_text( ($hours > 1 ? 'hours' : 'hour') );
}

/**
 * Change a duration into human readable format for admin, e.g. 2h 15min
 *  @param duration: Duration in minutes
 *  @since 3.0
 */
function wpb_readable_duration( $duration ) {
	$hours = floor( $duration/60 );
	if ( $hours > 0 ) {
		$hour_text = $hours . BASE()->get_text('hour_short');
	} else {
		$hour_text = '';
	}

	$mins = $duration - $hours *60;
	if ( $mins ) {
		$min_text = $mins . BASE()->get_text('min_short');
	} else {
		$min_text = '';
	}

	$text = trim( $hour_text . ' '. $min_text );

	return $text ? esc_attr( $text ) : '&nbsp;';
}

/**
 * Fir Input timestamp or date/time, output timestamp
 * Does not check if input timestamp is valid
 * @param $dt:		integer/string		Timestamp or date/time
 * @since 3.0
 * @return 			integer				Timestamp
 */
function wpb_strtotime( $dt ) {
		if ( is_numeric( $dt ) ) {
		return $dt;
	} else 	if ( ! @strtotime( $dt ) ) {
		return 0;
	} else {
		return strtotime( $dt, BASE()->_time );
	}
}

/**
 * For Input timestamp or date/time, output date/time in mysql format (Y-m-d H:i:s)
 * Does not check if input date/time is valid
 * @param $dt			integer/string		Timestamp or date/time
 * @param $date_only	bool|string			If string and 'safe' use safe date format for admin. If true just the date part of time
 * @since 3.0
 * @return 				string
 */
function wpb_date( $dt, $date_only = false ) {
	if ( 'safe' === $date_only ) {
		$format = BASE()->safe_date_format();
	} else if ( $date_only ) {
		$format = 'Y-m-d';
	} else {
		$format = 'Y-m-d H:i:s';
	}

	return is_numeric( $dt ) ? date( $format, $dt ) : date( $format, strtotime( $dt, BASE()->_time ) );
}

/**
 * Return the thousand separator for prices
 * @since  3.0
 * @return string
 */
function wpb_thousands_separator() {
	$separator = apply_filters( 'app_thousands_separator', wpb_setting( 'thousands_separator' ) );
	return stripslashes( $separator );
}

/**
 * Return the decimal separator for prices
 * @since  3.0
 * @return string
 */
function wpb_decimal_separator() {
	$separator = apply_filters( 'app_decimal_separator', wpb_setting( 'decimal_separator' ) );
	return $separator ? stripslashes( $separator ) : '.';
}

/**
 * Calculate prices with 2 significant digits
 * This should be used for internally calculated values
 * @return string
*/
function wpb_round( $price ) {
	if ( empty( $price ) ) {
		$price = 0;
	}
	return (string)round( floatval( $price ), 2 );
}

/**
 * Make a price ready to be written to DB
 * Keeps 2 significant decimal digits
 * This should be used for user entries
 * @since  3.0
 * @return string
 */
function wpb_sanitize_price( $num ) {
	$num = str_replace( wpb_thousands_separator(), '', $num );
    $dotPos = strrpos($num, '.');
    $commaPos = strrpos($num, ',');
    $sep = (($dotPos > $commaPos) && $dotPos) ? $dotPos :
        ((($commaPos > $dotPos) && $commaPos) ? $commaPos : false);

    if (!$sep) {
        return wpb_round( preg_replace("/[^0-9]/", "", $num) );
    }

	$decs 		= preg_replace( "/[^0-9]/", "", substr( $num, $sep+1, strlen($num) ) );
	$decs_len 	= strlen( $decs );
	$decs 		= $decs_len > 2 ? round( $decs/pow(10, $decs_len-2) ) : $decs;

    return preg_replace("/[^0-9]/", "", substr($num, 0, $sep)) . '.' . $decs;
}

/**
 * Format a number to display currency
 * @param $amount			string		Price to be formatted
 * @param $currency			string		Currency code in ISO4217 format. If left empty, website currency setting is used
 * @param $hide_symbol		bool		Whether to hide currency symbol
 * @param $force_decimal	bool		Whether to force display decimal part. If not set, website setting will be used
 * @Since 2.0
 */
function wpb_format_currency( $amount = false, $currency = '', $hide_symbol = false, $force_decimal = null ) {

	if ( ! $currency ) {
		$currency = wpb_setting( 'currency', 'USD' );
	}

	# Get the currency symbol
	include_once( WPBASE_PLUGIN_DIR . '/includes/constant-data.php' );
	$currencies = WpBConstant::currencies();
	$symbol = $currencies[$currency][1];

	# If many symbols are found, rebuild the full symbol
	$symbols = explode( ', ', $symbol );
	if ( is_array( $symbols ) ) {
		$symbol = "";
		foreach ( $symbols as $temp ) {
		 $symbol .= '&#x'.$temp.';';
		}
	} else {
		$symbol = '&#x'.$symbol.';';
	}
	if ( $hide_symbol ) {
		$symbol ='';
	}

	if ( false === $force_decimal || ( null === $force_decimal && "0" === (string)wpb_setting('curr_decimal') ) ) {
		$decimal_place = 0;
		$zero = '0';
	} else {
		$decimal_place = 2;
		$zero = '0'. wpb_decimal_separator().'00';
	}

	# Fix if amount is saved in another decimal point system
	$amount = str_replace( ',', '.', $amount );

	$symbol_position = wpb_setting('curr_symbol_position');

	# Format currency amount
	if ( $amount ) {

		if ( $amount < 0 ) {
			$minus = '-';
			$amount = abs( $amount );
		} else {
			$minus = '';
		}

		$formatted_amount = number_format( $amount, $decimal_place, wpb_decimal_separator(), wpb_thousands_separator() );

		if ( !$symbol_position || $symbol_position == 1  ) {
			return $minus . $symbol . $formatted_amount;
		} else {
			switch ( (int) wpb_setting('curr_symbol_position') ) {
				case 2:		return $symbol . ' ' . $minus . $formatted_amount; break;
				case 3:		return $minus. $formatted_amount . $symbol; break;
				case 4:		return $minus. $formatted_amount . ' ' . $symbol; break;
				default:	return $minus . $symbol . $formatted_amount; break;
			}
		}
	} else if ( false === $amount || '' === (string)$amount ) {
		return $symbol;
	} else {
		if ( !$symbol_position || $symbol_position == 1 ) {
			return $symbol . $zero;
		} else {
			switch ( (int)$symbol_position ) {
				case 2:		return $symbol . ' ' . $zero; break;
				case 3:		return $zero . $symbol; break;
				case 4: 	return $zero . ' ' . $symbol; break;
				default:	return $symbol . $zero; break;
			}
		}
	}
}

/**
 * Converts 1 to 1st, 2 to 2nd, etc
 * https://stackoverflow.com/a/3110033
 * @return string
 */
function wpb_ordinal( $number ) {
	$ends = array('th','st','nd','rd','th','th','th','th','th','th');
	if ((($number % 100) >= 11) && (($number%100) <= 13))
		return $number. 'th';
	else
		return $number. $ends[$number % 10];
}

/**
 * Clear app shortcodes
 * Fast, but not 100% fail safe
 * Use in non-critical content modifications only
 * For critical functions use get_shortcode_regex( wpb_shortcodes() )
 * @since 1.1.9
 */
function wpb_strip_shortcodes( $content ) {
	if ( !is_string( $content ) )
		return $content;
	else
		return preg_replace( '%\[app_(.*?)\]%s', '', $content );
}

/**
 * Provide a list of supported WP BASE shortcodes, i.e. array( 'app_book', app_services',... )
 * User defined shortcodes can be added using the provided filter
 * @param $rebuild		bool		Force to rebuild
 * @since 3.0
 * @return array
 */
function wpb_shortcodes( $rebuild = false ) {
	# During install, an option should have been created
	# If empty, something must have gone wrong, regenerate it
	if ( $rebuild || ! $scodes = get_option( 'wp_base_shortcodes' ) ) {
		include_once( WPBASE_PLUGIN_DIR . '/includes/constant-data.php' );
		$scodes = array_keys( WpBConstant::shortcode_desc() );
		update_option( 'wp_base_shortcodes', $scodes );
	}

	return apply_filters( 'app_shortcodes', $scodes );
}

/**
 * Check if user is connected with a mobile device
 * @since 2.0
 */
function wpb_is_mobile(){
	return apply_filters( 'app_is_mobile', wp_is_mobile() );
}

/**
 * Check if user connected with a tablet
 * @since 2.0
 */
function wpb_is_tablet(){
	return wpb_is_mobile();
}

/**
 * Check if user is connected with a device with iOS operating system
 * @since 2.0
 * @return bool
 */
function wpb_is_ios(){
	if ( empty( $_SERVER['HTTP_USER_AGENT'] ) ) {
		return false;
	}

	$ua = strtolower( $_SERVER['HTTP_USER_AGENT'] );
	return ( strpos( $ua, 'iphone' ) !== false ) || ( strpos( $ua, 'ipod' ) !== false );
}

/**
 * Check if user is connected with a device with android operating system
 * @since 2.0
 * @return bool
 */
function wpb_is_android(){
	if ( empty( $_SERVER['HTTP_USER_AGENT'] ) ) {
		return false;
	}

	$ua = strtolower( $_SERVER['HTTP_USER_AGENT'] );
	$pos_android = strpos( $ua, 'android' );

	if ( $pos_android !== false ) {
		if ( ( strpos( $ua, 'opera' ) !== false && strpos( $ua, 'mini' ) !== false ) ||
			 ( strpos( $ua, 'opera' ) !== false && strpos( $ua, 'mobi' ) !== false ) ||
			 ( strpos( $ua, 'opera' ) !== false && strpos( $ua, 'nintendo dsi' ) !== false ) ||
			 ( strpos( $ua, 'fennec' ) !== false ) ) {
				 
			return false;
			
		} else {
			return true;
		}
	} else {
		return false;
	}
}

/**
 * Determine if this is localhost
 * @since 2.0
 * @return bool
 */
function wpb_is_localhost() {
	$whitelist = array( '127.0.0.1', '::1' );
	if( in_array( $_SERVER['REMOTE_ADDR'], $whitelist) ) {
		return true;
	} else {
		return false;
	}
}

/**
 * Determine if this is an admin page or doing ajax on behalf of an admin page
 * @since 2.0
 * @return bool
 */
function wpb_is_admin() {
	if ( wpb_doing_ajax() ) {

		if ( ! empty( $_POST['post_id'] ) ) {
			return false;
		}

		if ( ! empty( $_POST['bp_displayed_user_id'] ) ) {
			return false;
		}

		global $bp;

		if ( is_object( $bp ) && isset( $bp->displayed_user->domain ) ) {
			return false;
		}

		if ( ! empty( $_POST['screen_base'] ) ) {
			return true;
		}
	}

	return is_admin();
}

/**
 * Check if user has at least one of the required capability
 * @param $cap	array|string	Comma delimited string or array with required caps to check
 * @since 3.0
 * @return bool
 */
function wpb_current_user_can( $cap ) {
	return BASE('User') ? BASE('User')->is_capable( $cap ) : current_user_can( $cap );
}

/**
 * Provide WP BASE settings or a single setting
 * @param $item			string		Pick the required key. Leave empty for all options.
 * @param $fallback		mix			Return value when item is not set
 * @since 2.0
 * @return array
 */
function wpb_setting( $item = null, $fallback = null ) {

	$options = apply_filters( 'app_options', get_option( 'wp_base_options' ), $item, $fallback );
	
	if ( null === $item ) {
		$out = (array)$options;
	} else if ( empty( $options[ $item ] ) && null !== $fallback ) {
		$out = $fallback;
	} elseif ( isset( $options[ $item ] ) ) {
		$out = $options[ $item ];
	} else {
		$out = '';
	}
	
	return apply_filters( 'app_read_options', $out, $item, $fallback );
}

/**
 * Provide an item value from WP BASE payment gateway settings
 * @param $gateway		string		Internal registered name of the gateway
 * @param $key	 		string		Key of setting to be retreived
 * @param $fallback		mixed		Return value if key not set
 * @since 3.0
 * @return mixed
 */
function wpb_gateway_setting( $gateway, $key, $fallback = '' ) {
	$settings = wpb_setting( );
	return isset( $settings['gateways'][ $gateway ][ $key ] ) ? $settings['gateways'][ $gateway ][ $key ] : $fallback;
}

/**
 * Session start
 * WP BASE does not use PHP sessions, but uses its own session cookie. See includes/class.session.php
 * @since 2.0
 * @return string (Session ID)
 */
function wpb_session_start(){
	return BASE()->session()->get_id();
}

/**
 * Read a session variable
 * Will also attempt to start session
 * @param $item 		null|string		Key of the session to be retreived
 * @param $fallback		mixed			Return value if $item is null
 * @since 3.0
 * @return mixed
 */
function wpb_get_session_val( $item, $fallback = null ) {
	$val = BASE()->session()->get( $item );
	
	if ( null === $val ) {
		return $fallback;
	}

	return $val;
}

/**
 * Write a session variable
 * Will also attempt to start session
 * @param $item		integer|string 	Key whose value be written
 * @param $value	mixed 			Value to be written.
 * @since 3.0
 * @return mixed 	$value is reflected back
 */
function wpb_set_session_val( $item, $value ) {
	return BASE()->session()->set( $item, $value );
}

/**
 * Get the placeholder image URL for services etc.
 *
 * @access public
 * @return string
 */
function wpb_placeholder_img_src() {
	return apply_filters( 'app_placeholder_img_src', WPB_PLUGIN_URL . '/images/placeholder.png' );
}

/**
 * Clean variables using wp_unslash and sanitize_text_field. Arrays are cleaned recursively.
 * Non-scalar values are ignored.
 * @since 3.5.1
 * @param string|array $var Data to sanitize.
 * @return string|array
 */
function wpb_clean( $var ) {
	if ( is_array( $var ) ) {
		return array_map( 'wpb_clean', array_map( 'wp_unslash', $var ) );
	} else {
		return is_scalar( $var ) ? sanitize_text_field( wp_unslash ( $var ) ) : $var;
	}
}

/**
 * Replace placeholders in a text
 * For details see _replace method in /includes/core.php
 * @param $text		string		Text to be replaced
 * @param $app		object		WpB_Booking object or app SdClass object
 * @param $context	string		See a.m. method
 * @since 3.7.4.1
 * @return string
 */
function wpb_replace( $text, $app, $context = '' ) {
	return BASE()->_replace( $text, $app, $context );
}
