<?php
/**
 * WPB Front
 *
 * Includes methods and helpers for generating HTML for front end usually out of shortcodes
 * @author		Hakan Ozevin
 * @package		WP BASE
 * @license		http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since		3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpBFront' ) && class_exists( 'WpBCore' ) ) {

class WpBFront extends WpBCore {

	private $account_assets_called;

	private $nof_shortcode = 0;

	public function __construct(){
		parent::__construct();
	}

	/**
     * Add action and filter hooks
     */
	public function add_hooks_front() {
		include_once( WPBASE_PLUGIN_DIR . '/includes/front-listing.php' );						// List Shortcode
		include_once( WPBASE_PLUGIN_DIR . '/includes/front-account.php' );						// Account page
		include_once( WPBASE_PLUGIN_DIR . '/includes/front-cancel.php' );						// Handle cancel requests
		include_once( WPBASE_PLUGIN_DIR . '/includes/front-confirm.php' );						// Handle confirm requests
		include_once( WPBASE_PLUGIN_DIR . '/includes/front-pay-later.php' );					// Handle pay later requests

		add_filter( 'the_posts', array($this, 'maybe_load_assets') );							// Determine if we use shortcodes on the page

		add_shortcode( 'app_book', array( $this, 'book' ) );									// New in V2.0, compact book shortcode
		add_shortcode( 'app_is_mobile', array( $this, 'is_mobile_shortcode') );					// Check if user connected with a mobile
		add_shortcode( 'app_is_not_mobile', array( $this, 'is_not_mobile_shortcode') );			// Check if user connected with a mobile
		add_shortcode( 'app_no_html', array( $this, 'no_html') );								// Cleans everything inside, while loading js and css files
		add_shortcode( 'app_theme', array( $this, 'theme_selector' ) );							// Selects a theme on the front end
		add_shortcode( 'app_confirmation', array( $this,'confirmation' ) );

		add_filter( 'body_class', array( $this, 'body_class' ) );
		add_action( 'init', array( $this, 'front_init' ) );
		add_action( 'app_styles_enqueued', array( $this, 'add_inline_style' ) );

		$this->add_hooks();
	}

	/**
	 * Disable tooltips
	 * @since 3.7.3.1
	 * @return none
	 */
	public function front_init(){
		if ( 'yes' == wpb_setting( 'disable_tooltips' ) && ! WpBDebug::is_debug() ) {
			add_filter( 'app_tooltip_text', '__return_empty_string' );
		}
	}

	/**
	 * Load style and script only when they are necessary
	 * http://beerpla.net/2010/01/13/wordpress-plugin-development-how-to-include-css-and-javascript-conditionally-and-only-when-needed-by-the-posts/
	 *
	 */
	public function maybe_load_assets( $posts ) {
		if ( empty( $posts ) || is_admin() ) {
			return $posts;
		}

		$sc_found = false;

		foreach ( $posts as $post ) {

			if ( ! isset( $post->post_content ) ) {
				continue;
			}

			$post_content = $this->post_content( $post->post_content, $post );
			$post_content = apply_filters( 'app_the_posts_content', $post_content, $post );

			if ( has_shortcode( $post_content, 'app_confirmation') || has_shortcode( $post_content, 'app_book')
				|| has_shortcode( $post_content, 'app_sell_credit' ) || has_shortcode( $post_content, 'app_book_event' ) ) {

				do_action( 'app_shortcode_found', 'confirmation', $post );
				break;
			}

			if ( has_shortcode( $post_content, 'app_account') ) {
				$this->account_assets_called = true;
				do_action( 'app_shortcode_found', 'account', $post );
			}

			if ( strpos( $post_content, '[app_' ) !== false ) {
				$sc_found = true;
				# We do not break here yet, because we may still find conf
			}
		}

		if ( $sc_found ) {
			do_action( 'app_shortcode_found', '', $post );
		}

		return $posts;
	}

	/**
	 * Add date and time format to body class
	 * @since 3.7.6
	 * @return array
	 */
	public function body_class( $classes ) {
		$classes[] = 'app-tformat-'. sanitize_html_class( $this->time_format );
		$classes[] = 'app-dformat-'. sanitize_html_class( $this->date_format );
		return $classes;
	}

	/**
	 * Load styles after main stylesheet
	 * @since 3.4
	 * @return none
	 */
	public function add_inline_style(){
		wp_add_inline_style( 'wp-base', $this->inline_styles() );
	}

	/**
	 * css that will be added to the head, again only for app pages
	 * @since 3.4
	 * @return string
	 */
	public function inline_styles() {

		if ( ! empty( $this->head_printed ) ) {
			return;
		}

		$out = '';
		$color_set = wpb_setting( 'color_set' );

		foreach ( $this->get_legend_items() as $class => $name ) {

			if ( ! $color_set ) {
				$color = wpb_setting( $class."_color", wpb_get_preset( $class, 'base' ) );
			} else {
				$color = wpb_get_preset( $class, $color_set );
			}

			$color = apply_filters( 'app_color', $color, $class );

			$selector = 'has_appointment' == $class ? 'table:not(.app-has-inline-cell) td.' : 'td.';
			$out .= $selector.$class.',div.'.$class.' {background-color: #'. $color .' !important;}';
		}

		if ( ! is_admin() && $css = trim( wpb_setting("additional_css") ) ) {
			$out .= $css;
		}

		$out .= wpb_fix_font_colors( false);

		if ( $this->account_assets_called ) {
			$out .= '.app-account-page table, .app-account-page th, .app-account-page td {border:none;}';
		}

		if ( $caption_color = wpb_setting( 'caption_color' ) ) {
			$out .=
'.app-flex-menu.slider.with-caption .app-flex,
caption .app-flex,
.app-book-flex .app-flex {background: #'.$caption_color.';}';
		}

		if ( $font_color = wpb_setting( 'caption_font_color' ) ) {
			$out .=
'.app-flex-menu.slider.with-caption .app-flex,
.app-flex-menu.slider.with-caption .app-flex a,
caption .app-flex,
.app-book-flex .app-flex,
caption .app-next a,
caption .app-previous a,
.app-book-flex .app-next a,
.app-book-flex .app-previous a {color: #'.$font_color.';}';
		}

		$this->head_printed = true;

		return apply_filters( 'app_inline_styles', $out, $this );
	}

	/**
	 * Get post content. Allow theme builder type themes and page builders not using real post content hook into this filter
	 * Also see /includes/compat.php that already includes compatibility for some popular page builders
	 * @param $post: WP Post object
	 * @param $ajax: Whether this is an ajax request
	 * @since 2.0
	 * @return string
	 */
	public function post_content( $content, $post, $ajax = false ) {
		return apply_filters( 'app_post_content', $content, $post, $ajax );
	}

	/**
	 * Format date for placeholder START_END to be used in calendar title
	 * @param start: Start timestamp
	 * @param end: End timestamp
	 * @since 2.0
	 * @return string
	 */
	public function format_start_end( $start, $end ) {
		if ( "F j, Y" == $this->date_format ) {
			if ( date( "F", $start ) == date( "F", $end ) ) {
				$formatted_end = date_i18n( "j", $end );
				if ( date_i18n( "j", $start ) == $formatted_end ) {
					return date_i18n( "F j", $start ) . ", " . date_i18n( "Y", $start );
				} else {
					return date_i18n( "F j", $start ) . " - " . $formatted_end . ", " . date_i18n( "Y", $start );
				}
			} else if ( date( "Y", $start ) == date( "Y", $end ) ) {
				return date_i18n( "M j", $start ) . " - " . date_i18n( "M j", $end ) . ", " . date_i18n( "Y", $start );
			} else {
				return date_i18n( "M j, Y", $start ) . " - " . date_i18n( "M j, Y", $end );
			}
		} else if ( strpos( $this->date_format, "-" ) !== false ) {
			return date_i18n( $this->date_format, $start ) . " / " . date_i18n( $this->date_format, $end );
		} else {
			return date_i18n( $this->date_format, $start ) . " - " . date_i18n( $this->date_format, $end );
		}
	}

	/**
	 * Shortcode which checks if user connected with a mobile device
	 * @since 2.0
	 */
	public function is_mobile_shortcode( $atts, $content = '' ) {
		if ( ! $content ) {
			return '';
		}

		extract( shortcode_atts( array(), $atts, 'app_is_mobile' ) );

		if ( ! wpb_is_mobile() ) {
			return WpBDebug::debug_text( __( 'Connected with non-mobile device. Content wrapped by shortcode ignored.','wp-base' ) );
		} else {
			return do_shortcode( $content ); # Allow execution of nested shortcodes
		}
	}

	/**
	 * Shortcode which checks if user connected with a mobile device
	 * @since 2.0
	 */
	public function is_not_mobile_shortcode( $atts, $content = '' ) {
		if ( ! $content ) {
			return '';
		}

		extract( shortcode_atts( array(), $atts, 'app_is_mobile' ) );

		if ( wpb_is_mobile() ) {
			return WpBDebug::debug_text( __( 'Connected with mobile device. Content wrapped by shortcode ignored.','wp-base' ) );
		} else {
			return do_shortcode( $content );
		}
	}

	/**
	 * This shortcode does not produce any output
	 * It can be used to load js and style files on a page where a custom template is used
	 * @See /sample/sample-appointments-page.php
	 * @since 2.0
	 */
	public function no_html( $atts, $content = '' ) {
		return '';
	}

	/**
	 * Shortcode to select a theme on the front end
	 * @since 2.0
	 */
	public function theme_selector( $atts ) {

		if ( wpb_is_mobile() ) {
			return '';
		}

		$pars = array(
			'placeholder'	=> '',								# In demo mode Random can be entered
			'title'			=> $this->get_text('select_theme'),	# Title
			'cap'			=> WPB_ADMIN_CAP,					# Capability to view the shortcode output
		);

		extract( shortcode_atts( $pars, $atts, 'app_theme' ) );

		if ( ! wpb_current_user_can( $cap ) ) {
			return '';
		}

		$themes = (array)$this->get_themes();

		# Selected theme
		$cset = isset( $_GET['app_select_theme'] ) && ($_GET['app_select_theme'] === esc_attr(strtolower($placeholder))) ? wpb_clean( $_GET['app_select_theme'] ) : false;
		if ( !$cset )
			$cset = isset( $_GET['app_select_theme'] ) && in_array( $_GET['app_select_theme'], $themes ) ? wpb_clean( $_GET['app_select_theme'] ) : false;
		if ( !$cset )
			$cset = isset( $this->sel_theme ) && in_array( $this->sel_theme, $themes ) ? $this->sel_theme : false;
		if ( !$cset )
			$cset = wpb_get_session_val( 'app_theme', wpb_setting('theme') );

		$s = '';

		$s .= '<div class="app_themes">';
		if ( "0" !== (string)$title ) {
			$s .= '<div class="app_themes_dropdown_title  app-title">';
			$s .= $title;
			$s .= '</div>';
		}

		$href = esc_attr( add_query_arg( array('app_select_theme'=>false, "rand" => $placeholder), get_permalink( wpb_find_post_id() ) ) ) ."&app_select_theme=";

		$s .= '<select onchange="if (this.value) window.location.href=\''.$href.'\'+this.value" name="app_select_theme" class="app-sc app_select_theme">';
		if ( $placeholder )
			$s .= '<option value="'.esc_attr(strtolower($placeholder)).'">' . esc_html( $placeholder ). ' ('. ucwords( $this->selected_theme()). ')</option>';
		foreach ( $themes as $theme ) {
			$theme_name = ucfirst( str_replace( "-", " ", $theme ) );
			$s .= '<option value="'.$theme.'" '. selected( $cset, $theme, false ) . '>' . $theme_name . '</option>';
		}

		$s .= '</select>';
		$s .= '</div>';

		return $s;
	}

	/**
	 * Generate dropdown menu for users
	 * @since 2.0
	 */
	public function users( $atts ) {

		$pars = array(
			'title'				=> $this->get_text('select_user'),
			'show_avatar'		=> 1,
			'avatar_size'		=> '96',
			'order_by'			=> 'display_name',
			'cap'				=> WPB_ADMIN_CAP,
			'role'				=> '',
			'buttonwidth'		=> '400',
		);

		extract( shortcode_atts( $pars, $atts, 'app_users' ) );

		if ( ! wpb_current_user_can( $cap ) ) {
			return WpBDebug::debug_text( 'unauthorised' ); # Invisible to unauthorised users
		}

		if ( ! trim( $order_by ) ) {
			$order_by = 'ID';
		}

		$s = '<div class="app_users">';
		if ( !is_numeric( $title ) || 0 !== intval( $title ) ) {
			$s .= '<div class="app_users_dropdown_title app-title">';
			$s .= $title;
			$s .= '</div>';
		}

		$req_user_id = BASE('User')->read_user_id();

		# Self user is always on top
		$s .= BASE('User')->app_dropdown_users( apply_filters( 'app_users_dropdown_args', array(
				'show_option_all'	=> __('Not registered user','wp-base'),
				'echo'				=> 0,
				'add_email' 		=> true,
				'selected'			=> $req_user_id,
				'name'				=> 'app_select_users',
				'class'				=> 'app-sc app_ms app_select_users',
				'role'				=> $role,
				'data'				=> array(
					'buttonwidth'		=> $buttonwidth,
					'current_user_id'	=> get_current_user_id(),
					'requested_user_id' => $req_user_id,
		) ) ) );

		$s .= '</div>';

		return $s;
	}

	/**
	 * Generate an input field to select start date
	 * @since 2.0
	 */
	public function select_date( $atts ) {

		extract( shortcode_atts( array(
			'title'		=> $this->get_text('select_date'),
			'date'		=> '',
			'history'	=> 0,		// Allow past dates
		), $atts, 'app_select_date' ) );

		# Force a date
		$timestamp = isset( $_REQUEST['app_timestamp'] )
					 ? wpb_strtotime( wpb_clean( urldecode( $_REQUEST['app_timestamp'] ) ) )
					 : $this->_time;

		if ( !$date && (string)$date !== "0" ) {
			$date = date( $this->date_format, $timestamp );
		}

		$s = '<div class="app_date" data-role="date">';
		if ( '0' !== (string)$title ) {
			$s .= '<div class="app_date_title app-title">';
			$s .= $title;
			$s .= '</div>';
		}
		$s .= '<input autocomplete="off" type="text" '.(!$history ? 'data-mindate="0"' : '').' data-maxdate="'.$this->get_app_limit().'" name="app_timestamp" class="app-sc app_select_date ui-toolbar ui-state-default" value="'.$date.'"/>';
		$s .= '</div>';

		return $s;
	}

	/**
	 * Generate dropdown menu for services
	 */
	public function services( $atts, $content = '' ) {

		$atts = shortcode_atts( array(
			'title'					=> $this->get_text('select_service'),
			'placeholder'			=> $this->get_text('select'),			// Since 2.0
			'class'					=> '',									// Since 2.0
			'description'			=> 'excerpt',
			'excerpt_length'		=> 55,
			'order_by'				=> 'sort_order',						// Also valid for categories
			'location'				=> 0, 									// Since 2.0
			'worker'				=> 0, 									// Forcing for a certain worker. since 1.2.3
			'category'				=> 0,									// Since 3.0. List only services in a certain category (force)
			'always_show_cat_title'	=> 0,									// Since 3.0. Displays cat title even empty or forced
			'category_optgroup'		=> 1,									// Use category optgroups. Since 2.0
			'slider'				=> 0,									// Enable Slider. Since 3.1.1
			'design'				=> 'auto',								// Use caption. Since 3.7.7.1
			'_cont'					=> '',									// WpB_Controller object
		), $atts, 'app_services' );

		extract( $atts );

		$controller = $_cont instanceof WpB_Controller ? $_cont : $this->init_controller( $location, 0, $worker, $category, $order_by );

		$services = $controller->services;
		list( $use_cats, $sel_cats ) = $this->selected_cats( $category, $category_optgroup, $order_by ); # Categories

		# Pulldown menu
		$s  = '';
		$s .= $controller->display_errors();
		$s .= '<div class="app_services app-menu '.$class.'">';
		$s .= wpb_title_html( $title, $controller );
		$s .= '<select autocomplete="off" %SERVICES_WITH_PAGE% data-desc="'.esc_attr($description).'" data-ex_len="'.intval($excerpt_length).'" name="app_select_services" class="app-sc app_select_services app_ms">';
		if ( $placeholder || 'no' === wpb_setting('preselect_first_service') ) {
			$p_text = $placeholder ? $placeholder : $this->get_text('select');
			$s .= '<option value="" disabled %MAYBE_SELECTED% hidden>'.esc_attr( $p_text ).'</option>';	# We will replace %MAYBE_SELECTED% placeholder later
		}

		# Flexslider
		$f = '';
		if ( $slider ) {
			$f .= $controller->display_errors();
			$f .= wpb_title_html( $title, $controller, $design );
			$f .= '<div class="flexslider carousel"%ITEM_COUNT%><ul class="slides">';
		}

		$selected_id = 0;
		$with_page = array();

		foreach ( $sel_cats as $cat_id => $cat ) {
			$s .= '%CATEGORY_OPTSTART%';

			$cat_has_service = false;

			foreach ( (array)$services as $service ) {
				if ( $cat_id && ! $this->is_in_category( $cat_id, $service->ID ) ) {
					continue;
				}

				if ( $this->is_internal( $service->ID ) ) {
					continue;
				}

				if ( apply_filters( 'app_services_skip_service', false, $service, $controller ) ) {
					continue;
				}

				$page = apply_filters( 'app_service_page',
						( $service->page ?: ( wpb_get_service_meta( $service->ID, 'image_id' ) ?: false ) ),
						$service, $controller
				);

				if ( $page ) {
					$with_page[] = $service->ID;
				}

				$cat_has_service = true;

				# Check if this is the first service, so it would be displayed by default
				if ( $service->ID == $controller->get_service() ) {
					$sel = ' selected="selected"';
					$selected_id = $service->ID;
				} else {
					$sel = '';
				}

				if ( $slider ) {
					$f .= $controller->slider_item( $service, 'service' );
				} else {
					$s .= '<option value="'.$service->ID.'"'.$sel.'>';
					$s .= apply_filters( 'app_service_name_in_menu', $this->get_service_name( $service->ID ), $service );
					$s .= '</option>';
				}

				# Do something for each service
				do_action( 'app_services_item', $service, $controller, $cat_id );
			}
			$s .= '%CATEGORY_OPTEND%';

			# Replace category optgroup placeholders
			if ( $always_show_cat_title || ( $use_cats && $cat_has_service ) ) {
				$replace = array( '<optgroup data-category_id="'.intval($cat_id).'" label="'.esc_attr( $this->get_category_name( $cat_id ) ).'">', '</optgroup>' );
			} else {
				$replace = array();
			}

			$s = str_replace( array( '%CATEGORY_OPTSTART%', '%CATEGORY_OPTEND%' ), $replace, $s );
		}

		$s .= '</select>';

		# Replace "with page" placeholder
		$replace = ! empty( $with_page ) ? 'data-with_page="'.implode( ',', $with_page ).'"' : '';
		$s = str_replace( '%SERVICES_WITH_PAGE%', $replace, $s );

		# Replace selected placeholder
		$replace = $selected_id ? '' : 'selected';
		$s = str_replace( '%MAYBE_SELECTED%', $replace, $s );

		$s .= '</div>';

		$f .= '</ul></div>';

		if ( $slider && $selected_id ) {
			return str_replace( '%ITEM_COUNT%', ' data-item_count="'. substr_count( $f, 'app-service' ) .'"', $f );
		}

		return $s;
	}

	/**
	 * Prepare categories based on services shortcode attributes
	 * @return array
	 */
	private function selected_cats( $category, $category_optgroup, $order_by ) {
		$cats			= apply_filters( 'app_services_categories', $this->get_categories( $order_by ), $category, $category_optgroup, $order_by );
		$category 		= apply_filters( 'app_services_category', $category );
		$selected_cat	= $category && is_numeric( $category )
						  ? $category
						  : ( ! empty( $_GET['app_category'] ) && isset( $cats[$_GET['app_category']] ) ? wpb_clean( $_GET['app_category'] ) : false);
		$use_cats 		= ! empty( $cats ) && is_array( $cats ) && !$selected_cat && $category_optgroup;
		$sel_cats		= $use_cats
						  ? $cats
						  : ( ! empty($cats[$selected_cat]) ? array( $selected_cat => $cats[$selected_cat] ) : array( 0 => array('name' => '') ) );

		return array( $use_cats, $sel_cats );
	}

	/**
	 * Replace common placeholders in title texts
	 * @since 3.0
	 * @return string
	 */
	public function calendar_title_replace( $title, $controller ) {
		return str_replace(
			array( "LOCATION", "WORKER", "SERVICE", "CATEGORY", ),
			array(
				$this->get_location_name( $controller->get_location() ),
				$this->get_worker_name( $controller->get_worker() ),
				$this->get_service_name( $controller->get_service() ),
				$this->guess_category_name( $controller->get_service() ),
				),
			$title
		);
	}

	/**
	 * Generate html codes for calendar subtitle
	 * @since 2.0
	 * @return string
	 */
	public function calendar_subtitle( $logged, $notlogged ) {
		$c ='';
		if ( is_user_logged_in() || 'yes' != wpb_setting("login_required") ) {
			if ( '0' !== (string)$logged )
				$c .= '<span>' . $logged . '</span>';
		} else if ( !( BASE('Login') && BASE('Login')->is_login_active() ) ) {
			if ( !is_numeric( $notlogged ) || 0 !== intval( $notlogged ) ) {
				$c .= str_replace(
					array( 'LOGIN_PAGE', 'REGISTRATION_PAGE' ),
					array( '<a class="appointments-login_show_login" href="'.esc_attr( wp_login_url( get_permalink() ) ).'">'. $this->get_text('login') .'</a>',
					'<a class="appointments-register" href="' . wpb_add_query_arg( 'redirect', get_permalink(), wp_registration_url() ) . '">'. $this->get_text('register') . '</a>'
					),
					$notlogged );
			}
		} else if ( '0' !== (string)$notlogged ) {
			$c .= '<div class="app-sc appointments-login">';
				$c .= str_replace(
					array( 'LOGIN_PAGE', 'REGISTRATION_PAGE' ),
					array( '<a class="appointments-login_show_login" href="'.esc_attr( wp_login_url( get_permalink() ) ).'">'. $this->get_text('login') .'</a>',
					'<a class="appointments-register" href="' . wpb_add_query_arg( 'redirect', get_permalink(), wp_registration_url() ) . '">'. $this->get_text('register') .'</a>'
					),
					$notlogged );
			$c .= '<div class="appointments-login_inner">';
			$c .= '</div>';
			$c .= '</div>';
		}

		return $c;
	}

	/**
	 * Normalize calendar start timestamp from settings or app_timestamp value
	 * @param $start	integer|string	Timestamp or date/time
	 * @since 3.0
	 * @return	integer	Timestamp
	 */
	public function calendar_start_ts( $start ) {
		if ( 'auto' === $start && $maybe_first = $this->find_first_free_slot() ) {
			$start = $maybe_first;
		} else if ( ! empty( $_REQUEST['app_timestamp'] ) ) {
			$start =  wpb_strtotime( wpb_clean( urldecode( $_REQUEST['app_timestamp'] ) ) );
		} else if ( $start ) {
			$start = wpb_strtotime( $start );
		} else {
			$start = $this->_time;
		}

		return $start;
	}

	/**
	 * Initiate controller
	 * @since 3.0
	 * @return object	WpB_Controller object
	 */
	public function init_controller( $location, $service, $worker, $category = 0, $order_by = 'sort_order', $force_priority = false ) {
		return new WpB_Controller( new WpB_Norm( $location, $service, $worker, $category ), $order_by, $force_priority );
	}

	/**
	 * Compact shortcode function that generates a full appointment page
	 * @since 2.0
	 */
	public function book( $atts ) {

		$scode = wpb_esc_json( wp_json_encode( array( 'app_book' => wp_unslash( $atts ) ) ) );

		$atts = shortcode_atts( array(
			'title'				=> wp_unslash( $this->get_text( 'weekly_title' ) ), // Default also works for monthly
			'mobile_title'		=> 0,
			'location_title'	=> wp_unslash( $this->get_text('select_location') ),
			'service_title'		=> wp_unslash( $this->get_text('select_service') ),
			'worker_title'		=> wp_unslash( $this->get_text('select_provider') ),
			'duration_title'	=> wp_unslash( $this->get_text('select_duration') ),
			'pax_title'			=> wp_unslash( $this->get_text('select_seats') ),
			'id'				=> '',								// Since 3.6.2.1. id of the wrapper
			'location'			=> 0,
			'service'			=> 'auto',
			'category'			=> 0,								// Since 3.0. Force a category
			'worker'			=> 0,								// Set "auto" to select worker of page if this is a bio page
			'order_by'			=> 'sort_order',					// Name, id or sort_order. Single setting for all lsw + cat
			'description' 		=> 'auto',							// Description source: content, excerpt, etc
			'type'				=> 'monthly',						// Table, weekly, monthly. Flex (Pro)
			'mobile_type'		=> 0,								// If zero, follow 'type'
			'columns'			=> 'date,day,time,button',			// Table columns when type is table
			'columns_mobile'	=> 'date_time,button',
			'display'			=> 'minimum',						// Since 3.0. full, with_break, minimum
			'mode'				=> 1,								// Flex mode
			'mobile_mode'		=> 6,								// Flex mode for mobile
			'range'				=> 0,								// How much to display: week, 2weeks, month, 2months, day, number. Weekly and monthly calendar old "count" parameter is same as numerical range
			'start'				=> 0,
			'from_week_start'	=> 'auto',							// Days start from start day of the week for vertical flex, does not for horizontal
			'add'				=> 0,
			'swipe'				=> 0,								// Use swipe mobile function. When swipe=1 only range=1day is allowed
			'select_date'		=> wpb_is_mobile() ? 0 : 1,
			'logged'			=> 0,
			'notlogged'			=> wp_unslash( $this->get_text('not_logged_message') ),
			'slider'			=> 'auto',
			'use_cart'			=> 'auto',
			'fields'			=> '',
			'no_pref_last'		=> 0,								// No preference in latest position. Since V3.4
			'_final_note_pre'	=> '',								// Required for Paypal Express (Please confirm ... amount)
			'_final_note'		=> '',
			'_app_id'			=> 0,								// Required for Paypal Express
			'_button_text'		=> wp_unslash( $this->get_text('checkout_button') ), // Required for Paypal Express
			'_layout'			=> 600,								// Two divide confirmation form into 2 columns
			'_countdown'		=> 'auto',							// since 3.0 use countdown to refresh the page or not. 'auto' determines if a refresh is better
			'_countdown_hidden'	=> 0,								// whether countdown will be hidden
			'_countdown_title'	=> wp_unslash( $this->get_text('conf_countdown_title') ), // Use 0 to disable
			'_continue_btn'		=> 1,
			'_editing'			=> 0,								// Required for Paypal Express (Automatically set to 2)
			'_just_calendar'	=> 0,								// Exclude pagination + confirmation form and outer wrapper app-compact-book-wrapper
			'_hide_cancel'		=> 0,								// Hide cancel button
			'_force_priority'	=> '',
			'_force_min_time'	=> 0,
			'_hide_next'		=> '',
			'_hide_prev'		=> '',
			'_called_by_widget' => '',
			'_disable_seats'	=> 0,
			'_payment_required' => 'auto',							 // since V3.8.6
			'design'			=> 'auto',							 // Since V3.7.6
			'caption'			=> '',
		), $atts, 'app_book' );

		extract( $atts );

		if ( 'auto' == $design ) {
			$design = wpb_setting( 'calendar_design' );
		}

		$error 		= '';
		$sel_mode 	= wpb_is_mobile() ? $mobile_mode : $mode;
		$sel_type 	= wpb_is_mobile() ? ( '0' === (string)$mobile_type ? strtolower( $type ) : strtolower( $mobile_type )) : strtolower( $type );

		$allowed_types = array( 'table', 'weekly', 'monthly', 'week', 'month' );

		if ( BASE('Pro') ) {
			$allowed_types[] = 'flex';
		}

		if ( ! in_array( $sel_type, $allowed_types ) ) {
			$error = 'type ';
		}

		if ( 'auto' == $from_week_start && 'flex' == $sel_type ) {
			$from_week_start = $sel_mode < 5; # Week start for vertical layouts
		}

		list( $range, $pag_step, $pag_unit, $error ) = $this->book_range( $range, $sel_type );

		if ( $error ) {
			return WpBDebug::debug_text( sprintf( __( 'Check "%s" attribute(s) in Book shortcode', 'wp-base'), $error ) );
		}

		$use_swipe = wpb_is_mobile() && $swipe && ('table' === $sel_type || 'flex' === $sel_type);

		$controller = $this->init_controller( $location, $service, $worker, $category, $order_by, $_force_priority );

		$ret  = '';
		$ret .= $controller->display_errors();

		$add_location	= $controller->show_locations_menu();
		$add_service	= $controller->show_services_menu();
		$add_recurring	= $this->is_recurring( $controller->get_service() );
		$add_seldur		= BASE('SelectableDurations') && BASE('SelectableDurations')->is_duration_selectable( $controller->get_service() );
		$set			= wpb_setting('client_selects_worker');
		$add_worker		= $controller->show_workers_menu() && $controller->workers && 'no' != $set && 'forced_late' != $set;
		$add_seats		= ! $_disable_seats && BASE('GroupBookings') && BASE('GroupBookings')->is_enabled( $controller->get_service() );
		$cnt			= count( array_filter( array( $add_location, $add_service, $add_worker, $add_recurring, $add_seldur, $add_seats ) ) );
		$slider_used	= false;

		if ( 'auto' == $slider ) {
			$sname = wpb_is_mobile() ? 'slider_mobile' : 'slider';
			$slider = 'yes' === wpb_setting( $sname ) && count( array_filter( array( $add_location, $add_service, $add_worker ) ) ) > 0;
		}

		if ( $cnt > 3 && $add_recurring ) {
			$cnt = $cnt + 1; # Add 1 for recurring
		}

		$sp_class = $cnt > 0 ?  'app_'.$cnt.'_basis app-flex-item'  : '';

		if ( $add_seats ) {
			$sp_class = $sp_class . ' has-seats';
		}

		if ( ! $_just_calendar ) {
			$ret  .= '<div class="app-compact-book-wrapper">';
		}

		if ( ! $id ) {
			$this->nof_shortcode = $this->nof_shortcode + 1;
			$id = 'app_book_'. $this->nof_shortcode;
		}

		$ret .= '<div data-scode="'.$scode.'" '. ($id ? 'id="'.sanitize_key( $id ).'"' : '').' class="app-scode app-compact-book-wrapper-gr1'.(wpb_is_mobile() ? ' app-mobile' : '').'">';
		$ret .= '<div class="app-flex-menu'.($slider ? ' slider' : '').('compact' == $design ? ' with-caption' : '').'">';

		$common_args = array(
			'location'		=> $controller->get_location(),
			'service'		=> $controller->get_service(),
			'category'		=> $category,
			'worker'		=> $controller->get_worker(),
			'class'			=> $sp_class,
			'description' 	=> $description,
			'order_by'		=> $order_by,
			'slider'		=> $slider && ! $slider_used,
			'design'		=> $design,
			'_cont'			=> $controller,
		);

		# Manage Menu display order acc. to priority
		$pri = apply_filters( 'app_lsw_priority', wpb_setting( 'lsw_priority', WPB_DEFAULT_LSW_PRIORITY ), $controller );

		foreach ( str_split( $pri ) as $lsw ) {

			switch ( $lsw ) {
				case 'L':
					if ( $add_location ) {
						$ret .= BASE('Locations')->locations( array_merge( array( 'title' => $location_title ), $common_args ) );
						$slider_used = true;
					}
					break;
				case 'S':
					if ( $add_service ) {
						$ret .= $this->services( array_merge( array( 'title' => $service_title ), $common_args ) );
						$slider_used = true;
					}
					if ( $add_seldur ) {
						$ret .= BASE('SelectableDurations')->durations( array_merge( array( 'title' => $duration_title ), $common_args ) );
					}
					break;
				case 'W':
					if ( $add_worker  ) {
						$ret .= BASE('SP')->service_providers( array_merge( array( 'title' => $worker_title, 'no_pref_last' => $no_pref_last ), $common_args ) );
						$slider_used = true;
					}
					break;
			}
		}

		if ( $add_seats ) {
			$ret .= BASE('GroupBookings')->seats( array_merge( array( 'title' => $pax_title ), $common_args ) );
		}

		if ( $add_recurring ) {
			$ret .= BASE('Recurring')->recurring( $common_args );
		}

		$ret .= '</div>'; # Close flex-menu

		$start = $this->calendar_start_ts( $start );

		switch ( $sel_type ) {
			case 'flex'	:
			case 'table'	: $pagination = $use_swipe ? '' : $this->pagination( array('_hide_next'=>$_hide_next,'_hide_prev'=>$_hide_prev,'unit'=>$pag_unit,'step'=>$pag_step,'disable_legend'=>1,'start'=>$start,'select_date'=>$select_date) ); break;
			case 'week'		:
			case 'weekly'	: $pagination = $this->pagination( array('_hide_next'=>$_hide_next,'_hide_prev'=>$_hide_prev,'unit'=>'week','step'=>$pag_step,'start'=>$start,'select_date'=>$select_date) ); break;
			case 'month'	:
			case 'monthly'	: $pagination = $this->pagination( array('_hide_next'=>$_hide_next,'_hide_prev'=>$_hide_prev,'unit'=>'month','step'=>$pag_step,'start'=>$start,'select_date'=>$select_date) ); break;
			default			: $pagination = ''; break;
		}

		$common_args_cal = array(
			'location'			=> $controller->get_location(),
			'service'			=> $controller->get_service(),
			'category'			=> $category,
			'worker'			=> $controller->get_worker(),
			'order_by'			=> $order_by,
			'logged'			=> $logged,
			'notlogged' 		=> $notlogged,
			'title'				=> $this->book_main_title( $title, $mobile_title, $sel_type ),
			'start'				=> $start,
			'range'				=> $range,
			'swipe'				=> $swipe,
			'display'			=> $display,
			'from_week_start'	=> $from_week_start,	# Table and Flex only
			'columns'			=> $columns,
			'columns_mobile'	=> $columns_mobile,
			'_cont'				=> $controller,
			'_force_min_time'	=> $_force_min_time,
			'design'			=> $design,
			'_caption'			=> $caption,
		);

		if ( 'table' === $sel_type ) {
			$ret .= $this->book_table( array_merge( array( 'columns' => $columns, 'class' => 'app-book-child' ), $common_args_cal ) );
		} else if ( 'flex' === $sel_type ) {
			if ( ! BASE('Pro') ) {
				return WpBDebug::debug_text( sprintf( __( 'Check "%s" attribute in Book shortcode', 'wp-base'), 'type' ) );
			}

			$ret .= BASE('Pro')->book_flex( array_merge( array( 'mode' => $sel_mode, 'class' => 'app-book-child' ), $common_args_cal ) );
		} else {
			$class = $pag_step > 1 && !wpb_is_mobile() && 'monthly' != $sel_type ? 'app_2column app-book-child' : 'app-book-child';

			for ( $i = 1; $i < 13; $i++ ) {
				$args = array_merge( array( 'class' => $class, 'add' => ($i - 1 + $add) ), $common_args_cal );
				if ( 'weekly' === $sel_type || 'week' === $sel_type ) {
					$ret .= $this->calendar_weekly( $args );
				} else {
					$ret .= $this->calendar_monthly( $args );
				}

				if ( $i >= $pag_step ) {
					break;
				}
			}
		}

		$ret .= '<div style="clear:both"></div>';
		$ret .= '</div>'; # Close gr1

		$ret  = apply_filters( 'app_book_after_gr1', $ret, $atts, $common_args, $common_args_cal );

		if ( $_just_calendar ) {
			return $ret;
		}

		$ret .= '<div class="app-compact-book-wrapper-gr2'.(wpb_is_mobile() ? ' app-mobile' : '').'">';

		if ( 'compact' != $design && ! $use_swipe ) {
			$ret .= $pagination;
		}

		$ret .= $this->confirmation( array(
			'use_cart'			=> $use_cart,
			'fields'			=> $fields,
			'layout'			=> $_layout,
			'button_text'		=> $_button_text,
			'countdown'			=> $_countdown,
			'countdown_hidden'	=> $_countdown_hidden,
			'countdown_title'	=> $_countdown_title,
			'continue_btn'		=> $_continue_btn,
			'_editing'			=> $_editing,
			'_app_id'			=> $_app_id,
			'_hide_cancel'		=> $_hide_cancel,
			'_final_note_pre'	=> $_final_note_pre,
			'_final_note'		=> $_final_note,
			)
		);
		$ret .= '</div>'; # Close gr2
		$ret .= '</div>'; # Close compact-book-wrapper

		return $ret;
	}

	/**
	 * Pick main default title depending on type
	 * @return string
	 */
	private function book_main_title( $title, $mobile_title, $sel_type ) {
		$main_title	= wpb_is_mobile() ? $mobile_title : $title;

		if ( '0' === (string)$main_title ) {
			if ( wpb_is_mobile() ) {
				$main_title = $title;
			} else {
				$main_title = 0;
			}
		} else if ( 1 === $main_title ) {
			if ( 'monthly' === $sel_type ) {
				$main_title = $this->get_text('monthly_title');
			} else {
				$main_title = $this->get_text('weekly_title');
			}
		}

		return $main_title;
	}

	/**
	 * Find range and paging parameters for book in table view
	 * @return array
	 */
	public function book_range( $range, $sel_type ) {
		$error = '';

		if ( '0' === (string)$range ) {
			if ( 'weekly' == $sel_type ) {
				$range = '2 weeks';
			} else if ( 'flex' == $sel_type ) {
				$range = '1 week';
			} else if ( 'table' == $sel_type ) {
				$range = 10;
			} else {
				$range = '1 month';
			}
		}

		$_range = trim( str_replace( array( 'months','month','weeks','week','days','day', ), '', $range ) );

		# Weekly and monthly calendar "count" parameter is same as numerical range
		if ( strpos( $range, 'month' ) !== false ) {
			$pag_step = is_numeric( $_range) ? $_range : 1;
			$pag_unit = 'month';
		} else if ( strpos( $range, 'week' ) !== false ) {
			$pag_step = is_numeric( $_range) ? $_range : 1;
			$pag_unit = 'week';
		} else if ( strpos( $range, 'day' ) !== false ) {
			$pag_step = is_numeric( $_range) ? $_range : 1;
			$pag_unit = 'day';
		} else if ( is_numeric( $range ) && $range > 0 ) {
			$pag_step = $range;
			$pag_unit = 'number';
		} else {
			$error = 'range';
		}

		return array( $range, $pag_step, $pag_unit, $error );
	}

	/**
	 * Shortcode function to generate table/list view booking
	 */
	public function book_table( $atts ) {

		$args1 = shortcode_atts( array(
			'title'			=> $this->get_text('weekly_title'),
			'book_now'		=> $this->get_text('book_now_short'),
			'logged'		=> '',
			'notlogged'		=> $this->get_text('not_logged_message'),
			'location'		=> 0,
			'service'		=> 0,
			'worker'		=> 0,
			'start'			=> 0,									// Will not work for range=number
			'add'			=> 0,									// How many days to add
			'class'			=> '',									// Add a class
			'range'			=> 10,									// What to display: week, 2weeks, month, 2months, day, number
			'complete_day'	=> 0,									// Forces to complete the day when range=number
			'net_days'		=> 0,									// Compansate for empty days
			'columns'		=> 'date,day,time,date_time,button',	// date_time will be hidden for wide area
			'columns_mobile'=> 'time,button',
			'tabletools'	=> 0,
			'id'			=> '',
			'swipe'			=> 0,
			'only_buttons'	=> 0,									// Omit table and all columns except buttons
			'_max'			=> 99999,								// Maximum number while trying to finish day (Total number of rows in the table)
			'_min'			=> 1,									// Minimum number of slots at which break is allowed at the end of a day
			'_tablesorter'	=> 1,
			'_cont'			=> '',									// since 3.0. WpB_Controller object
			'design'		=> '',
			'_caption'		=> '',
			'caption'		=> 'START_END',
		), $atts, 'app_book_table' );

		extract( $args1 );

		$controller = $_cont instanceof WpB_Controller ? $_cont : $this->init_controller( $location, $service, $worker );
		$ret  = '';
		$ret .= $controller->display_errors();

		$use_swipe 		= (bool)(wpb_is_mobile() && $swipe);
		$args1['start'] = $start_ts = $this->calendar_start_ts( $start );

		if ( $use_swipe ) {
			$args1['range'] = '1day';
			$args1['class'] = trim( $class . ' app-swipe-child' );
			$args4 = $args3 = $args2 = $args1;

			$args2['start'] = date( 'Y-m-d', $start_ts + 24*3600 ); 	# Doesnt need to be 00:00. book_table will correct it
			$args3['start'] = date( 'Y-m-d', $start_ts + 2*24*3600 );
			$args4['start'] = date( 'Y-m-d', $start_ts + 3*24*3600 );
			$upper_limit 	= $this->get_upper_limit();

			$ret .= '<div id="app-slider" class="app-swipe"><div class="app-swipe-wrap">' .
					$this->book_table_( $args1, $controller ) . $this->book_table_( $args2, $controller ). 
					$this->book_table_( $args3, $controller ) . $this->book_table_( $args4, $controller );

			for ( $i = 4; $i <= $upper_limit; $i++ ) {
				$ret .= '<div class="app-sc app-wrap app-swipe-child" data-type="book_table" data-title="'.esc_attr($title).'"';
						' data-logged="'.esc_attr($logged).'" data-notlogged="'.esc_attr($notlogged).'" data-start-ts="'.($start_ts + $i*24*3600).'"></div>';
			}
			$ret .= '</div></div>';
		} else {
			$ret .= $this->book_table_( $args1, $controller );
		}

		return $ret;
	}

	/**
	 * Helper for book table
	 */
	public function book_table_( $pars, $controller ) {

		extract( $pars );

		list( $time, $num_limit, $upper_limit, $compans, $error ) = $this->book_table_start_ts( $start, $range, $add, $net_days, $controller );

		if ( $error ) {
			return $error;
		}

		# Prepare table HTML
		$c  = '<div class="app-sc app-wrap'. (wpb_is_admin() ? ' app-wrap-admin ' : ' '). $class.'" data-start-ts="'.$time.'">';

		if ( 'compact' != $design ) {
			$c .= $this->book_table_title( $title, $time, $upper_limit, $num_limit, $controller );
			$c .= $this->calendar_subtitle( $logged, $notlogged );
		}

		$args 			= compact( array_keys( $pars ) );
		$allowed		= $only_buttons
						? array('button')
						: array( 'date','day','time','date_time','server_date_time','server_day','seats_total','seats_left','seats_total_left','button' );
		$allowed		= apply_filters( 'app_book_table_allowed_columns', $allowed, $args );
		$_columns		= wpb_is_mobile() && trim( $columns_mobile ) ? $columns_mobile : $columns;
		$is_cols_def	= 'date,day,time,date_time,button' === $_columns;
		$cols			= array_map( 'strtolower', explode( ',', wpb_sanitize_commas( $_columns ) ) );
		$colspan		= 0;
		$class			= $class .' '. (wpb_is_mobile() ? 'app-mobile' : '');

		$c .= $only_buttons
			? '<div class="app-book '.$class.'">'
			: '<table data-last_values="%LASTVAL_PLaCeHOLDER%" id="'.$id.'"'.
			  ' class="app-book dt-responsive display '.$class.'" data-time="'.date_i18n( $this->dt_format, $time ).'" ><thead><tr>';

		if ( 'compact' == $design ) {
			$caption = $_caption ?: $caption;
			$class	 = 'app-caption-'. strtolower( sanitize_html_class( $caption ) );
			$caption = str_replace(
				array( 'START_END', 'START' ),
				array( '%STND_PLaCeHOLDER%', date_i18n( $this->date_format, $start ) ),
				$caption
			);

			$caption = $this->calendar_title_replace( $caption, $controller );

			list( $range, $pag_step, $pag_unit, $error ) = $this->book_range( $range, 'table' );
			$c .= '<caption class="'.$class.'"><div class="app-flex">'. $this->prev( $time, $pag_step, $pag_unit ).
				'<div class="week-range">'.$caption.'</div>'. $this->next( $time, $pag_step, $pag_unit ).'</div></caption>';
		}

		# Prepare table headers
		foreach( $cols as $col ) {

			if ( ! in_array( $col, $allowed ) ) {
				continue;
			}

			$colspan++;

			$default_hidden = $is_cols_def && 'date_time' == $col ? ' default-hidden' : '';
			$c .= $only_buttons ? '' : '<th class="app-book-col app-book-'.strtolower($col).$default_hidden.'">';
			$c .= $this->table_col_name( $col, $only_buttons );
			$c .= $only_buttons ? '' : '</th>';
		}

		$c .= $only_buttons ? '' : '</tr></thead><tbody>';

		# Find free time slots
		$more_day		= 0;
		$added			= 0;
		$scan_time		= false;
		$slots			= array();
		$first_start	= $time;
		$upper_limit	= min( $upper_limit, $this->get_app_limit() );
		$final			= $time + $upper_limit*DAY_IN_SECONDS;
		$is_daily		= $controller->is_daily();
		$format 		= $is_daily ? $this->date_format : $this->dt_format;

		$calendar = new WpB_Calendar( $controller );
		$calendar->setup( array(
			'disable_limit_check' 	=> true,
			'assign_worker'			=> ! $controller->get_worker() && $controller->workers,
		) );

		for ( $d = $time; $d < $final + $more_day; $d = $d + DAY_IN_SECONDS ) {

			if ( $calendar->doing_lazy_load() ) {
				break;
			}

			$slots = array_merge( $slots, $calendar->find_slots_in_day( $d ) );

			# Check if we are in limits
			if ( $scan_time = $calendar->is_hard_limit_exceeded() ) {
				break;
			}

			$count = count( $slots );

			if ( $num_limit && $count >= $num_limit ) {
				break;
			}

			# After last step, Add extra day if we could not reach the goal
			if ( $compans && $d === ( $final - DAY_IN_SECONDS ) && $num_limit < $count ) {
				$more_day	= $more_day + DAY_IN_SECONDS;
				$final		= $final + DAY_IN_SECONDS;
			}
		}

		# Save latest end timestamp
		$last_end = $d;

		# Start creating html inside the table
		foreach ( $slots as $slot ) {

			if ( $num_limit ) {
				if ( (! $complete_day && $added >= $num_limit) || ($complete_day && $added >= $_max && $added <= $_min) ) {
					$last_end = $slot->get_start();
					break;
				}
			}

			$added++;

			$c .= $only_buttons ? '' : '<tr class="app-book-'.strtolower( date("l", $slot->get_start() ) ).'">';

			foreach( $cols as $col ) {

				if ( ! in_array( $col, $allowed ) ) {
					continue;
				}

				$default_hidden = $is_cols_def && 'date_time' == $col ? ' default-hidden' : '';
				$c .= $only_buttons ? '' : '<td class="app-book-'.strtolower($col).$default_hidden.'">';

				switch ( $col ) {
					case 'date':				$c .= $slot->client_date(); break;
					case 'day':					$c .= $slot->client_day(); break;
					case 'server_day':			$c .= date_i18n( 'l', $slot->get_start() ); break;
					case 'time':				$c .= $is_daily ? '' : $slot->client_time(); break;
					case 'date_time':			$c .= $slot->client_dt(); break;
					case 'server_date_time':	$c .= date_i18n( $format, $slot->get_start() ); break;
					case 'seats_total':
					case 'seats_left':
					case 'seats_total_left': 	$c .= $this->table_cell_seats( $col, $slot ); break;
					case 'button' :				$c .= $this->table_cell_button( $slot, $book_now, $only_buttons, $controller ); break;
					case $col:					$c .= apply_filters( 'app_book_table_add_cell', '', $col, $slot, $args ); break;
				}
				$c .= $only_buttons ? '' : '</td>';
			}
			$c .= $only_buttons ? '' : '</tr>';
		}

		if ( ! count( $slots ) ) {
			$none_free = '<span class="app_book_table_full">'. $this->get_text('no_free_time_slots'). '</span>';
			$c .= $only_buttons ? $none_free : '<tr><td colspan="'.$colspan.'">'.$none_free.'</td></tr>';
		}
		$c .= $only_buttons ? '<div style="clear:both"></div></div>' : '</tbody></table>';

		$c  = apply_filters( 'app_book_table_after_table', $c );

		if ( (bool)$scan_time ) {
			$c .= WpBDebug::debug_text( sprintf( __('Hard limit activated. Execution time: %s secs.', 'wp-base' ), number_format( $scan_time, 1 ) ) );
		}

		$arr = array(
			'start' 	=> $first_start,
			'end' 		=> $last_end,
		);

		# Replace placeholders in HTML
		$c = str_replace( '%STND_PLaCeHOLDER%', $this->format_start_end( $first_start, $last_end ), $c );
		$c = str_replace( '%SON_PLaCeHOLDER%', date_i18n( $this->date_format, $last_end ), $c );
		$c = str_replace( '%LASTVAL_PLaCeHOLDER%', $arr ? htmlspecialchars( wp_json_encode( $arr ) ) : '', $c );

		$calendar->save_cache();

		$c .= '</div>'; # app-wrap

		return $c;
	}

	/**
	 * Render title of the table
	 * return string
	 */
	private function book_table_title( $title, $time, $upper_limit, $num_limit, $controller ) {
		if ( '0' === (string)$title ) {
			$title_html = '';
		} else {
			# We dont know END value yet, because it may be variable. So put a placeholder there.
			$end_replace		= $num_limit
								? '%SON_PLaCeHOLDER%'
								: date_i18n( $this->date_format, $this->client_time($time) + $upper_limit*DAY_IN_SECONDS-60 );
			$start_end_replace 	= $num_limit
								? '%STND_PLaCeHOLDER%'
								: $this->format_start_end( $this->client_time($time), $this->client_time($time) + $upper_limit*DAY_IN_SECONDS-60 );
			$title 				= str_replace(
									array( "START_END", "START", "END" ),
									array( $start_end_replace,
											date_i18n( $this->date_format, $this->client_time($time) ),
											$end_replace,
									), $title
			);

			$title_html = wpb_title_html( $title, $controller );
		}

		return $title_html;
	}

	/**
	 * Find name of the column
	 * return string
	 */
	private function table_col_name( $col, $only_buttons ) {
		if ( 'button' === $col ) {
			return $only_buttons ? '' : $this->get_text('action');
		} else if ( 'day' === $col ) {
			return $this->get_text('day_of_week');
		} else {
			return $this->get_text($col);
		}
	}

	/**
	 * Render available/busy seat numbers
	 * return string
	 */
	private function table_cell_seats( $col, $slot ) {
		$worker			= $slot->get_worker();
		$slot_start		= $slot->get_start();
		$slot_end		= $slot->get_end();
		$avail_workers  = isset( $slot->stat[ $slot_start ][ $slot_end ]['available'] )
						 ? $slot->stat[ $slot_start ][ $slot_end ]['available']
						 : $slot->available_workforce();

		switch( $col ) {
			case 'seats_total':
				return $worker ? 1 : $avail_workers;

			case 'seats_left':
				if ( $worker ) {
					$avail 	= 1;
					$busy	= 0;
				} else {
					$avail = $avail_workers;
					$busy = ! empty( $slot->stat[ $slot_start ][ $slot_end ]['count'] )
							? $slot->stat[ $slot_start ][ $slot_end ]['count']
							: $slot->nof_booked_seats();
				}

				return max( 0, $avail - $busy );

			case 'seats_total_left':
				$total = $worker ? 1 : $avail_workers;
				if ( $worker ) {
					$left = 1;
				} else {
					$busy = ! empty( $slot->stat[ $slot_start ][ $slot_end ]['count'] )
							? $slot->stat[ $slot_start ][ $slot_end ]['count']
							: $slot->nof_booked_seats();
					$left = max( 0, $avail_workers - $busy );
				}

				return ( $total . ' / ' . $left );
		}
	}

	/**
	 * Render button code
	 * return string
	 */
	private function table_cell_button( $slot, $book_now, $only_buttons, $controller ){
		if ( $book_now ) {
			$button_text = str_replace(
				array( "DATE", "DAY", "START", ),
				array( $slot->client_date(), $slot->client_day(), $slot->client_time() ),
				$this->calendar_title_replace( $book_now, $controller )
			);
		} else if ( $only_buttons ) {
			$button_text = $slot->client_dt();
		} else {
			$button_text = $this->get_text('book_now_short');
		}

		$has_var_cl 	=  $button_text != $this->get_text('book_now_short') ? ' app-has-var' : '';
		$disabled_cl 	= ! is_user_logged_in() && 'yes' === wpb_setting("login_required") ? ' app-disabled-button' : '';
		$maybe_float 	= $only_buttons ? 'app_left app_only_buttons app-mrmb' : '';

		$c  = '<div title="'.WpBDebug::time_slot_tt($slot).'" class="app-book-now '.$maybe_float.'">';
		$c .= '<button class="app-book-now-button ui-button ui-btn ui-btn-icon-left ui-icon-shop'.$disabled_cl.$has_var_cl.'" data-start="'.date( $this->dt_format, $slot->get_start() ).'" data-end="'.date( $this->dt_format, $slot->get_end() ).'">'. $button_text .
			  '<input type="hidden" class="app-packed" value="'.$slot->pack( ).'" />'.'</button>';
		$c .= '</div>';

		return $c;
	}

	/**
	 * Calculate start time of Book Table
	 * return array
	 */
	private function book_table_start_ts( $start, $range, $add, $net_days, $controller ) {
		$error 	= '';
		$time	= strtotime( date( 'Y-m-d', $this->calendar_start_ts( $start ) ) ) + (int)$add * 3600 *24;

		# Previous timestamps. If this is set, list is being updated after prev/next buttons and override the above $time value
		# We have to calculate start values from previous values, because start of this table depends on previous end value
		$ts			= isset( $_POST['app_last_timestamps'] ) ? json_decode( wp_unslash( $_POST['app_last_timestamps'] ) ) : false;
		$is_daily	= $controller->is_daily();

		# We are only interested in 0th element
		if ( isset( $ts[0] ) && is_object( $ts[0] ) ) {
			if ( ! empty( $_POST['prev_clicked'] ) ) {
				$time = $ts[0]->start;
			} else {
				$time = $is_daily ? $ts[0]->end + DAY_IN_SECONDS : $ts[0]->end;
			}
		}

		# Range
		$compans 	= false;	# Can be set only for day selection
		$num_limit 	= 0; 		# Number of lines to be displayed. If not set, time setting will be used
		$_range 	= trim( str_replace( array( 'months','month','weeks','week','days','day', ), '', $range ) );

		if ( strpos( $range, 'month' ) !== false ) {
			$_range 		= is_numeric( $_range) ? $_range : 1;
			$upper_limit 	= intval( ( wpb_last_of_month( $time,($_range - 1) ) - $time )/DAY_IN_SECONDS );
		} else if ( strpos( $range, 'week' ) !== false ) {
			$_range 		= is_numeric( $_range) ? $_range : 1;
			$upper_limit 	= 7*$_range + 1 - intval( ( $time - wpb_sunday( $time ) )/DAY_IN_SECONDS );
		} else if ( strpos( $range, 'day' ) !== false ) {
			$_range 		= is_numeric( $_range) ? $_range : 1;
			$upper_limit 	= $_range;
			$compans 		= $net_days ? true : false;
		} else if ( $range && is_numeric( $range ) && $range > 0 ) {
			$num_limit		= $range;
			$upper_limit	= $this->get_app_limit();
			# Start scanning from current time or earliest allowed time
			$time = max( $time, strtotime( date("d F Y", intval( $this->get_lower_limit()*3600 + $this->_time -3600 ) ), $this->_time ) );
		} else {
			$error = WpBDebug::debug_text( __('Check "range" attribute in book shortcode','wp-base') ) ;
		}

		return array( $time, $num_limit, $upper_limit, $compans, $error );
	}

	/**
	 * Shortcode function to generate monthly calendar
	 */
	public function calendar_monthly( $atts ) {

		$atts = shortcode_atts( array(
			'title'				=> $this->get_text('monthly_title'),
			'logged'			=> $this->get_text('logged_message'),
			'notlogged'			=> $this->get_text('not_logged_message'),
			'location'			=> 0,
			'service'			=> 0,
			'worker'			=> 0,
			'long'				=> 0,
			'class'				=> '',
			'add'				=> 0,
			'start'				=> 0, 				// Previously "date". "auto" starts from first available date
			'display'			=> 'minimum',		// Since 3.0. full, with_break, minimum
			'_widget'			=> 0,				// Use as widget. Makes quick check
			'_no_timetable'		=> 0, 				// since 2.0. Disable timetable, e.g. for admin side
			'_force_min_time'	=> 0,				// since 2.0. Force min time to be used, so that it can catch bookings (on admin side)
			'_admin'			=> 0,
			'_cont'				=> '',				// since 3.0. WpB_Controller object
			'design'			=> '',
			'_caption'			=> '',
			'caption'			=> 'START',
		), $atts, 'app_monthly_schedule' );

		extract( $atts );

		$controller = $_cont instanceof WpB_Controller ? $_cont : $this->init_controller( $location, $service, $worker );
		$ret  = '';
		$ret .= $controller->display_errors();

		$cl_offset		= $this->get_client_offset( );
		$month_start 	= wpb_first_of_month( $this->calendar_start_ts( $start ) + $cl_offset, $add  );

		if ( 'compact' == $design || '0' === (string)$title ) {
			$title_html = '';
		} else {
			$title 		= str_replace( array( "START_END", "START" ), date_i18n( "F Y",  $month_start ), $title );
			$title_html = '<div class="app-title">' . esc_html( $this->calendar_title_replace( $title, $controller ) ) . '</div>';
		}

		if ( wpb_is_admin() ) {
			$more = $class.' app-wrap-admin';
		} else if ( function_exists( 'buddypress' ) ) {
			$more = 'app-bp '.$class;
		} else {
			$more = $class;
		}

		if ( $this->is_daily( $controller->get_service() ) ) {
			$more .= ' app-monthly-daily';
		}

		global $is_gecko;
		if ( $is_gecko ) {
			$more .= ' app-firefox';
		}

		$ret .= '<div class="app-sc app-wrap'.($_widget ? '-widget' : '').' '.$more.'">';
		$ret .= $title_html;
		$ret .= $this->calendar_subtitle( $logged, $notlogged );
		$ret .= '<div class="app-list'.( 'compact' == $design ? ' has-caption' : '' ).'">';
		$ret .= '<div class="app-monthly-wrapper '.(wpb_is_admin() ? 'app-monthly-admin ' : '').(wpb_is_account_page() ? 'app-monthly-account ' : '').'" data-display_mode="'.esc_attr($display).'" data-force_min_time="'.esc_attr($_force_min_time).'">';
		$ret  = apply_filters( 'app_monthly_schedule_before_table', $ret );
		$ret .= "<table>";

		if ( 'compact' == $design ) {
			$caption = $_caption ?: $caption;
			$class	 = 'app-caption-'. strtolower( sanitize_html_class( $caption ) );
			$caption = str_replace( array( "START_END", "START" ), date_i18n( "F Y",  $month_start ), $caption );
			$caption = $this->calendar_title_replace( $caption, $controller );
			$ret .= '<caption class="'.$class.'"><div class="app-flex">'. $this->prev( $month_start ).
					'<div class="month-name">'. $caption. '</div>'.
					$this->next( $month_start ).'</div></caption>';
		}

		$ret .= $this->_get_table_meta_row_monthly( 'thead', $long, $month_start );
		$ret .= '<tbody>';
		$ret .= '<tr>';

		$first_found	= false;
		$month_end		= wpb_last_of_month( $month_start );
		$first 			= $month_start - DAY_IN_SECONDS * wpb_mod( date( 'w', $month_start ) - $this->start_of_week, 7 );
		$last 			= $month_end + DAY_IN_SECONDS * wpb_mod( $this->start_of_week - date( 'w', $month_end ), 7 ) - DAY_IN_SECONDS;
		$time_table		= '';
		$is_daily		= $controller->is_daily();
		$setup_args		= array( 'disable_lazy_load' => $_widget, 'admin' => $_admin, 'force_min_time' => intval($_force_min_time) );
		$calendar 		= new WpB_Calendar( $controller );
		$calendar->setup( $setup_args );

		for ( $d = $first; $d <= $last; $d = $d + DAY_IN_SECONDS ) {
			$date 			= date( 'Y-m-d', $d );
			$day_start		= strtotime( "{$date} 00:00" );
			$day_end		= strtotime( "{$date} 23:59:59" );
			$dow			= (int)date('w', strtotime($date) );

			if ( $this->start_of_week == $dow )
				$ret .= '</tr><tr>';

			$classes = $code = array();
			$calendar->has_appointment = false;
			$slot = new WpB_Slot( $calendar, $day_start, $day_end, false, false, $cl_offset );

			if ( $day_start < strtotime( 'today', $this->_time + $cl_offset ) ) {
				$code[] = 13; # Yesterday or before
			} else if ( $day_end > $this->_time + $cl_offset + ( $this->get_app_limit() + 1 )*DAY_IN_SECONDS ) {
				$code[] = 3; # no_time
			} else if ( !$cl_offset && $reason = $slot->why_not_free( 'quick' ) ) {
				$code[] = $reason;
			} else {
				$setup_args = array_merge( $setup_args, array(
					'display' 		=> $_widget || $_admin ? 'minimum' : $display,
					'assign_worker'	=> !$controller->get_worker() && $controller->workers,
				) );
				$calendar->setup( $setup_args );
				$out = $calendar->find_slots_in_day( $day_start, 1 );

				if ( $calendar->all_busy ) {
					if ( $calendar->has_waiting ) {
						$code[] = 20; # All busy and waiting, but no free
					} else {
						$code[] = 1; # All slots busy
					}
				} else if ( !$calendar->nof_free ) {
					if ( $calendar->has_waiting ) {
						$code[] = 20; # All unavailable and waiting, but no free
					} else if ( $calendar->get_worker() ) {
						$code[] = 17; # Possibly worker is not working
					} else if ( $calendar->reason ) {
						$code[] = $calendar->reason;
					} else {
						$code[] = 19; # We do not know the reason
					}
				}
			}

			if ( empty( $code ) ) {
				$click_hint_text = 	! empty( $_no_timetable )
									? $this->get_text('click_to_book')
									: $this->get_text('click_to_select_date');
			} else {
				$click_hint_text = WpBDebug::display_reason( $code );
			}

			if ( $calendar->has_appointment ) {
				$classes[] = 'has_appointment';
			}

			if ( $this->_time + $cl_offset >= $day_start && $this->_time + $cl_offset < $day_end ) {
				$classes[] = 'today';
			}

			if ( $is_daily ) {
				$classes[] = 'daily';
			}

			if ( empty( $code ) ) {
				$classes[] = 'free';

				if ( empty( $first_found ) && $d >= $month_start ) {
					$classes[] = 'first-free';
					$first_found = true;
				}
			} else {
				$classes[] = 'notpossible';
				$classes[] = wpb_code2reason( max( $code ) );
			}

			if ( $d < $month_start || $d >= $month_end ) {
				$classes[] = 'app-other-month';
			}

			$new_link = '';

			if ( $_no_timetable && in_array( 'free', $classes ) ) {
				$url = false;

				if ( current_user_can(WPB_ADMIN_CAP ) ) {
					$url = admin_url('admin.php?page=app_bookings&add_new=1&app_id=0&app_worker='.$slot->get_worker().'&app_timestamp='.$day_start);
				}

				if ( $url = apply_filters( 'app_new_app_url', $url, $slot ) ) {
					$new_link ='<input type="hidden" class="app-new-link" value="'.esc_attr( $url ).'" />';
				}

				if ( ! $new_link ) {
					$classes[] = 'no-new-link';
					$click_hint_text = '';
				}
			}

			$classes = apply_filters( 'app_monthly_calendar_day_class', $classes, $slot );

			$class_name = implode( ' ', $classes );

			$click_hint_text = apply_filters( 'app_tooltip_text', $click_hint_text, $calendar, 'monthly_calendar' );

			$ret .= '<td class="'.$class_name.' app_day app_day_'.$day_start.' app_worker_'.$calendar->get_worker().'" '.
				'data-title="'.date_i18n( $this->date_format, $day_start ).'" title="'.esc_attr( $click_hint_text ).'"><p>'.date( 'j', $d ).'</p>'.
				'<input type="hidden" class="app-select-ts" value="'.$day_start .'" />'.
				'<input type="hidden" class="app-packed" value="'.$calendar->slot( $day_start )->pack().'" />';

			$ret = apply_filters( 'app_monthly_calendar_html_after_td', $ret, $slot, $calendar );

			$ret .= $new_link;

			$ret .= '</td>';
		}

		$calendar->save_cache();

		// Markup cleanup
		$ret = str_replace( '<tr></tr>', '', $ret );

		$ret .= '</tr>';
		$ret = apply_filters( 'app_monthly_schedule_after_last_row', $ret );
		$ret .= '</tbody>';
		$ret .= '</table>';
		$ret  = apply_filters( 'app_monthly_schedule_after_table', $ret );
		$ret .= '</div>';

		if ( ! $_widget && ! $_no_timetable ) {
			$ret .= '<div class="app-timetable-wrapper '.($is_daily ? 'daily' : '').'">';
			$ret .= $time_table;
			$ret .= '</div>';
		}

		$ret .= '<div style="clear:both"></div>';
		$ret .= '</div>'; # app-list
		$ret .= '</div>'; # app-wrap

		return $ret;
	}

	/**
	 * Shortcode function to generate weekly calendar
	 */
	public function calendar_weekly( $atts ) {

		$atts = shortcode_atts( array(
			'title'				=> $this->get_text('weekly_title'),
			'logged'			=> $this->get_text('logged_message'),
			'notlogged'			=> $this->get_text('not_logged_message'),
			'location'			=> 0,
			'service'			=> 0,
			'worker'			=> 0,				// Worker Id or 'all'
			'long'				=> 0,
			'class'				=> '',
			'add'				=> 0,
			'start'				=> 0,				// Previously "date"
			'display'			=> 'with_break',	// Since 3.0. full, with_break, minimum
			'_force_min_time'	=> 0,				// Force min time (in minutes) to be used, so that it can catch bookings (on admin side)
			'_inline'			=> 0,				// Displays bookings inline instead of tooltip
			'_daily'			=> 0,				// Can be used as daily too
			'_admin'			=> 0,				// For admin side usage
			'_cont'				=> '',				// since 3.0. WpB_Controller object
			'design'			=> '',
			'_caption'			=> '',
			'caption'			=> 'START_END',
			'range'				=> '',
		), $atts, 'app_schedule' );

		extract( $atts );

		$controller = $_cont instanceof WpB_Controller ? $_cont : $this->init_controller( $location, $service, $worker );
		$ret  = '';
		$ret .= $controller->display_errors();

		$cl_offset 	= $this->get_client_offset( );
		$time 		= $this->calendar_start_ts( $start ) + $cl_offset + ($add * WEEK_IN_SECONDS);
		if ( $_daily ) {
			$cal_start = $time;
		} else {
			if ( 6 == $this->start_of_week ) {
				$cal_start = wpb_saturday( $time );
			} else if ( 1 == $this->start_of_week ) {
				$cal_start = wpb_monday( $time );
			} else {
				$cal_start = wpb_sunday( $time ) + $this->start_of_week * DAY_IN_SECONDS;
			}
		}

		if ( 'compact' == $design || '0' === (string)$title ) {
			$title_html = '';
		} else {
			$title = str_replace(
				array( 'START_END', 'START', 'END', ),
				array( $this->format_start_end( $cal_start, $cal_start + 6*DAY_IN_SECONDS ),
					date_i18n($this->date_format, $cal_start ),
					date_i18n($this->date_format, $cal_start + 6*DAY_IN_SECONDS ), ),
				$title
			);
			$title_html = wpb_title_html( $title, $controller );
		}

		$classes = explode( ' ', $class );

		if ( wpb_is_admin() ) {
			$classes[] = 'app-wrap-admin';
		} else if ( function_exists( 'buddypress' ) ) {
			$classes[] = 'app-bp';
		}

		global $is_gecko;
		if ( $is_gecko ) {
			$classes[] = 'app-firefox';
		}

		$ret .= '<div class="app-sc app-wrap '. implode( ' ', $classes ) .'">';
		$ret .= $title_html;
		$ret .= $this->calendar_subtitle( $logged, $notlogged );
        $ret .= '<div class="app-list">';
		$ret .= '<div class="app-schedule-wrapper'.( wpb_is_admin() ? ' app-weekly-admin' : '' ).( wpb_is_account_page() ? ' app-weekly-account' : '' ).'">';
		$ret .= $this->calendar_weekly_h( $cal_start, compact( array_keys( $atts ) ), $controller );
		$ret .= '</div>';
		$ret .= '</div>'; # app-list
		$ret .= '</div>'; # app-wrap

		return $ret;
	}

	/**
	 * Helper for weekly calendar
	 */
	public function calendar_weekly_h( $time, $args, $controller ) {

		$args['_widget'] = false;
		extract ( $args );

		$tbl_class  = $class ? "class='{$class}'" : '';
		$cl_offset 	= $this->get_client_offset( $time );
		$date		= $time ? $time : $this->_time + $cl_offset;
		$week_start = 6 == $this->start_of_week
					  ? wpb_saturday( $date ) - 6*DAY_IN_SECONDS
					  : wpb_sunday( $date ); # Care for calendars starting on Saturday

		$ret  = '';
		$ret .= apply_filters( 'app_schedule_before_table', $ret );
		$ret .= "<table {$tbl_class}>";

		if ( 'compact' == $design ) {
			$caption = $_caption ?: $caption;
			$class	 = 'app-caption-'. strtolower( sanitize_html_class( $caption ) );
			$caption = str_replace(
				array( 'START_END', 'START', 'END', ),
				array( $this->format_start_end( $week_start, $week_start + 6*DAY_IN_SECONDS ),
					date_i18n( $this->date_format, $week_start ),
					date_i18n( $this->date_format, $week_start + 6*DAY_IN_SECONDS ), ),
				$caption
			);

			$caption = wpb_title_html( $caption, $controller );

			list( $range, $pag_step, $pag_unit, $error ) = $this->book_range( $range, 'weekly' );

			$ret .= '<caption class="'.$class.'"><div class="app-flex">'. $this->prev( $week_start, $pag_step, 'week' ).
				'<div class="week-range">'. $caption. '</div>'. $this->next( $week_start, $pag_step, 'week' ).'</div></caption>';
		}

		if ( ! $_daily ) {
			$ret .= $this->_get_table_meta_row( 'thead', $long, $week_start );
		}

		$ret .= '<tbody>';

		$ret = apply_filters( 'app_schedule_before_first_row', $ret );

		$days_to_scan = $_daily ? (array)date( "w", $date ) : wpb_arrange( range( 0, 6 ), false, true );

		$calendar = new WpB_Calendar( $controller );
		$calendar->setup( array(
			'admin'			=> $_admin,
			'inline'		=> $_inline,
			'force_min_time'=> $_force_min_time,
			'display'		=> $display,
			'assign_worker'	=> ! $controller->get_worker() && $controller->workers,
		));

		$cell_vals = array();

		foreach ( $days_to_scan as $key => $i ) {
			# Get html for each day (one column)
			$one_col 	= $calendar->find_slots_in_day( $week_start + $i*DAY_IN_SECONDS + $cl_offset );
			$cell_vals 	= $cell_vals + $one_col;
		}

		$no_results = true;
		$min_time	= $this->get_min_time()*60;
		$start		= -1* $cl_offset;
		$days		= $_daily
					  ? array( -1, date( "w", $date ) )
					  : wpb_arrange( range( 0, 6 ), -1, true ); # Arrange days acc. to start of week

		for ( $tval = $start; $tval < $start + DAY_IN_SECONDS; $tval = $tval + $min_time ) {
			$disp_row = false;
			$ret_temp = '';

			foreach ( $days as $key => $i ) {

				if ( $i == -1 ) {
					# Admin side uses fixed hours:mins column
					$hours_mins = $_admin
								  ? date_i18n( $this->time_format, $tval )
								  : ( $controller->is_daily() ? '' : date_i18n( $this->time_format, $tval + $cl_offset ) );
					$f_time_val = apply_filters( 'app_weekly_calendar_from', $hours_mins, $tval );
					$ret_temp  .= "<td class='app-weekly-hours-mins ui-state-default'>".$f_time_val."</td>";
				} else {
					$slot_start = $tval + $week_start + $i*DAY_IN_SECONDS;

					if ( isset( $cell_vals[ $slot_start ] ) ) {
						$slot 	   = $cell_vals[ $slot_start ];
						$ret_temp .= $slot->weekly_calendar_cell_html( );
						$disp_row  = $this->display_row( $slot, $calendar->get_display() );
					} else {
						# We are creating empty cells that are not generated by calendar_timetable to complete the table
						$ret_temp .= '<td class="notpossible app_dummy app_slot_'.$slot_start.'" '.
							'data-title="'.date_i18n( $this->dt_format, $this->client_time($slot_start) ).'" title="'.esc_attr( WpBDebug::display_reason( 16 ) ).'"></td>';
					}
				}
			}

			if ( apply_filters( 'app_display_row', $disp_row, $calendar ) ) { # Custom rule per calendar
				$no_results = false;
				$ret .= '<tr>';
				$ret .= $ret_temp;
				$ret .= '</tr>';
			}
		}

		$calendar->save_cache();

		if ( $no_results ) {
			$ret .= '<tr><td class="app_center" colspan="8">' . $this->get_text( 'no_free_time_slots' ) .'</td></tr>';
		}

		$ret .= '</tbody>';
		$ret .= '</table>';

		$ret  = apply_filters( 'app_weekly_schedule_after_table', $ret );

		return $ret;
	}

	/**
	* Determine if row in calendar table will be displayed, based on current slot
	* @return bool
	*/
	private function display_row( $slot, $display ) {
		$disp_row = false;

		if ( !$slot->reason ) { # Always show free
			$disp_row = true;
		} else if ( 20 == $slot->reason ) {
			$disp_row = true;
		} else if ( ('with_break' === $display || 'full' === $display) && 1 === $slot->reason ) { # Almost always show busy
			$disp_row = true;
		} else if ( 'with_break' === $display && in_array( $slot->reason, array( 2,4,5,10,20 ) ) ) { # Show holiday and breaks if selected
			$disp_row = true;
		}

		return apply_filters( 'app_display_row_slot', $disp_row, $slot ); # Custom rule per slot
	}

	/**
	* Get weekly calendar table header
	* @param $long		bool|null	If true, long name. If false, short name. If null, initial letter
	* @return string
	*/
	public function _get_table_meta_row( $which, $long, $week_start = '', $range = 0 ) {
		$cells = '<th class="hourmin_column">&nbsp;</th>';

		foreach( wpb_arrange( range(0,6), false ) as $day ) {
			$dname_long = strtolower( date( 'l', strtotime( "Sunday +{$day} days") ) );
			$dname = $long
					 ? $this->get_text( $dname_long ) 
					 : ( $long === null ? $this->get_text( $dname_long.'_initial' ) : $this->get_text( $dname_long.'_short' ) );
			$cells .= '<th class="app_'.$dname_long.'"><span class="normal">' . $dname . '</span>';
			$cells .= '<span class="initial">' . $this->get_text( $dname_long.'_initial' ) . '</span></th>';
		}

		return apply_filters( 'app_weekly_table_meta_row', "<{$which}><tr class='ui-state-default'>{$cells}</tr></{$which}>", $which, $long, $week_start, $range );
	}

	/**
	* Get monthly calendar table header
	* @return string
	*/
	public function _get_table_meta_row_monthly ( $which, $long, $month_start = '' ) {
		$cells = '';
		$today_no = date( 'w', $this->client_time( $this->_time ) );

		foreach( wpb_arrange( range(0,6), false ) as $day ) {
			$today = $today_no == $day ? ' today' : '';
			$dname_long = strtolower( date('l', strtotime("Sunday +{$day} days")) );
			$dname = $long ?  $this->get_text( $dname_long ) : $this->get_text( $dname_long.'_short' );
			$cells .= '<th class="app_'.$dname_long.$today.'"><span class="normal">' . $dname . '</span>';
			$cells .= '<span class="initial">' . $this->get_text( $dname_long.'_initial' ) . '</span></th>';
		}

		return apply_filters( 'app_monthly_table_meta_row', "<{$which}><tr class='ui-state-default'>{$cells}</tr></{$which}>", $which, $long, $month_start );
	}

	/**
	 * Create html for previous link for compact design
	 * @since 3.7.6
	 * @return string
	 */
	public function prev( $start, $step = 1, $unit = 'month' ){
		$time = $this->client_time( $this->calendar_start_ts( $start ) );
		list( $prev, $prev_min, $prev_text, $next, $next_max, $next_text ) = wpb_prev_next( $time, $step, $unit );

		$c  = '';
		$c .= '<div class="app-previous '.$prev.'"'.($prev <= $prev_min ? " style='visibility:hidden'" : "").'>';
		$c .= '<a class="app-browse" data-unit="'.$unit.'" data-step="'.$step.'" data-tstamp="'.$prev.'" href="javascript:void(0)"><em class="app-icon icon-left-open"></em></a>';
		$c .= '</div>';

		return $c;
	}

	/**
	 * Create html for next link for compact design
	 * @since 3.7.6
	 * @return string
	 */
	public function next( $start, $step = 1, $unit = 'month' ){
		$time = $this->client_time( $this->calendar_start_ts( $start ) );
		list( $prev, $prev_min, $prev_text, $next, $next_max, $next_text ) = wpb_prev_next( $time, $step, $unit );

		$c  = '';
		$c .= '<div class="app-next '.$next.'"'.($next >= $next_max ? " style='visibility:hidden'" : "").'>';
		$c .= '<a class="app-browse" data-unit="'.$unit.'" data-step="'.$step.'" data-tstamp="'.$next.'" href="javascript:void(0)"><em class="app-icon icon-right-open"></em></a>';
		$c .= '</div>';

		return $c;
	}

	/**
	 * Format a field of confirmation form
	 * @since 3.0
	 * @return string
	 */
	public function conf_line_html( $field_name, $value = '' ) {
		$value_html = $value ? '<span class="app-conf-text">'. $value . '</span>' : '';
		return  wpb_is_hidden( $field_name )
				? ''
				: '<label><span class="app-conf-title">'.$this->get_text($field_name). '</span>'.$value_html.'</label>';
	}

	/**
	 * Default confirmation shortcode atts
	 * @since 3.0
	 * @return array
	 */
	public function confirmation_atts() {
		return array(
			'confirmation_title' 	=> $this->get_text('confirmation_title'),
			'button_text'			=> $this->get_text('checkout_button'),
			'name'					=> $this->get_text('name'),
			'first_name'			=> $this->get_text('first_name'),
			'last_name'				=> $this->get_text('last_name'),
			'email'					=> $this->get_text('email'),
			'phone'					=> $this->get_text('phone'),
			'address'				=> $this->get_text('address'),
			'city'					=> $this->get_text('city'),
			'zip'					=> $this->get_text('zip'),
			'state'					=> $this->get_text('state'),
			'country'				=> $this->get_text('country'),
			'note'					=> $this->get_text('note'),
			'remember'				=> $this->get_text('remember'),
			'use_cart'				=> 'auto',
			'class'					=> '',		// Since 3.6.2
			'fields'				=> '',		// Since 2.0. Default user fields can be sorted
			'layout'				=> 600,		// 1 (column), 2(columns) or a number equal or greater than 600. Since 3.0.
			'countdown'				=> 'auto',	// since 3.0 use countdown to refresh the page or not. 'auto' determines if a refresh is better
			'countdown_hidden'		=> 0,		// whether countdown will be hidden
			'countdown_title'		=> $this->get_text('conf_countdown_title'), // Use 0 to disable
			'continue_btn'			=> 1,
			'_app_id'				=> 0,
			'_editing'				=> '',		// If equals 2, this is confirmation of Paypal Express payment form only
			'_edit_cap'				=> '',
			'_final_note_pre'		=> '',
			'_final_note'			=> '',
			'_hide_cancel'			=> 0,		// Hides cancel overriding other settings
		);
	}

	/**
	 * Shortcode function to generate a confirmation box
	 */
	public function confirmation( $atts = array() ) {

		extract( shortcode_atts( $this->confirmation_atts(), $atts, 'app_confirmation' ) );

		$post 			= get_post();
		$post_id 		= isset( $post->ID ) ? $post->ID : 0;
		$post_content	= isset( $post->post_content ) ? $post->post_content : '';

		# Confirmation form can only be added to the page once
		# However, some page builders scan the content, but they do not reflect it; they serve thru post_meta
		if ( ! empty( $this->conf_form_added ) && strpos( $post_content, '<div class="app-sc app-conf-wrapper' ) !== false ) {
			return WpBDebug::debug_text( __('On a page only one instance of Confirmation Form is allowed; this confirmation form has been skipped.', 'wp-base' ) );
		}

		$ecom_active = 	wpb_is_product_page( $post_id ) ||
						BASE('EDD') && BASE('EDD')->is_app_edd_page( $post_id ) ||
						BASE('WooCommerce') && BASE('WooCommerce')->is_app_wc_page( $post_id );

		$use_cart = ! $ecom_active && BASE('ShoppingCart') && 'no' !== $use_cart
					&& (('auto' === $use_cart && 'yes' === wpb_setting( 'use_cart' )) || ('auto' != $use_cart && $use_cart));

		# If editing, bring data of the owner of the app
		if ( $_editing && $_app_id && $app = wpb_get_app( $_app_id ) ) {
			$service = $this->get_service( $app->service );
		}

		/* Start preparing HTML */
		$mobile_cl  = wpb_is_mobile() ? ' app-mobile' : ' app-non-mobile';
		$style 		= $_editing || BASE('Multiple')->get_cart_items() ? '' : "style='display:none'";

		/* General Notes */
		$ret  = apply_filters( 'app_edit_general_notes', '', $_app_id, $_editing );

		/* Title */
		$ret .= '<fieldset>';
		$ret  = apply_filters( 'app_confirmation_after_fieldset', $ret, $_app_id, $_editing );

		/* Countdown & Continue Shopping */
		$ret .= $this->countdown( $countdown, $countdown_title, $countdown_hidden, $continue_btn, $ecom_active, $use_cart, $_editing );

		/* Grouping of form. Default is column = 1 */
		$gr_class = $data = '';
		if ( ! $_editing && 2 == $layout ) {
			$gr_class = ' app_2column';
		} else if ( ! $_editing && is_numeric( $layout ) && $layout > 599 ) {
			$gr_class = ' app-conf-fields-gr-auto';
			$data = 'data-edge_width="'.$layout.'"';
		}

		$ret .= '<div '. $data .' class="app-conf-fields-gr app-conf-fields-gr1'. $gr_class  .'">';
		$ret  = apply_filters( 'app_confirmation_before_service', $ret, $_app_id, $_editing );

		/* Service */
		$ret .= '<div class="app-conf-service">';
		if ( 1 == $_editing )
			$ret = apply_filters( 'app_edit_service', $ret, $_app_id );
		else if ( 2 == $_editing ) {
			$service_id = isset( $service->ID ) ? $service->ID : 0;
			$ret .= '<label><span class="app-conf-title">'. $this->get_text('service_name') .  '</span>'. $this->get_service_name( $service_id ) . '</label>';
		}
		$ret .= '</div>';

		/* Worker */
		$ret .= '<div class="app-conf-worker" '. $style .'>';
		if ( 1 == $_editing )
			$ret  = apply_filters( 'app_edit_worker', $ret, $_app_id );
		else if ( 2 == $_editing ) {
			$worker_id = isset( $app->worker ) ? $app->worker : 0;
			if ( $worker_id ) {
				$ret .= '<label><span class="app-conf-title">'. $this->get_text('provider_name') .'</span>'. $this->get_worker_name( $worker_id ) . '</label>';
			}
		}
		$ret .= '</div>';

		$ret  = apply_filters( 'app_confirmation_after_worker', $ret, $_app_id, $_editing );

		/* Start */
		$ret .= '<div class="app-conf-start">';
		if ( 1 == $_editing ) {
			$ret  = apply_filters( 'app_edit_start', $ret, $_app_id, $_editing );
		} else if ( 2 == $_editing ) {
			$_start = isset( $app->start ) ? $app->start : 0;
			$ret .= '<label><span class="app-conf-title">'. $this->get_text('date_time') .  '</span>'. date_i18n($this->dt_format, strtotime( $_start ) ) . '</label>';
		}
		$ret .= '</div>';

		/* End */
		$ret .= '<div class="app-conf-end">';
		$ret .= '</div>';

		/* Duration/Lasts */
		$ret .= '<div class="app-conf-lasts" style="display:none">';
		$ret .= '</div>';

		/* Cart Contents */
		$ret .= '<div class="app-conf-details" style="display:none">';
		$ret .= '</div>';

		/* Price */
		$ret .= '<div class="app-conf-price app-clearfix" '. $style .'>';
		if ( 1 === $_editing )
			$ret  = apply_filters( 'app_edit_price', $ret, $_app_id, $_editing );
		else if ( 2 === $_editing ) {
			$_price = isset( $app->price ) ? $app->price : 0;
			$ret .= '<label><span class="app-conf-title">'. $this->get_text('price') .  '</span>'. wpb_format_currency( $_price ) . '</label>';
		}
		$ret .= '</div>';

		/* Deposit */
		$ret .= '<div class="app-conf-deposit" '.$style.'>';
		$ret .= '</div>';

		/* Paypal Amount */
		$ret .= '<div class="app-conf-amount" '.$style.'>';
		$ret .= '</div>';
		$ret  = apply_filters( 'app_confirmation_after_booking_fields', $ret, $_app_id, $_editing );

		if ( ! $_editing && wpb_setting('payment_method_position', 'after_booking_fields') === 'after_booking_fields' ) {
			$ret .= $this->payment_methods();
		}

		$ret .= '</div>'; // Closing of app-conf-fields-gr1
		$ret .= '<div class="app-conf-fields-gr app-conf-fields-gr2 app-clearfix '.$gr_class.'">';
		$ret  = apply_filters( 'app_confirmation_before_user_fields', $ret, $_app_id, $_editing );

		# Get values of all user variables like $user_name, $user_email, etc
		extract( BASE('User')->get_app_userdata( $_app_id, BASE('User')->read_user_id() ), EXTR_PREFIX_ALL, 'user' );

		# A non-functional form so that browser autofill can be used
		$ret .='<form class="app-conf-client-fields" onsubmit="return false;">';
		$ret .='<input type="text" autocomplete="off" name="confirm-email-username" class="app_confirm_email_username" id="app_confirm_email_username" />';

		/* Sanitize and filter User and UDF fields */
		$sorted_fields = wpb_sanitize_user_fields( $fields, $_app_id, $_editing );

		foreach ( $sorted_fields as $f ) {
			# Standard user fields
			if ( in_array( $f, $this->get_user_fields() ) ) {
				# For user defined field using filter hook, e.g, telefax
				if ( ! isset( ${$f} ) ) {
					${$f} = wpb_get_field_name( $f );
				}
				
				$is_readonly = apply_filters( 'app_confirmation_field_is_readonly', false, $f, $_app_id, $_editing );

				$ret .= '<div class="app-'.$f.'-field" '.$style.'>';
				$ret .= '<label><span class="app-conf-title">'. ${$f} . '<sup> *</sup></span>';
				$ret .= '<input type="text" placeholder="'.$this->get_text($f.'_placeholder').'" class="app-'.$f.'-field-entry '.$mobile_cl.'" value="'.esc_attr(${'user_'.$f}).'" '. ($is_readonly ? 'readonly' : '').' />';
				$ret .= '</label>';
				$ret .= '</div>';
				$ret  = apply_filters( 'app_confirmation_after_'.$f.'_field', $ret, $_app_id, $f, $_editing, $fields );
			} else {
				# Other fields, i.e. udf
				$ret .= apply_filters( 'app_confirmation_add_field', '', $_app_id, $f, $_editing, $fields );
			}
		}

		$ret  = apply_filters( 'app_confirmation_before_note_field', $ret, $_app_id, $_editing );
		$ret .= '<div class="app-note-field" '.$style.'>';
		$ret .= '<label><span class="app-conf-title">'. $note . '</span>';
		$ret .= '<textarea class="app-note-field-entry '.$mobile_cl.'">'.esc_textarea( $user_note ).'</textarea>';
		$ret .= '</label>';
		$ret .= '</div>';
		$ret .= '</form>';
		$ret  = apply_filters( 'app_confirmation_after_user_fields', $ret, $_app_id, $_editing );

		/* Remember me - Only for NON logged in users - Default is checked */
		$ret .= $this->remember( $remember, $user_remember, $_editing );

		/* Payment Method fields */
		if ( ! $_editing && wpb_setting('payment_method_position') === 'after_user_fields' ) {
			$ret .= $this->payment_methods( );
		}

		$ret .= '</div>'; # Closing of app-conf-fields-gr2
		$ret .= '<div style="clear:both"></div>';

		if ( ! $_editing && wpb_setting('payment_method_position') === 'after_user_fields_full' ) {
			$ret .= $this->payment_methods( true );
		}

		/* Instructions or forms */
		$ret .= '<div class="app_gateway_form" style="display:none;">';
		$ret .= '</div>';

		$ret  = apply_filters( 'app_confirmation_before_buttons', $ret, $_app_id, $_editing );
		$ret .= $_final_note_pre;

		/* Submit, Cancel Buttons */
		$button_text = apply_filters( 'app_confirmation_button_text', $button_text, $_app_id, $_editing );
		$ret .= '<div class="app-conf-buttons">';
		$ret .= '<input type="hidden" class="has-cart" value="'.($use_cart ? 1: 0) .'"/>';
		$ret .= '<input type="hidden" class="app-disp-price" value="" />';
		$ret .= "<input type='hidden' class='app-user-fields' value='".esc_attr( wp_json_encode( $sorted_fields ) )."' />";

		if ( $_app_id ) {
			$ret .= '<input type="hidden" class="app-edit-id" value="'.$_app_id.'"/>';
		}

		$ret .= '<input type="hidden" name="app_editing_value" class="app_editing_value" value="'. $_editing .'"/>';
		$ret .= '<button data-button_text="'.$button_text.'" data-icon="check" class="app-conf-button ui-button ui-btn ui-btn-icon-left">'.$button_text.'</button>';
		$ret .= $this->cancel_button( $_hide_cancel, $use_cart, $_editing );
		$ret .= '</div>';
		$ret .= '<div class="app-conf-final-note">';
		$ret .= $_final_note;
		$ret .= '</div>';
		$ret .= '</fieldset>';

		$this->conf_form_added = true;

		$classes = array( 'app-sc', 'app-conf-wrapper' );
		if ( $_editing ) {
			$classes[] = 'app-edit-wrapper';
		}
		if ( ! $_editing && 'above' === wpb_setting( 'conf_form_title_position' ) ) {
			$classes[] = 'above-input';
		}
		if ( $class ) {
			$classes[] = $class;
		}

		$out  = "<div data-user_fields='".json_encode( $sorted_fields )."' data-cap='".esc_attr($_edit_cap)."' class='".implode( ' ', $classes )."' ".$style.">";
		$out .= $ret;
		$out .= '</div>'; # Close app-sc app-conf-wrapper

		return $out;
	}

	/**
	 * Render Countdown and continue shopping
	 * @return string
	 */
	private function countdown( $countdown, $countdown_title, $countdown_hidden, $continue_btn, $ecom_active, $use_cart, $_editing ) {
		$use_button		= ! wpb_is_mobile() && $continue_btn && ! $ecom_active && $use_cart;
		$button_html 	= $use_button
						? '<button class="app-cont-btn ui-button ui-btn ui-corner-all ui-btn-icon-left">' . $this->get_text('continue_button') . '</button>'
						: '';

		$use_cdown			= $countdown == 1 || ('auto' == $countdown && ($ecom_active || $use_cart || BASE('Packages') || BASE('Recurring')) ) ? true : false;
		$disp_cdown			= $use_cdown && !$countdown_hidden;
		$cdown_style		= $disp_cdown ? '' : ' style="display:none" ';
		$cdown_title_style	= $use_button ? 'style="visibility:hidden"' : 'style="display:none"';
		$cdown_title_html	= $disp_cdown && "0" !== (string)$countdown_title
							  ? '<div class="app_countdown_dropdown_title app-title" '.$cdown_title_style.'>' . $countdown_title . '</div>'
							  : '';

		$two_column 		= $use_button && $disp_cdown ? 'app_2column_continue' : '';

		$ret = '';

		if ( ! $_editing && ( $use_button || $use_cdown ) ) {
			$ret .= '<div class="app-conf-continue'.($button_html ? ' has-button' : '').'">';
			if ( $use_cdown ) {
				$ret .= $button_html ? '<div class="'.$two_column.'">' . $button_html .'</div>' : '';
				$ret .= '<div ' . $cdown_style. ' class="app_countdown-wrapper '. $two_column .'">' .
						$cdown_title_html .
						'<div class="app-conf-countdown app-clearfix" data-height="72" data-size="70"></div>' .
						'</div>';
			} else {
				$ret .= '<label>';
				$ret .= '<span class="app-conf-title">'. '&nbsp;' .  '</span>';
				$ret .= $button_html;
				$ret .= '</label>';
			}
			$ret .= '</div>';
		}

		return $ret;
	}

	/**
	 * Render Remember Me field
	 * @return string
	 */
	private function remember( $remember, $user_remember, $_editing ) {
		$ret = '';

		if ( ! is_user_logged_in() && ! $_editing ) {
			$ret .= '<div class="app-remember-field" style="display:none">';
			$ret .= '<label><span class="app-conf-title">&nbsp;</span>';
			$ret .= '<span class="app-conf-text">';
			$ret .= '<input type="checkbox" checked="checked" class="app-remember-field-entry" '.$user_remember.' />&nbsp;';
			$ret .= $remember;
			$ret .= '</span>';
			$ret .= '</label></div>';
		}

		return $ret;
	}

	/**
	 * Render Cancel button
	 * @return string
	 */
	private function cancel_button( $_hide_cancel, $use_cart, $_editing ) {
		$hide	= false;
		$btn	= '';

		if ( $_hide_cancel ) {
			$hide = true;
		} else if ( ! $_editing && wpb_setting('conf_form_hide_cancel') === 'yes') {
			$hide = true;
		} else if ( ! $_editing && wpb_setting('conf_form_hide_cancel') === 'without_cart' && ! $use_cart ) {
			$hide = true;
		}

		if ( ! $hide ) {
			$btn = '<button data-icon="delete" data-iconpos="left" class="app-conf-cancel-button app-cancel-button ui-button ui-btn ui-btn-icon-left" >'.
					($use_cart && ! $_editing ? $this->get_text('cancel_cart') : $this->get_text('cancel')).'</button>';
		}

		return $btn;
	}

	/**
	 * Prepares HTML for payment methods
	 * @return string
	 */
	 private function payment_methods( $is_full_width = false ){
		# Allow limiting of gateways e.g. according to previous user preferences or user IP
		$active = apply_filters( 'app_confirmation_active_gateways', wpb_active_gateways() );

		# Click/Check radio if it is the only option
		$is_single_gateway =  $active && 1 == count( $active ) ? ' checked="checked"' : '';

		$ret = '';

		if ( wpb_is_mobile() ) {
			$ret .= '<fieldset data-iconpos="right" data-role="controlgroup" data-type="vertical" class="app-payment-field" style="display:none">';
			$ret .= '<legend>'.__('Payment method', 'wp-base' ) .'</legend>';
			foreach ( $active as $plugin ) {
				$public_name = apply_filters( 'app_gateway_public_name', $plugin->public_name, $plugin );

				$ret .= '<input id="app-radio-'.$plugin->plugin_name.'" type="radio" '.$is_single_gateway.' class="app_choose_gateway" name="app_choose_gateway" value="'.$plugin->plugin_name.'" />';
				$ret .= '<label for="app-radio-'.$plugin->plugin_name.'">'. esc_html( $public_name ) .'</label>';
			}
			$ret .= '</fieldset>';
		} else {
			$ret .= '<div class="app-payment-field app-clearfix '.($is_full_width ? 'full_width' : '').'" style="display:none">';
			$ret .= '<span class="app-payment-title">'. $this->get_text('pay_with') .'<sup> *</sup></span>';
			$ret .= '<div class="app-payment-inner app-clearfix">';

			foreach ( $active as $plugin ) {
				$public_name = apply_filters( 'app_gateway_public_name', $plugin->public_name, $plugin );
				$img		 = apply_filters( 'app_payment_method_image_url', $plugin->method_img_url, $plugin->plugin_name );
				$instr		 = apply_filters( 'app_gateway_instructions', wpb_gateway_setting( $plugin->plugin_name, 'instructions' ), $plugin );

				$ret .= '<div data-gateway_title="'.esc_attr($public_name).'" data-gateway="'.$plugin->plugin_name.'" class="app-payment-gateway-item app-'.$plugin->plugin_name.'">';
				$ret .= '<input type="radio" '.$is_single_gateway.' class="app_choose_gateway" name="app_choose_gateway" value="'.$plugin->plugin_name.'" />';
				$ret .= '<a href="javascript:void(0)">';
				if ( $img ) {
				  $ret .= '<img src="'. esc_attr( $img ) .'" alt="' . esc_attr( $public_name ) . '" />';
				}
				$ret .= '<span>'. esc_html( $public_name ) .'</span></a>';
				$ret .= '</div>';
				# Instructions: Contents of this will be read by qtip
				$ret .= '<div class="app-'.$plugin->plugin_name.'-instr app-gateway-instr" style="display:none">';
				$ret .= esc_html( $instr );
				$ret .= '</div>';
			}

			$ret .= '</div>';
			$ret .= '</div>';
		}

		return $ret;
	}

	/**
	 * Shortcode function to generate pagination links. Includes legend area
	 */
	public function pagination( $atts ) {

		extract( shortcode_atts( array(
			'step'				=> 1,
			'unit'				=> 'week',		// Number, day, week or month
			'start'				=> 0,			// Previously "date"
			'disable_legend'	=> 0,			// since 2.0
			'history'			=> 0,			// Enable visiting past days. Since 3.4
			'select_date'		=> wpb_is_mobile() ? 0 : 1,
			'_hide_next'		=> 0,
			'_hide_prev'		=> 0,
		), $atts, 'app_next' ) );

		if ( ! in_array( $unit, array( 'week','month','day','number' ) ) ) {
			return WpBDebug::debug_text( __('Check "Unit" parameter in Pagination shortcode','wp-base') ) ;
		}

		$c = '';

		// Legends
		if ( ! $disable_legend && 'yes' === wpb_setting('show_legend') ) {
			$items = $this->get_legend_items();

			$c .= '<div class="app-legend">';
			foreach ( $items as $class => $name ) {
				$c .= '<div class="app-legend-div">' .$name . '</div>';
			}
			$c .= '<div style="clear:both;"></div>';
			foreach ( $items as $class => $name ) {
				$c .= '<div class="app-legend-div app-legend-div-color '.$class.'">&nbsp;</div>';
			}
			$c .= '<div style="clear:both;"></div>';
			$c .= '</div>';
		}

		$time = $this->client_time( $this->calendar_start_ts( $start ) );
		list( $prev, $prev_min, $prev_text, $next, $next_max, $next_text ) = wpb_prev_next( $time, $step, $unit );

		$cl_sel_date = $select_date ? ' has-select-date' : '';

		if ( wpb_is_mobile() && $select_date ) {
			$next_text = $this->get_text( 'next' );
			$prev_text = $this->get_text( 'previous' );
		}

		// Pagination
		$c .= '<div class="app-pagination">';
		$c .= '<div class="app-previous '.$prev.$cl_sel_date.'"'.($_hide_prev || (!$history && $prev <= $prev_min) ? " style='visibility:hidden'" : "").'>';
		$c .= '<a class="app-browse" data-unit="'.$unit.'" data-step="'.$step.'" data-tstamp="'.$prev.'" href="javascript:void(0)"><em class="app-icon icon-'.( is_rtl() ? 'right' : 'left').'-open"></em><span>'. $prev_text . '</span></a>';
		$c .= '</div>';

		if ( $select_date )  {
			$c .= $this->select_date( array( 'title' => 0, 'history' => $history ) );
		}

		$c .= '<div class="app-next '.$next.$cl_sel_date.'"'.($_hide_next || $next >= $next_max ? " style='visibility:hidden'" : "").'>';
		$c .= '<a class="app-browse" data-unit="'.$unit.'" data-step="'.$step.'" data-tstamp="'.$next.'" href="javascript:void(0)"><span>'. $next_text . '</span><em class="app-icon icon-'.( is_rtl() ? 'left' : 'right').'-open"></em></a>';
		$c .= '</div>';

		$c .= '<div style="clear:both"></div>';
		$c .= '</div>';

		return $c;
	}

}
	BASE()->add_hooks_front();

	if ( is_admin() ) {
		include_once( WPBASE_PLUGIN_DIR . '/includes/admin/base-admin.php' );
	} else {
		$GLOBALS['appointments'] = BASE();	// For backwards compatibility
	}

} else {
	add_action( 'admin_notices', '_wpb_plugin_conflict_own' );
}

