<?php
/**
 * Addon Name: Listing
 * Description: Handles listing of client bookings on front end or user page
 * Class Name: WpBListing
 * Version: 3.0.2
 *
 * @package WP BASE
 * @author Hakan Ozevin
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpBListing' ) ) {

class WpBListing {

	private $colspan = 0;

	/**
     * WP BASE Core + Front [+Admin] instance
     */
	protected $a = null;

	/**
     * Constructor
     */
	public function __construct() {
		$this->a = BASE();
	}

	/**
     * Add hooks
     */
	public function add_hooks() {
		add_shortcode( 'app_list', array( $this,'display' ) );
	}

	/**
	 * Shortcode showing user's or worker's, or all appointments
	 * @since 2.0
	 */
	public function display( $atts = array() ) {

		# Let addons modify default columns
		$default_cols =  implode( ',', apply_filters( 'app_list_columns', array('id','service','client','date_time','end_date_time','status','cancel','confirm','edit','pay','zoom','jitsi','hangout') ) );

		$atts = shortcode_atts( array(
			'title'			=> '',
			'columns'		=> $default_cols,
			'columns_mobile'=> $default_cols,
			'what'			=> wpb_is_worker() ? 'worker' : 'client', 	// client, worker, all
			'user_id'		=> 0,
			'status'		=> implode( ',', apply_filters( 'app_list_status', array('paid','confirmed','pending','running') ) ),
			'service'		=> 0,										// Select a single or multiple services separated by comma
			'start'			=> 0,										// Start time of the appointments
			'end'			=> 0,										// End time of the appointments
			'order_by'		=> 'ID DESC',
			'limit'			=> 22,										// Client name character limit
			'edit_button'	=> $this->a->get_text('edit_button'),		// Text of edit button
			'edit_fields'	=> '',										// Which fields are allowed to be edited
			'cancel_button'	=> $this->a->get_text('cancel_button'),		// Text of cancel button
			'pay_button'	=> $this->a->get_text('pay_now'),			// Text of pay button
			'no_table'		=> 1, 										// Do Not display table if no results found
			'cap'			=> WPB_ADMIN_CAP, 							// Capability to view others appointments
			'id'			=> '',										// Optional ID
			'override'		=> 'auto',									// Will admin obey edit/cancel global settings of clients or override them?
			'tabletools'	=> 1,										// Adds tabletools buttons
			'_tablesorter'	=> 1, 										// To disable tablesorter just in case
			'_as_tooltip'	=> 0,										// Use monthly shortcode as tooltip for users page.
			'_email'		=> 0,										// To be used in email. Automatically set to 1 by _replace method
			'_children_of'	=> 0,										// Show only (filter) children of the app given its ID
			'_wide_coverage'=> 0,										// Includes apps started before start and ending after end too. Only valid if start and end are both defined
			'_user_page'	=> 0,										// User Page or BuddyPress
		), $atts, 'app_list' );

		extract( $atts );

		/* Compacted parameters */
		$args = compact( array_keys( $atts ) );

		# Set status clause
		$statuses = array_filter( array_map( 'trim', explode( ',', $status ) ) );

		if ( empty( $statuses ) ) {
			return WpBDebug::debug_text( __('Check "status" parameter in List shortcode','wp-base') ) ;
		}

		$can_edit		= wpb_current_user_can( $cap );
		$stat			= $this->stat_sql( $statuses );
		$service_sql	= $this->service_sql( $service );
		$datetime_sql	= $this->dt_sql( $start, $end, $_wide_coverage );

		# If this is a client shortcode
		if ( 'client' === $what ) {
			$q = $this->client_sql( $what, $cap, $user_id );

			if ( $q && $stat ) {
				$query = " SELECT * FROM " . $this->a->app_table .
						 " WHERE (".$q.") AND (".$stat.") AND ".$datetime_sql." AND (".$service_sql.")
						   ORDER BY " . $this->a->sanitize_order_by( $order_by );
				$results = $this->a->db->get_results( $query, OBJECT_K );
			} else {
				$results = false;
			}
		} else if ( 'worker' === $what ) {
			if ( $_as_tooltip )
				$results = $this->a->get_daily_reserve_apps_by_worker( $user_id, date("d F Y", strtotime( $start, $this->a->_time ) ) );
			else {
				$q 	= $this->worker_sql( $what, $cap, $user_id );
				$query = " SELECT * FROM " . $this->a->app_table .
						 " WHERE (".$q.") AND (".$stat.") AND ".$datetime_sql." AND (".$service_sql.")
 						  ORDER BY ". $this->a->sanitize_order_by( $order_by );
				$results = $this->a->db->get_results( $query, OBJECT_K  );
			}
		} else if ( 'all' === $what ) {
			if ( $can_edit ) {
				$query = " SELECT * FROM " . $this->a->app_table .
						 " WHERE (".$stat.") AND ".$datetime_sql." AND (".$service_sql.")
						   ORDER BY ".$this->a->sanitize_order_by( $order_by );
				$results = 	$this->a->db->get_results( $query, OBJECT_K  );
			} else {
				return WpBDebug::debug_text( __('Not enough capability to view bookings when "what" attribute is "all"','wp-base') ) ;
			}
		} else {
			return WpBDebug::debug_text( __('Check "what" parameter in List shortcode','wp-base') ) ;
		}

		$_columns 		= wpb_is_mobile() && trim( $columns_mobile ) ? $columns_mobile : $columns;
		$cols	 		= array_map( 'strtolower', explode( ',', wpb_sanitize_commas( $_columns ) ) );
		$allowed 		= $this->allowed_cols( $args );
		$no_results 	= true;
		$hard_limit 	= false;
		$this->colspan	= 0;

		$ret  = '';
		$ret .= '<div class="app-sc app-list-wrapper table-responsive">';
		$ret .= $this->title( $title, $what, $cap, $user_id );
		$ret  = apply_filters( 'app_list_before_table', $ret, $args );
		$ret .= "<table style='width:100%' data-args='".json_encode( $args )."' id='".$id."' class='app-list dt-responsive display dataTable nowrap'>";
		$ret .= '<thead><tr class="ui-state-default">';
		$ret .= $this->table_head( $cols, $allowed, $what, $results, $args );
		$ret .= '</tr></thead>';
		$ret .= '<tbody>';

		if ( $results ) {
			wp_cache_set( wpb_cache_prefix() . 'apps_in_listing', $results ); # Cache this so that we may use it.

			$kids			= $_children_of ? array_keys( BASE('Multiple')->get_children( $_children_of ) ) : array();
			$mtime			= wpb_microtime();
			$skip_children	= apply_filters( 'app_list_skip_children', false, $results );
			$skip_internal	= apply_filters( 'app_list_skip_internal', true, $results );

			wpb_update_app_meta_cache( array_keys( $results ) ); # Prime meta cache

			foreach ( $results as $app_id => $r ) {
				if ( apply_filters( 'app_list_skip_row', false, $cols, $args, $results, $r ) ) {
					continue;
				}

				if ( $_children_of && ! in_array( $r->ID, $kids ) ) {
					continue;
				}

				if ( $skip_children && $r->parent_id ) {
					continue;
				}

				if ( $skip_internal && $this->a->is_internal( $r->service ) ) {
					continue;
				}

				$no_results = false;

				$ret .= '<tr id="app-tr-'.$r->ID.'" class="app-service-'.$r->service.' app-worker-'.$r->worker.'">';

				foreach( $cols as $col ) {

					if ( ! in_array( $col, $allowed ) ) {
						continue;
					}

					if ( apply_filters( 'app_list_skip_cell', ( ('client' === $what && 'client' === $col) || (('worker' === $what || 'provider' === $what) && 'worker' === $col) ), $col, $args, $results, $r ) ) {
						continue;
					}

					$ret .= '<td class="'.$col.'-app-mng">';
					$ret .= $this->table_cell( $col, $r, $args );
					$ret .= '</td>';

				}
				$ret .= '</tr>';

				if ( ( wpb_microtime() - $mtime ) > WPB_HARD_LIMIT ) {
					$hard_limit = true;
					break;
				}
			}
		} else {
			$ret .= '<tr><td colspan="'.$this->colspan.'">'. $this->a->get_text('no_appointments'). '</td></tr>';
		}

		$ret .= '</tbody></table>';

		if ( $hard_limit ) {
			$ret .= WpBDebug::debug_text( sprintf(
					__('Hard limit activated. Execution time: %s secs.', 'wp-base' ),
					number_format( wpb_microtime() - $mtime, 1 )
				));
		}

		# If no results, maybe not produce a table
		if ( $no_results && $no_table ) {
			$ret = '<div class="app-list">';
			$ret .= $this->a->get_text( 'no_appointments' );
		}

		$ret  = apply_filters( 'app_list_table', $ret, $args, $results );
		$ret .= '</div>';

		return $ret;
	}

	/**
	 * Determine whose bookings are being displayed
	 * @return integer	User ID
	 */
	private function whose( $what, $cap, $user_id ) {
		$can_edit = wpb_current_user_can( $cap );

		if ( 'client' == $what ) {
			if ( $user_id && $can_edit ) {
				$user_id = $user_id;
			} else if ( isset( $_GET["app_user_id" ] ) && $can_edit ) {
				$user_id = absint( $_GET["app_user_id" ] );
			} else if ( is_user_logged_in() ) {
				$user_id = get_current_user_id();
			}
		} else if ( 'worker' == $what ) {
			if ( $user_id && $can_edit ) {
				$user_id = $user_id;
			} else if ( isset( $_GET["app_worker_id" ] ) && $can_edit ) {
				$user_id = absint( $_GET["app_worker_id" ] );
			} else if ( isset( $_GET["app_user_id" ] ) && $can_edit ) {
				$user_id = absint( $_GET["app_user_id" ] );
			} else if ( is_user_logged_in() ) {
				$user_id = get_current_user_id();
			}
		}

		return $user_id;
	}

	/**
	 * Build service sql. Intrinsically sanitized
	 * @param $service	integer|string|array	Service ID, multiple service IDs separated with comma or array of service IDs
	 * @return string
	 */
	public function service_sql( $service ) {
		$service_sql = '';
		if ( $service ) {
			$services = is_array( $service ) ? $service : explode( ',', $service );
			foreach ( $services as $s ) {
				if ( ! is_numeric( $s ) ) {
					$s = $this->a->find_service_id_from_name( $s );
				}

				# Allow only defined services
				if ( $this->a->service_exists( $s ) ) {
					$service_sql .= " service=".trim( $s )." OR ";
				}
			}

			$service_sql = rtrim( $service_sql, "OR " );
		}

		if ( !trim( $service_sql ) ) {
			$service_sql = ' 1=1 ';
		}

		return $service_sql;
	}

	/**
	 * Build status sql. Intrinsically sanitized
	 * @return string
	 */
	public function stat_sql( $statuses ) {
		if ( ! is_array( $statuses ) ) {
			$statuses = array_filter( array_map( 'trim', explode( ',', $status ) ) );
		}

		# Check for 'all'
		if ( empty( $statuses ) || in_array( 'all', $statuses ) ) {
			$stat = '1=1';
		} else {
			$stat = '';
			foreach ( $statuses as $s ) {
				# Allow only defined stats
				if ( array_key_exists( trim( $s ), $this->a->get_statuses() ) ) {
					$stat .= " status='".trim( $s )."' OR ";
				}
			}
			$stat = rtrim( $stat, "OR " );
		}

		return $stat;
	}

	/**
	 * Build date/time sql. Intrinsically sanitized
	 * @return string
	 */
	public function dt_sql( $start, $end, $_wide_coverage ) {
		if ( $start && $end ) {
			if ( $_wide_coverage ) {
				$datetime_sql = " start <= '" . wpb_date( $end ) . "' AND end > '" . wpb_date( $start ) . "' ";
			} else {
				$datetime_sql = " start >= '" . wpb_date( $start ) . "' AND start <= '" . wpb_date( $end ) . "' ";
			}
		} else {
			$datetime_sql = ' 1=1 AND';
			# This is different than is_busy. Here, we want to catch the start time of an appointment. So we dont look at app->end
			if ( $start ) {
				$datetime_sql = " start>='" . wpb_date( $start ) . "' AND";
			}

			if ( $end ) {
				$datetime_sql .= " start<'" . wpb_date( $end ) . "' ";
			}
		}

		return rtrim( $datetime_sql, "AND" );
	}

	/**
	 * Build sql query specific to client. Intrinsically sanitized
	 * Client can see a combination of bookings in his cookie and if logged in, bookings made with his ID
	 * @return string
	 */
	public function client_sql( $what, $cap, $user_id ) {
		$apps = (array)$this->a->get_apps_from_cookie();

		$q = '';
		foreach ( $apps as $app_id ) {
			if ( is_numeric( $app_id ) ) {
				$q .= " ID=".$app_id." OR ";
			}
		}
		$q = rtrim( $q, "OR " );

		# But he may as well has appointments added manually (requires being registered user) or we may be forcing to see a user
		if ( $user_id = $this->whose( $what, $cap, $user_id ) ) {
			$q .= " OR user=".$user_id;
		}

		return ltrim( $q, " OR" );
	}

	/**
	 * Build sql query specific to worker. Intrinsically sanitized
	 * Worker can see bookings where he is the worker
	 * @return string
	 */
	public function worker_sql( $what, $cap, $user_id ) {
		$user_id = $this->whose( $what, $cap, $user_id );

		# Only worker can see his bookings
		$q = $this->a->is_worker( $user_id ) ? "worker=".$user_id : "1=2";

		# Special case: If this is a single provider website, show staff appointments in his schedule too
		$workers = $this->a->get_workers();

		if ( current_user_can(WPB_ADMIN_CAP) && ( ( $workers && count( $workers ) == 1 ) || !$workers ) || apply_filters( 'app_list_allow_unassigned', false, $user_id ) ) {
			$q .= ' OR worker=0';
		}

		return $q;
	}

	/**
	 * Find a list of allowed columns
	 * @return array
	 */
	public function allowed_cols( $args ) {

		$allowed = array( 'id','created','location','location_address','service','worker','client','email','phone','city','address',
						'zip','country','note','price','date_time','end_date_time','date','day','time','status','pdf','balance',
						'deposit','total_paid' );

		# Filter pdf, paypal depending on existence of addons
		$allowed = array_flip( $allowed );

		if ( ! class_exists( 'WpBPDF' ) ) {
			unset( $allowed['pdf'] );
		}

		return apply_filters( 'app_list_allowed_columns', array_flip( $allowed ), $args );
	}

	/**
	 * Prepare title of the table
	 * @return string
	 */
	private function title( $title, $what, $cap, $user_id ) {
		if ( "0" !== (string)$title ) {
			if ( ! trim( $title ) ) {
				switch ( $what ) {
					case 'worker':
					case 'client':	$title = __('Bookings of USER_NAME', 'wp-base' ); break;
					case 'all':		$title = __('All Bookings', 'wp-base' ); break;
					default:		return WpBDebug::debug_text( __('Check "what" parameter in List shortcode','wp-base') ) ; break;

				}
			}
			$title_html = '<div class="app-title">' . esc_html( $title ) . '</div>';
		} else {
			$title_html = '';
		}

		return str_replace( 'USER_NAME', BASE('User')->get_name( $this->whose( $what, $cap, $user_id ) ), $title_html );
	}

	/**
	 * Prepare header of the table
	 * @return string
	 */
	private function table_head( $cols, $allowed, $what, $results, $args ) {
		$ret = '';
		foreach( $cols as $col ) {

			if ( ! in_array( $col, $allowed ) ) {
				continue;
			}

			if ( apply_filters( 'app_list_skip_col', ( ('client' === $what && 'client' === $col) || (('worker' === $what || 'provider' === $what) && 'worker' === $col) ), $col, $args, $results, $cols ) ) {
				continue;
			}

			$this->colspan++;
			$ret .= '<th class="app-list-col app-list-'.$col.'-header">';
			switch ($col) {
				case 'id':			$ret .= $this->a->get_text('app_id'); break;
				case 'provider':
				case 'worker':		$ret .= $this->a->get_text('provider'); break;
				case 'confirm':		$ret .= $this->a->get_text('confirm'); break;
				case 'gcal':		$ret .= $this->a->get_text('gcal'); break;
				case $col:			if ( 'udf_' == substr( $col, 0, 4 ) ) {
										$ret .= apply_filters('app_list_udf_column_title','',$col, $args);
									} else { # Any other column can get its name from custom texts, i.e. deposit
										$ret .= apply_filters( 'app_list_column_name', $this->a->get_text($col), $col, $args);
									}
									break;
				default:			break;
			}

			$ret .= '</th>';

		}

		return $ret;
	}

	/**
	 * Prepare a single cell in listing
	 * @since 3.0
	 */
	private function table_cell( $col, $r, $args ) {
		$ret 		= '';
		$paid 		= wpb_get_paid_by_app( $r->ID );
		$booking 	= new WpB_Booking( $r->ID );

		switch ( $col ) {
			case 'id':					$ret .= $this->table_cell_ID( $r, $args );	break;
			case 'created':				$ret .= date_i18n( $this->a->time_format, $this->a->client_time( strtotime( $r->created ) ) ); break;
			case 'created_by':			$ret .= BASE('User')->get_name( wpb_get_app_meta( $r->ID, 'created_by' ) ); break;
			case 'location':			$ret .= $this->a->get_location_name( $r->location ); break;
			case 'location_address': 	$ret .= wpb_get_location_meta( $r->location, 'address' ); break;

			case 'service':				$sname = apply_filters( 'app_list_service_name', $this->a->get_service_name( $r->service ), $r );
										$ret .= $this->is_parent( $r )
												? '<abbr class="app-list-service-name" data-app_id="'.$r->ID.'">'. $sname . '<abbr>'
												: $sname;
										break;

			case 'category':			if ( $maybe_category = get_app_meta( $r->ID, 'category' ) )
											$ret .= $this->a->get_category_name( $maybe_category );
										else
											$ret .= $this->a->guess_category_name( $r->service );
										break;
			case 'provider':
			case 'worker':				$ret .= $this->a->get_worker_name( $r->worker ); break;
			case 'client':				$ret .= BASE('User')->get_client_name( $r->ID, $r, true ). apply_filters( 'app_bookings_add_text_after_client', '', $r->ID, $r ); break;
			case 'price':				$ret .= wpb_format_currency( $r->price ); break;
			case 'deposit':				$ret .= wpb_format_currency( $r->deposit ); break;
			case 'total_paid':			$ret .= wpb_format_currency( $paid/100 ); break;
			case 'balance':				$balance = $paid/100 - $r->price + $r->deposit;
										$ret .= wpb_format_currency( $balance );
										break;
			case 'email':
			case 'phone':
			case 'address':
			case 'zip':
			case 'city':
			case 'country':
			case 'note':				$ret .= $booking->get_meta( $col ); break;
			case 'date_time': 			$ret .= $booking->client_dt(); break;
			case 'end_date_time':		$ret .= $booking->client_end_dt(); break;
			case 'date':				$ret .= $booking->client_date(); break;
			case 'day':					$ret .= $booking->client_day(); break;
			case 'time':				$ret .= $booking->client_time(); break;
			case 'status':				$ret .= $this->a->get_status_name( $r->status ); break;
			case $col:					$ret .= apply_filters( 'app_list_add_cell', '', $col, $r, $args );break;
		}

		return $ret;
	}

	/**
	 * Prepare ID cell
	 * @return string
	 */
	private function table_cell_ID( $r, $args ) {

		$id_text = apply_filters( 'app_ID_text', $r->ID, $r );

		if ( $args['_email'] ) {
			return $id_text;
		}

		$href		= '';
		$title_text = sprintf( __('Click to manage booking #%s','wp-base'), $r->ID );

		# Redirect user to manage bookings page (on admin or front end), if it exists
		if ( current_user_can( WPB_ADMIN_CAP ) ) {
			$href = admin_url( 'admin.php?page=app_bookings&status=all&stype=app_id&app_s='.$r->ID );
		}

		if ( $href = apply_filters( 'app_listing_ID_url', $href, $r, $args ) ) {
			return '<a href="'.esc_attr( $href ).'" title="'.$title_text.'">' .$id_text . '</a>';
		} else {
			return $id_text;
		}
	}

	/**
	 * Check if booking is parent of a recurring or package booking
	 * @return string
	 */
	private function is_parent( $r ) {
		return ( $r->parent_id && ( $this->a->is_app_recurring( $r->ID ) || $this->a->is_app_package( $r->ID ) ) );
	}

}
	BASE('Listing')->add_hooks();
}