<?php
/**
 * Addon Name: Confirm
 * Description: Handles confirmation of booking requests
 * Class Name: WpBConfirm
 * Version: 3.0.1
 * 
 * @package WP BASE
 * @author Hakan Ozevin
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( !class_exists( 'WpBConfirm' ) ) {

class WpBConfirm {
	
	/**
     * WP BASE Core + Front [+Admin] instance
     */
	protected $a = null;
	
	/**
     * Constructor
     */
	public function __construct() {
		$this->a = BASE();
	}
	
	/**
     * Add hooks
     */
	public function add_hooks(){
		add_filter( 'app_list_allowed_columns', array( $this, 'add_column' ), 10, 2 );		// Add column
		add_filter( 'app_list_add_cell', array( $this, 'add_button' ), 10, 4 );				// Add button in cell
		
		add_action( 'init', array( $this, 'confirm_from_email' ), 3 ); 						// Check confirmation of an appointment by email link
		add_action( 'wp_ajax_confirm_app', array( $this, 'confirm_from_list' ) ); 
	}
	
	/**
     * Check if a user (probably editor) can override confirm
	 * @return bool
     */
	public function can_override( $args ) {
		# Not capable
		$cap = ! empty( $args['cap'] ) ? $args['cap'] : '';
		
		if ( ! BASE('User')->is_capable( $cap ) ) {
			return false;
		}
		
		$override = ! empty( $args['override'] ) ? $args['override'] : '';
		
		# Can override because of shortcode setting
		if ( $override && 'inherit' != $override ) {
			return true;
		}
		
		return false;
	}

	/**
     * Whether add a column to Bookings List
	 * @return bool
     */
	public function add_column( $allowed, $args ) {
		
		if ( $this->a->is_worker( get_current_user_id() ) && 'yes' === wpb_setting('allow_worker_confirm') ) {
			$allowed[] = 'confirm';
		}
		
		return $allowed;
	}
	
	/**
     * Combine all conditions to see if confirm allowed
     */
	public function is_allowed( $r, $args ) {
		# We want only pending bookings to be confirmed
		$stat				= ! empty( $r->status ) ? $r->status : '';
		$allowed_as_client	= wpb_is_owner( $r ) && 'yes' == wpb_setting('allow_confirm');
		$allowed_as_worker	= $r->worker == get_current_user_id() && 'yes' == wpb_setting('allow_worker_confirm');
		$override			= $this->can_override( $args );
		
		if ( 'pending' == $stat && ( $override || $allowed_as_client || $allowed_as_worker ) ) {
			return true;
		}
		
		return false;		
	}
	
	/**
     * Whether add a button to the cell in Bookings List
     */
	public function add_button( $ret, $col, $r, $args ) {
		if ( 'confirm' != $col ) {
			return $ret;
		}
		
		if ( $this->is_allowed( $r, $args ) ) {
			$is_disabled = '';
			$title = '';
		} else {
			$is_disabled = ' app-disabled-button';
			$title = 'title="'. $this->a->get_text('not_possible').'"';
		}

		$ret .= '<button class="app-list-confirm ui-button ui-state-default '.$is_disabled.'" data-app_id="'.$r->ID.'" name="app_confirm['.$r->ID.']" />'.esc_html($this->a->get_text('confirm_button')).'</button>';
		
		return $ret;
	}
	
	/**
	 * Handle confirmation of an appointment by the client using email link
	 * @since 3.0
	 */
	public function confirm_from_email() {
		if ( ! empty( $_GET['app_re_confirm_login'] ) && !is_user_logged_in() ) {
			wpb_notice( $this->a->get_text('login_for_confirm') );
			return;
		}

		if ( ! empty( $_GET['app_confirmed'] ) && ! empty( $_GET['confirm_nonce'] ) ) {
			$app = wpb_get_app( wpb_clean( $_GET['app_confirmed'] ) );
			
			if ( $app && $_GET['confirm_nonce'] == $this->a->create_hash( $app, 'confirm' ) ) {
				BASE('Notices')->open_confirmation_dialog( $app );
			}

			return;
		}

		/* confirm by the link in email */
		if ( empty( $_GET['app_confirm'] ) || empty( $_GET['app_id'] ) || empty( $_GET['confirm_nonce'] ) ) {
			return;
		}

		if ( 'yes' != wpb_setting('allow_confirm') ) {
			if ( isset( $_REQUEST['app_id'] ) && isset( $_REQUEST['confirm_nonce'] ) ) {
				
				if ( wpb_doing_ajax() ) {
					die( json_encode( array( 'error' => $this->a->get_text('confirm_disabled') ) ) );
				} else {
					wpb_notice( 'confirm_disabled', 'error' );
				}
			}

			return;
		}

		$app_id = wpb_clean( $_GET['app_id'] );
		$app = wpb_get_app( $app_id );

		if (  $_GET['confirm_nonce'] != $this->a->create_hash( $app, 'confirm' ) ) {
			wpb_notice( 'error', 'error' );
			return;
		}

		if ( empty( $app->created ) || 'pending' != $app->status ) {
			wpb_notice( 'not_possible', 'error' );
			return; # Appt deleted completely
		}

		if ( ! empty( $app->user ) && !is_user_logged_in() ) {
			# Come back to the same page after login
			$redirect = wpb_add_query_arg( 'app_re_confirm_login', $app_id, wp_login_url( esc_url_raw($_SERVER['REQUEST_URI']) ) );
			wp_redirect( $redirect );
			exit;
		}

		if ( ! wpb_is_owner( $app ) ) {
			wpb_notice( 'not_owner', 'error' );
			return;
		}

		if ( $this->a->change_status( 'confirmed', $app_id ) ) {
			$this->a->log( sprintf( __('Client %1$s confirmed appointment having ID: %2$s','wp-base'), BASE('User')->get_client_name( $app_id, $app, false ), $app_id ) );
			$this->a->maybe_send_message( $app_id, 'confirmation' );
			
			# If headers not sent we can open the dialog
			if ( false && ! headers_sent() ) {
				BASE('Notices')->open_confirmation_dialog( $app );
				return;
			} else {
				$url = wpb_refresh_url( home_url() );
				wp_redirect( wpb_add_query_arg( array( 
					'app_confirmed' => $app_id,
					'confirm_nonce'	=> $this->a->create_hash( $app, 'confirm' ), 
				), $url ) );
				exit;
			}
		} else {
			# If failed (maybe already confirmed before), do something else here
			do_action( 'app_confirm_failed', $app_id );
		}
	}

	/**
     * Handle confirmation when client/worker clicks button in Booking List
     */
	public function confirm_from_list(){
		if ( empty( $_POST['app_id'] ) || empty( $_POST['confirm_nonce'] ) ) {
			die( json_encode( array( 'error' => $this->a->get_text('error') ) ) );
		}

		if ( ! check_ajax_referer( 'confirm-app', 'confirm_nonce', false ) ) {
			die( json_encode( array( 'error' => $this->a->get_text('unauthorised') ) ) );
		}

		$app_id	= absint( $_POST['app_id'] );
		$app	= wpb_get_app( $app_id );

		# Only pending bookings can be confirmed
		if ( 'pending' != $app->status ) {
			die( json_encode( array( 'error' => $this->a->get_text('not_possible') ) ) );
		}

		$has_right = ( $app->worker == get_current_user_id() && 'yes' == wpb_setting('allow_worker_confirm') ) 
					 || ( wpb_is_owner( $app ) && 'yes' == wpb_setting('allow_confirm') );

		if ( ! $has_right ) {
			die( json_encode( array( 'error' => $this->a->get_text( 'not_possible' ) ) ) );
		}

		// Now we can safely continue for confirm
		if ( $this->a->change_status( 'confirmed', $app_id, false ) ) {

			if ( wpb_is_owner( $app ) ) {
				$this->a->log( sprintf( __('Client %1$s confirmed own booking with ID: %2$s','wp-base'), BASE('User')->get_client_name( $app_id, null, false ), $app_id ) );
			} else if ( $app->worker == get_current_user_id() ) {
				$this->a->log( sprintf( __('Provider %1$s confirmed own booking with ID: %2$s','wp-base'), $this->a->get_worker_name( $app->worker ), $app_id ) );
			} else {
				$this->a->log( sprintf( __(' %1$s confirmed appointment of %2$s having ID: %3$s','wp-base'), BASE('User')->get_name(), BASE('User')->get_client_name( $app_id, null, false ), $app_id ) );
			}

			$this->a->maybe_send_message( $app_id, 'confirmation' );
			
			do_action( 'app_booking_confirmed', $app_id );

			die( json_encode( array( 'success' => 1 ) ) );

		} else {
			die( json_encode( array( 'error' => $this->a->get_text( 'not_possible' ) ) ) );
		}
	}	
	
}
	BASE('Confirm')->add_hooks();
}