<?php
/**
 * Addon Name: Cancel
 * Description: Handles cancel requests
 * Class Name: WpBCancel
 * Version: 3.0.1
 *
 * @package WP BASE
 * @author Hakan Ozevin
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( !class_exists( 'WpBCancel' ) ) {

class WpBCancel {

	/**
     * WP BASE Core + Front [+Admin] instance
     */
	protected $a = null;

	/**
     * Constructor
     */
	public function __construct() {
		$this->a = BASE();
	}

	/**
     * Add hooks
     */
	public function add_hooks(){
		add_filter( 'app_list_allowed_columns', array( $this, 'add_column' ), 10, 2 );		// Add column
		add_filter( 'app_list_add_cell', array( $this, 'add_button' ), 10, 4 );				// Add button in cell

		add_action( 'init', array( $this, 'cancel_from_email' ), 3 ); 						// Check cancellation of an appointment from email link
		add_action( 'wp_ajax_cancel_app', array( $this, 'cancel_from_list' ) ); 			// Cancel appointment from List shortcode
		add_action( 'wp_ajax_nopriv_cancel_app', array( $this, 'cancel_from_list' ) ); 		// Cancel appointment from List shortcode
	}

	/**
     * Check if a user (probably editor) can override cancel
	 * @return bool
     */
	public function can_override( $args ) {
		# Not capable
		$cap = ! empty( $args['cap'] ) ? $args['cap'] : '';

		if ( ! wpb_current_user_can( $cap ) ) {
			return false;
		}

		$override = ! empty( $args['override'] ) ? $args['override'] : '';

		# Can override because of shortcode setting
		if ( $override && 'auto' != $override ) {
			return true;
		}

		$is_client = ! empty( $args['what'] ) && 'worker' != $args['what'];

		# Can override because of global setting
		if ( 'auto' == $override && ( ($is_client && 'yes' == wpb_setting( 'allow_cancel' )) || (!$is_client && 'yes' == wpb_setting( 'allow_worker_cancel' )) ) ) {
			return true;
		}

		return false;
	}

	/**
     * Check if a worker can cancel
	 * @return bool
     */
	public function can_worker_cancel( $args ) {
		return $this->a->is_worker( get_current_user_id() ) && 'yes' == wpb_setting( 'allow_worker_cancel' );
	}

	/**
     * Check if clients can cancel (owner check not done here)
	 * @return bool
     */
	public function can_client_cancel( $args ) {

		$is_client = ! empty( $args['what'] ) && 'worker' != $args['what'];

		return $is_client && 'yes' == wpb_setting('allow_cancel');
	}

	/**
     * Add cancel column to Bookings List
	 * @return array
     */
	public function add_column( $allowed, $args ) {

		# User (editor) can override cancel
		if ( $this->can_override( $args ) || $this->can_worker_cancel( $args ) || $this->can_client_cancel( $args ) ) {
			$allowed[] = 'cancel';
		}

		return $allowed;
	}

	/**
     * Check if status of a booking is allowed to cancel
	 * @param $r		object		Booking object
	 * @return bool
     */
	public function in_allowed_status( $r, $args  = array() ) {
		$stat = ! empty( $r->status ) ? $r->status : '';

		if ( !$stat || 'removed' == $stat ) {
			return false;
		}

		$statuses = apply_filters( 'app_cancel_allowed_status', array( 'pending','confirmed','paid' ), $r->ID, $args );

		return in_array( $stat, $statuses );
	}

	/**
     * Combine all conditions to see if cancel allowed
     */
	public function is_allowed( $r, $args ) {
		# We don't want completed appointments to be cancelled
		# Also check if cancel time has been passed
		# Even admin cannot cancel already cancelled app
		$stat				= ! empty( $r->status ) ? $r->status : '';
		$in_allowed_status	= $this->in_allowed_status( $r, $args );
		$allowed_as_client	= wpb_is_owner( $r ) && $this->can_client_cancel( $args );
		$allowed_as_worker	= $r->worker == get_current_user_id() && $this->can_worker_cancel( $args );
		$override			= $this->can_override( $args );
		$is_late			= $this->is_too_late( $r );

		if ( ( 'removed' != $stat && $override ) || ( $in_allowed_status && !$is_late && ( $allowed_as_client || $allowed_as_worker ) ) ) {
			return true;
		}

		return false;
	}

	/**
     * Whether add a button to the cell in Bookings List
     */
	public function add_button( $ret, $col, $r, $args ) {
		if ( 'cancel' != $col ) {
			return $ret;
		}

		if ( $this->is_allowed( $r, $args ) ) {
			$is_disabled	= '';
			$title			= '';
		} else {
			$is_disabled	= ' app-disabled-button';
			$too_late		= $this->is_too_late( $r );
			$title			= 'title="'. $this->a->get_text( $too_late ? 'too_late' : 'not_possible' ).'"';
		}

		$button_text = ! empty( $args['cancel_button'] ) ? $args['cancel_button'] : $this->a->get_text('cancel_button');
		$ret .= '<button '.$title.' class="app-list-cancel ui-button ui-state-default '.$is_disabled.'" data-app_id="'.$r->ID.'" name="app_cancel['.$r->ID.']" >'.$button_text.'</button>';

		return $ret;
	}

	/**
     * Handle cancellation when client clicks link in email
     */
	public function cancel_from_email(){

		# Bypass/skip cancel hash check
		$bypass = apply_filters( 'app_cancel_skip_check', false );

		# Display cancelled notice
		if ( ! empty( $_GET['app_cancelled'] ) && ( ! empty($_GET['cancel_nonce'] || $bypass ) ) ) {
			$app = wpb_get_app( wpb_clean( $_GET['app_cancelled'] ) );
			if ( $bypass || $_GET['cancel_nonce'] == $this->a->create_hash( $app, 'cancel' ) ) {
				wpb_notice( $this->a->get_text('cancelled') );
				return;
			}
		}

		if ( ! empty( $_GET['app_re_cancel_login'] ) && !is_user_logged_in() ) {
			wpb_notice( $this->a->get_text('login_for_cancel') );
			return;
		}

		# Check if we received required parameters
		if ( empty( $_GET['app_cancel'] ) || empty( $_GET['app_id'] ) || ( empty( $_GET['cancel_nonce'] ) && ! $bypass ) ) {
			return;
		}

		# Only clients can use this option - Check if cancellation is still open
		if ( 'yes' != wpb_setting('allow_cancel') ) {
			if ( isset( $_GET['app_id'] ) && isset( $_GET['cancel_nonce'] ) ) {

				if ( wpb_doing_ajax() ) {
					die( json_encode( array( 'error' => $this->a->get_text('cancel_disabled') ) ) );
				} else {
					wpb_notice( 'cancel_disabled', 'error' );
				}
			}

			return;
		}

		$app_id = wpb_clean( $_GET['app_id'] );
		$app = wpb_get_app( $app_id );

		# Check provided hash
		if ( ! $bypass && $_GET['cancel_nonce'] != $this->a->create_hash( $app, 'cancel' ) ) {
			wpb_notice( 'error', 'error' );
			return;
		}

		if ( empty( $app->created ) || !$this->in_allowed_status( $app ) ) {
			wpb_notice( 'not_possible', 'error' );
			return; # Appt deleted completely or already cancelled
		}

		if ( $this->is_too_late( $app ) ) {
			wpb_notice( 'too_late', 'error' );
			return;
		}

		if ( ! empty( $app->user ) && !is_user_logged_in() ) {
			$redirect = wpb_add_query_arg( 'app_re_cancel_login', $app_id, wp_login_url( esc_url_raw($_SERVER['REQUEST_URI']) ) );
			wp_redirect( $redirect );
			exit;
		}

		if ( ! wpb_is_owner( $app ) ) {
			wpb_notice( 'not_owner', 'error' );
			return;
		}

		add_filter( 'app_skip_child_status_change', array( $this, 'skip_child_status_change' ), 10, 2 );

		if ( $this->a->change_status( 'removed', $app_id ) ) {
			$this->a->log( sprintf( __('Client %1$s cancelled appointment having ID: %2$s','wp-base'), BASE('User')->get_client_name( $app_id, $app, false ), $app_id ) );
			$this->a->maybe_send_message( $app_id, 'cancellation' );
			wpb_update_app_meta( $app_id, 'abandoned', 'cancelled' );
			
			do_action( 'app_booking_cancelled', $app_id );

			# If there would be a header warning other plugins can do whatever they could do
			if ( ! headers_sent() ) {
				$url = wpb_setting('cancel_page') ? get_permalink( wpb_setting('cancel_page') ) : home_url();
				wp_redirect( wpb_add_query_arg( array(
					'app_cancelled'	=> $app_id,
					'cancel_nonce'	=> $this->a->create_hash( $app, 'cancel' )
					), $url ) );
				exit;
			}
		} else {
			# If failed for any unknown reason, gracefully go to home page or do something else here
			do_action( 'app_cancel_failed', $app_id );
		}
	}

	/**
     * Prevent completed child bookings to be removed
     */
	public function skip_child_status_change( $skip, $child ) {
		if ( ! empty( $child->status ) && 'completed' == $child->status ) {
			return true;
		}

		return $skip;
	}

	/**
     * Handle cancellation when client clicks button in Booking List
     */
	public function cancel_from_list(){
		if ( empty( $_POST['app_id'] ) || empty( $_POST['cancel_nonce'] ) ) {
			die( json_encode( array( 'error' => $this->a->get_text('error') ) ) );
		}

		# If client has been logged off, this will fail
		if ( ! check_ajax_referer( 'cancel-app', 'cancel_nonce', false ) ) {
			die( json_encode( array( 'error' => $this->a->get_text('unauthorised') ) ) );
		}

		$app_id	= absint( $_POST['app_id'] );
		$app	= wpb_get_app( $app_id );
		$args	= ! empty( $_POST['args'] ) ? wpb_clean( $_POST['args'] ) : array();

		# If it is already cancelled, e.g. by admin
		if ( 'removed' === $app->status ) {
			die( json_encode( array( 'error' => $this->a->get_text('not_possible') ) ) );
		}

		$override_cancel = $this->can_override( $args );

		# Addons may want to do something here
		$is_owner = apply_filters( 'app_cancellation_owner', wpb_is_owner( $app ), $app );

		$has_right = ( $app->worker == get_current_user_id() && $this->can_worker_cancel( $args ) ) || ( $is_owner && $this->can_client_cancel( $args ) );

		# Too late message and final checks
		$in_allowed_stat = $this->in_allowed_status( $app );
		$too_late = $this->is_too_late( $app );

		if ( ! ($override_cancel || ( $in_allowed_stat && !$too_late && $has_right )) ) {
			die( json_encode( array( 'error' => $this->a->get_text( $too_late ? 'too_late' : 'unauthorised' ) ) ) );
		}

		// Now we can safely continue for cancel
		if ( $this->a->change_status( 'removed', $app_id, false ) ) {

			if ( wpb_is_owner( $app ) ) {
				wpb_update_app_meta( $app_id, 'abandoned', 'cancelled' );
				$this->a->log( sprintf( __('Client %1$s cancelled own booking with ID: %2$s','wp-base'), BASE('User')->get_client_name( $app_id, null, false ), $app_id ) );
			} else if ( $app->worker == get_current_user_id() ) {
				wpb_update_app_meta( $app_id, 'abandoned', 'worker_cancelled' );
				$this->a->log( sprintf( __('Provider %1$s cancelled own booking with ID: %2$s','wp-base'), $this->a->get_worker_name( $app->worker ), $app_id ) );
			} else {
				wpb_update_app_meta( $app_id, 'abandoned', 'editor_cancelled' );
				$this->a->log( sprintf( __(' %1$s cancelled appointment of %2$s having ID: %3$s','wp-base'), BASE('User')->get_name(), BASE('User')->get_client_name( $app_id, null, false ), $app_id ) );
			}

			$this->a->maybe_send_message( $app_id, 'cancellation' );
			
			do_action( 'app_booking_cancelled', $app_id );

			die( json_encode( array( 'success' => 1 ) ) );

		} else {
			die( json_encode( array( 'error' => $this->a->get_text( 'not_possible' ) ) ) );
		}
	}

	/**
	 * Check if appointment is too late to be cancelled
	 * @param app_id: ID of the appointment
	 * @since 2.0
	 */
	public function is_too_late( $app ) {
		$cancel_limit = apply_filters( 'app_cancel_limit', (int)wpb_setting( 'cancel_limit' ), $app->ID );

		$result = $this->a->_time > strtotime( $app->start, $this->a->_time ) - intval( $cancel_limit * 3600 );

		return apply_filters( 'app_cancel_is_too_late', $result, $app );
	}


}
	BASE('Cancel')->add_hooks();
}