<?php
/**
 * Addon Name: Account
 * Description: Handles account page
 * Class Name: WpBAccount
 * Version: 3.6.0
 *
 * @package WP BASE
 * @author Hakan Ozevin
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpBAccount' ) ) {

class WpBAccount {

	/**
     * WP BASE Core + Front [+Admin] instance
     */
	protected $a = null;

	public $account_assets_called;

	/**
     * Constructor
     */
	public function __construct() {
		$this->a = BASE();
	}

	/**
     * Add hooks
     */
	public function add_hooks() {
		add_shortcode( 'app_account', array( $this, 'account' ) );
		add_action( 'init', array( $this, 'save_profile' ) );
		add_action( 'template_redirect', array( $this, 'save_account_settings' ) );
		add_action( 'app_shortcode_found', array( $this, 'request_account_assets' ), 10, 2 );
	}	

	public function request_account_assets( $context, $post ){
		if ( 'account' != $context ) {
			return;
		}
		
		add_filter( 'app_is_admin_page', '__return_true' );
		add_filter( 'app_requires_front', '__return_true' );
		add_filter( 'app_is_account_page', '__return_true' );
		add_action( 'wp_enqueue_scripts', array( $this, 'load_account_assets' ), 14 );
	}

	/**
	 * Load js and css files
	 */
	public function load_account_assets( ) {
		if ( ! empty( $this->account_assets_called ) ) {
			return;
		}

		wp_enqueue_style( 'wp-admin' );
		wp_enqueue_style( 'buttons' );
		wp_enqueue_style( 'ie' );
		add_filter( 'body_class', array( $this, 'body_class' ) );
		add_action( 'wp_enqueue_scripts', array( BASE('Assets'), 'load_admin' ), 16 );
		add_action( 'wp_enqueue_scripts', array( BASE('Assets'), 'load_admin_css' ), 24 );
		
		add_filter( 'app_wh_default_selected', 'wpb_wh_default_selected' );
		add_filter( 'app_wh_annual_default_selected', 'wpb_wh_annual_default_selected' );
		add_filter( 'app_holidays_default_selected', 'wpb_wh_annual_default_selected' );

		do_action( 'app_account_load_assets', $this );

		$this->account_assets_called = true;
	}

	/**
	 * If this is an WP BASE account page mark body class
	 * @param $classes	array	body classes
	 * @uses body_classes filter
	 * @return array
	 */
	public function body_class( $classes ) {
		$classes[] = 'app-account-page';

		return $classes;
	}

	/**
	 * Add account tabs
	 */
	public function account( $atts ) {
		$pars = array(
			'title'				=> 0,	# Title. 0 means no title
			'exclude'			=> '',	# Which tab to exclude. Comma delimited list from: bookings, weekly, settings, etc. Full list is in show_profile method of user.php
			'listing_columns'	=> '',	# Select specific columns for Bookings List. For selectable options see show_profile_bookings method in user.php
			'listing_status'	=> '',	# Booking list will include bookings in which status. For selectable options see show_profile_bookings method in user.php
			'manage_status'		=> '',	# Status to display for manage bookings tab. Default (when empty): any, active, pending, removed
			'manage_add_export' => '',	# Whether enable Export CSV options to manage bookings tab. Set 0 to disable. 
			'payment_columns'	=> 'payment_id,client,app_id,location,service,worker,date,amount,reference',	# Which columns to display in Payment tab. For selectable options see allowed_columns methods in transactions.php file
		);

		extract( shortcode_atts( $pars, $atts, 'app_account' ) );

		if ( $title !== '' ) {
			$this->account_title = $title;
			add_filter( 'app_account_title', array( $this, 'account_title' ) );
		}

		if ( $exclude ) {
			$this->exclude_account_tabs = $exclude;
			add_filter( 'app_user_profile_tabs', array( $this, 'account_tabs' ), 20, 2 );
		}

		if ( $listing_columns ) {
			$this->listing_columns = $listing_columns;
			add_filter( 'app_user_listing_columns', array( $this, 'account_listing_columns' ), 20, 2 );
		}

		if ( $listing_status ) {
			$this->listing_status = $listing_status;
			add_filter( 'app_user_listing_status', array( $this, 'account_listing_status' ), 20, 2 );
		}
		
		if ( $manage_status || '' != $manage_add_export ) {
			$this->manage_status		= $manage_status;
			$this->manage_add_export	= $manage_add_export;
			add_filter( 'app_account_manage_args', array( $this, 'account_manage_args' ), 20, 2 );
		}

		if ( $payment_columns ) {
			$this->payment_columns = $payment_columns;
			add_filter( 'app_transactions_allowed_columns', array( $this, 'account_payment_columns' ), 20, 2 );
		}

		ob_start();
		BASE('User')->show_profile();
		$out = ob_get_contents();
		ob_end_clean();

		return $out;
	}

	/**
	 * Rename account page
	 */
	public function account_title( $title ) {
		if ( !isset( $this->account_title ) ) {
			return $title;
		}

		return '0' === (string)$this->account_title ? '' : $this->account_title;
	}

	/**
	 * Filter out account tabs
	 */
	public function account_tabs( $tabs, $profileuser_id ) {
		if ( empty( $this->exclude_account_tabs ) ) {
			return $tabs;
		}

		$exclude = wpb_explode( $this->exclude_account_tabs, ',' );

		return array_diff_key( $tabs, array_flip( $exclude ) );
	}

	/**
	 * Control which columns to be displayed
	 */
	public function account_listing_columns( $array, $profileuser_id ) {
		if ( empty( $this->listing_columns ) ) {
			return $array;
		}

		return wpb_explode( $this->listing_columns, ',' );
	}

	/**
	 * Control which status to be displayed
	 */
	public function account_listing_status( $array, $profileuser_id ) {
		if ( empty( $this->listing_status ) ) {
			return $array;
		}

		return wpb_explode( $this->listing_status, ',' );
	}

	/**
	 * Control which status and columns to be displayed in manage tab
	 */
	public function account_manage_args( $array, $profileuser_id ) {
		
		if ( ! empty( $this->manage_status ) ) {
			$array['status'] = $this->manage_status;
		}

		if ( isset( $this->manage_add_export ) ) {
			$array['add_export'] = $this->manage_add_export;
		}

		return $array;
	}

	/**
	 * Control which columns to be displayed
	 */
	public function account_payment_columns( $array, $profileuser_id ) {
		if ( empty( $this->payment_columns ) ) {
			return $array;
		}

		return wpb_explode( $this->payment_columns, ',' );
	}
	
	/**
	 * Save profile for account page
	 */
	public function save_profile() {
		if ( wpb_is_admin() ) {
			return;
		}

		if ( empty( $_POST['app_nonce'] ) || empty( $_POST['action_app'] ) ||
			 empty( $_POST['profileuser_id'] ) || 'save_profile' != $_POST['action_app'] ) {
			return;
		}

		if ( ! wp_verify_nonce( $_POST['app_nonce'], 'update_app_settings') ) {
			return;
		}

		BASE('User')->save_profile( );
	}

	/**
	 * Save settings in account page
	 */
	public function save_account_settings() {
		if ( wpb_is_admin() ) {
			return;
		}

		if ( empty( $_POST['app_nonce'] ) || empty( $_POST['action_app'] ) ) {
			return;
		}

		if ( ! wp_verify_nonce( $_POST['app_nonce'], 'update_app_settings') ) {
			return;
		}

		include_once( WPBASE_PLUGIN_DIR . '/includes/admin/base-admin.php' );

		do_action( 'app_save_account_settings' );
	}

	
}
	BASE('Account')->add_hooks();
}