<?php
/**
 * WPB LSW Menu/Hierarchy Controller
 *
 * Controls menu and selection behaviour accounting for priority (variable leader/follower control)
 * Cuts out related subset of locations/services/workers
 *
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpB_Controller' ) ) {

class WpB_Controller {

	private $set_location, $set_service, $set_worker;
	
	private $category = 0;

	private $req_location, $req_service, $req_worker, $order_by, $norm, $priority;

	public $locations, $services, $workers;

	public $display_errors = true;

	private $error = array();

	private $force_priority, $no_pref_can_stay_flag, $is_event;
	
	private $slide_no = 0;

	protected $a;

	# These will be used for priority comparisons
	# Just for better readability
	const LOCATION	= 'L';
	const SERVICE 	= 'S';
	const WORKER	= 'W';

	/**
	 * Constructor
	 * @param	$norm			object		WpB_Norm or WpB_Booking instance
	 * @param	$order_by		string		How menus will be ordered
	 * @param	$force_priority	string		Change priority by instance, e.g. in admin inline edit
	 */
	public function __construct( $norm_or_booking, $order_by = "sort_order", $force_priority = false, $event = null ) {
		$this->a = BASE();
		if ( $norm_or_booking instanceof WpB_Booking ) {
			$booking					= $norm_or_booking;
			$this->req_location			= $booking->get_location();
			$this->req_service			= $booking->get_service();
			$this->req_worker			= $booking->get_worker();
			$this->is_event				= $booking->is_event() || $event;
			$this->error				= array();
			$this->norm					= new WpB_Norm( $this->req_location, $this->req_service, $this->req_worker );
		} else {
			$norm						= $norm_or_booking;
			$this->norm					= $norm;
			$this->req_location			= $norm->get_location();
			$this->req_service			= $norm->get_service();
			$this->req_worker			= $norm->get_worker();
			$this->category				= $norm->get_category();
			$this->error				= $norm->error;
			$this->is_event				= $event;
		}

		$this->order_by				= $order_by;
		$this->force_priority		= is_string( $force_priority ) && '' === str_ireplace( array( 'LSW','LWS','SWL','SLW','WSL','WLS' ), '', $force_priority ) 
									  ? strtoupper( $force_priority ) 
									  : false;

		if ( 'all' === $this->req_service ) {
			$this->set_service = 'all';
			$this->set_worker = $this->a->get_def_wid();
			return;
		}

		if ( $this->is_event || $this->req_service < 0 ) {
			$this->is_event		= true;
			$this->set_location = $this->req_location;
			$this->set_service 	= $this->req_service;
			$this->locations 	= $this->a->get_locations();
			$this->services 	= apply_filters( 'app_controller_services', array() );
			$this->workers 		= array();
			return;
		}

		foreach ( str_split( $this->get_priority() ) as $lsw ) {
			switch( $lsw ) {
				case self::LOCATION:	$this->adjust_location();	break;
				case self::SERVICE:		$this->adjust_service();	break;
				case self::WORKER:		$this->adjust_worker();		break;
			}
		}
	}

	/**
	 * Getters
	 * @return integer
	 */
	 public function get_location(){
		 return $this->set_location;
	 }
	 public function get_service(){
		 return $this->set_service;
	 }
	 public function get_worker(){
		 return $this->set_worker;
	 }
	 public function req_location(){
		 return $this->req_location;
	 }
	 public function req_service(){
		 return $this->req_service;
	 }
	 public function req_worker(){
		 return $this->req_worker;
	 }	 
	 public function set_service( $s ){
		 $this->set_service = $s;
	 }

	 /**
	 * Whether display locations menu
	 * @return bool
	 */
	public function show_locations_menu(){
		return $this->norm ? $this->norm->show_locations_menu && $this->is_loc_active() : false;
	}

	/**
	 * Whether display services menu
	 * @return bool
	 */
	public function show_services_menu(){
		return $this->norm ? $this->norm->show_services_menu : false;
	}

	/**
	 * Whether display locations menu
	 * @return bool
	 */
	public function show_workers_menu(){
		return $this->norm ? $this->norm->show_workers_menu && $this->is_workers_active() : false;
	}

	/**
	 * Whether locations is active
	 * @return bool
	 */
	private function is_loc_active(){
		return class_exists( 'WpBLocations' ) && $this->a->get_nof_locations();
	}

	/**
	 * Whether workers is active
	 * @return bool
	 */
	private function is_workers_active(){
		return class_exists( 'WpBSP' ) && $this->a->get_nof_workers();
	}

	/**
	 * Get a code showing priority among Location (L), Service (S), Worker (W)
	 * e.g. SLW means Service > Location > Worker
	 * @return 		string		Priority coded with 3 letters for Location (L), Service (S), Worker (W)
	 */
	private function get_priority(){
		if ( $this->force_priority ) {
			return $this->force_priority;
		}

		if ( ! empty( $this->priority ) ) {
			return $this->priority;
		}

		return $this->priority = apply_filters( 'app_lsw_priority', wpb_setting( 'lsw_priority', WPB_DEFAULT_LSW_PRIORITY ), $this ); # default is in defaults.php
	}

	/**
	 * Check which variable has priority based on lsw_priority setting
	 * @param $first	string		L, S or W
	 * @param $second	string		L, S or W
	 * @return 			bool		true if $first is on the left of $second
	 */
	private function compare( $first, $second ) {
		$priority = $this->get_priority();
		return strpos( $priority, $first ) < strpos( $priority, $second );
	}

	/**
	 * Helper to check if no preference (i.e. worker=0) is allowed to stay
	 * @return bool
	 */
	private function no_pref_can_stay( ){
		if ( apply_filters( 'app_no_pref_can_stay', false, $this ) ) {
			return true;
		}
		
		if ( $this->force_priority || substr( $this->get_priority(), 0, 1 ) == 'W' ) {
			return false;
		}

		if ( null !== $this->no_pref_can_stay_flag ) {
			return $this->no_pref_can_stay_flag;
		}
		
		$set = wpb_setting( 'client_selects_worker' );

		return $this->no_pref_can_stay_flag = ( 'auto' == $set || 'forced_late' == $set ) && ((defined( 'WPB_AJAX' ) && WPB_AJAX) || 'yes' == wpb_setting( 'lazy_load' ));
	}

	/**
	 * Helper to get location or service or worker property value
	 * @return mix
	 */
	private function lsw( $context ) {
		if ( 'location' == $context ) {
			return $this->norm->get_location();
		}

		if ( 'service' == $context ) {
			return $this->norm->get_service();
		}

		if ( 'worker' == $context ) {
			return $this->norm->get_worker();
		}
	}

	/**
	 * Helper to set a random worker as follower 
	 * @param $ids		array	List of worker IDs
	 * @since 3.4.4
	 * @return none
	 */	
	private function set_random_worker( $ids ) {
		shuffle( $ids );
		if ( empty( $_POST['app_value'] ) && defined( 'WPB_CHECK_WORKER_AVAIL' ) && WPB_CHECK_WORKER_AVAIL ) {
			$today = strtotime( 'today', $this->a->_time );
			foreach ( $ids as $id ) {
				$slot = new WpB_Slot( new WpB_Calendar( $this->set_location, $this->set_service, $id ), $today, ($today + DAY_IN_SECONDS - 1) );
				if ( BASE('WH')->is_working_day( $slot ) && !BASE('Holidays')->is_holiday( $slot, 'worker' ) ) {
					$this->set_worker = $id;
					return;
				}
			}
			$this->set_worker = 0; # No available worker found	
		} else {
			$this->set_worker = current( $ids );
		}			
	}

	/**
	 * Helper to preselect a "follower", e.g. when service has first priority, worker is a follower
	 * @param $context	string		location, service or worker
	 * @return none
	 */
	private function preselect_follower( $context ) {

		$vars = $this->{$context.'s'}; # e.g. $this->services

		$ids = ! empty( $vars ) ? array_keys( $vars ) : array();
		$lsw = $this->lsw( $context );

		if ( $ids ) {
			if ( in_array( $this->{'req_'.$context}, $ids ) ) {
				$this->{'set_'.$context} = $this->{'req_'.$context};
				$this->{'show_'.$context.'_menu'} = false; # Force selection valid
			} else if ( $lsw && in_array( $lsw, $ids ) ) {
				$this->{'set_'.$context} = $lsw;
			} else if ( 'worker' == $context && $this->no_pref_can_stay( ) ) {
				$this->{'set_'.$context} = 0;
			} else if ( 'worker' == $context && 'random' == wpb_setting( 'assign_worker') ) {
				$this->set_random_worker( $ids );
			} else {
				$this->{'set_'.$context} = current( $ids );
			}
		} else {
			$this->{'set_'.$context} = 0;
		}
	}

	/**
	 * Helper to preselect a "leader", i.e. lsw that has first priority
	 * @param $context	string		location, service or worker
	 * @return none
	 */
	private function preselect_leader( $context ) {
		$lsw = $this->lsw( $context );

		$this->{'set_'.$context} =	$this->{'req_'.$context} > 0
									? $this->{'req_'.$context}
									: ( $lsw ? $lsw : key( $this->{$context.'s'} ) );
	}

	/**
	 * Manage locations object
	 * @return none
	 */
	private function adjust_location(){
		if ( false === $this->req_location ) {
			$this->set_location = false;
			return;
		}

		if ( ! class_exists( 'WpBLocations' ) ) {
			$this->set_location = false;
			return;
		}

		if ( $this->compare( self::SERVICE, self::LOCATION ) ) {

			$this->locations = $this->a->get_locations_by_service( $this->set_service, $this->order_by );

			if ( $this->is_workers_active() && $this->compare( self::WORKER, self::LOCATION ) ) {
				$this->locations = array_intersect_key( (array)$this->locations, (array)$this->a->get_locations_by_worker( $this->set_worker, $this->order_by ) );
			}
			
			$this->preselect_follower( 'location' );
		} else if ( $this->is_workers_active() && $this->compare( self::WORKER, self::LOCATION ) ) {
			$this->locations = $this->a->get_locations_by_worker( $this->set_worker, $this->order_by );

			$this->preselect_follower( 'location' );
		} else {
			$this->locations = $this->a->get_locations( $this->order_by );
			$this->preselect_leader( 'location' );
		}

		$this->norm->set_location( $this->set_location );
	}

	/**
	 * Manage services object
	 * @return none
	 */
	private function adjust_service(){
		if ( false === $this->req_service ) {
			$this->set_service = false;
			return;
		}

		if ( $this->is_loc_active() && $this->compare( self::LOCATION, self::SERVICE ) ) {

			$this->services = $this->a->get_services_by_location( $this->set_location, $this->order_by );

			if ( $this->is_workers_active() && $this->compare( self::WORKER, self::SERVICE ) ) {
				$this->services = array_intersect_key( (array)$this->services, (array)$this->a->get_services_by_worker( $this->set_worker, $this->order_by ) );
			}
			
			$this->preselect_follower( 'service' );
		} else if ( $this->is_workers_active() && $this->compare( self::WORKER, self::SERVICE ) ) {
			$this->services = $this->a->get_services_by_worker( $this->set_worker, $this->order_by );
			$this->preselect_follower( 'service' );
		} else {
			$this->services = $this->a->get_services( $this->order_by, !wpb_is_admin(), $this->category );
			if ( empty( $this->services ) ) {
				$this->error[] = sprintf( __( 'No services to select for category %s', 'wp-base' ), $this->category );
			}
			$this->preselect_leader( 'service' );
		}

		$this->norm->set_service( $this->set_service );
	}

	/**
	 * Manage workers object
	 * @return none
	 */
	private function adjust_worker(){
		if ( false === $this->req_worker ) {
			$this->set_worker = false;
			return;
		}

		if ( ! class_exists( 'WpBSP' ) ) {
			$this->set_worker = false;
			return false;
		}

		if ( $this->compare( self::SERVICE, self::WORKER ) ) {

			$this->workers = $this->a->get_workers_by_service( $this->set_service, $this->order_by );

			if ( $this->is_loc_active() && $this->compare( self::LOCATION, self::WORKER ) )
				$this->workers = array_intersect_key( (array)$this->workers, (array)$this->a->get_workers_by_location( $this->set_location, $this->order_by ) );

			$this->preselect_follower( 'worker' );
		} else if ( $this->is_loc_active() && $this->compare( self::LOCATION, self::WORKER ) ) {
			$this->workers = $this->a->get_workers_by_location( $this->set_location, $this->order_by );
			$this->preselect_follower( 'worker' );
		} else {
			$this->workers = $this->a->get_workers( $this->order_by );
			$this->preselect_leader( 'worker' );
		}

		$this->norm->set_worker( $this->set_worker );
	}

	/**
	 * Display errors
	 * @return string
	 */
	public function display_errors(){
		if ( empty( $this->error ) || !$this->display_errors || !WpBDebug::is_debug() ) {
			return;
		}

		$out = '<ul>';
		foreach ( array_filter( $this->error ) as $error ) {
			$out .= '<li>'. $error . '</li>';
		}
		$out .= '</ul>';
		
		$this->error = array();

		return $out;
	}

	/**
	 * Create html for locations
	 * @since 3.0
	 * @return string
	 */
	public function select_location( ) {
		$html = '<select class="app-admin-lsw app_select_locations" data-lsw="location" name="location">';

		foreach ( (array)$this->locations as $location ) {
			$sel = $this->set_location == $location->ID ? ' selected="selected"' : '';

			$html .= '<option value="'.$location->ID.'"'.$sel.'>'. $this->a->get_location_name( $location->ID ) . '</option>';
		}

		if ( ! $this->locations ) {
			$html .= '<option disabled="disabled">' . $this->a->get_text('no_free_time_slots'). '</option>';
		}
		
		$html .= '</select>';

		return $html;
	}

	/**
	 * Create html for services
	 * @param $new		bool	Package selection is allowed for new booking
	 * @since 3.0
	 * @return string
	 */
	public function select_service( $new = false ) {
		$html = '<select class="app-admin-lsw app_select_services" data-lsw="service" name="service">';

		foreach ( (array)$this->services as $service ) {
			$sel		= $this->set_service == $service->ID ? ' selected="selected"' : '';
			$disabled	= ! $new && $this->a->is_package( $service->ID ) ? ' disabled="disabled"' : '';

			$html .= '<option value="'.$service->ID.'"'.$sel.$disabled.'>'. $this->a->get_service_name( $service->ID ) . '</option>';
		}

		if ( ! $this->services ) {
			$html .= '<option disabled="disabled">' . $this->a->get_text('no_free_time_slots'). '</option>';
		}
		
		$html .= '</select>';

		return $html;
	}

	/**
	 * Create html for workers
	 * @since 3.0
	 * @return string
	 */
	public function select_worker( ) {
		$html = '<select class="app-admin-lsw app_select_workers" data-lsw="worker" name="worker">';
		
		if ( $this->workers ) {

			if ( apply_filters( 'app_controller_allow_unassigned', false, $this ) ) {
				$html .= '<option value="0">'. $this->a->get_text( 'our_staff' ) . '</option>';
			}

			foreach ( (array)$this->workers as $worker ) {
			
				if ( apply_filters( 'app_controller_skip_worker', false, $worker, $this ) ) {
					continue;
				}

				$sel = $this->set_worker == $worker->ID ? ' selected="selected"' : '';

				$html .= '<option value="'.$worker->ID.'"'.$sel.'>'. $this->a->get_worker_name( $worker->ID, false ) . '</option>';
			}

		} else if ( $this->a->get_nof_workers() ) {
			$html .= '<option value="0" selected="selected">'. $this->a->get_text( 'our_staff' ) . '</option>';
		}
		
		$html .= '</select>';

		return $html;
	}

	/**
	 * Create html of flexslider for location/service/worker selection for one slide
	 * @param $lsw			object		Location/service/worker object
	 * @param $context		string		location, service or worker
	 * @return string
	 */
	public function slider_item( $lsw, $context ) {
		$ID		 = $lsw->ID;
		$page_id = ! empty( $lsw->page ) ? $lsw->page : 0;
		
		switch ( $context ) {
			case 'location':	$name = BASE()->get_location_name( $ID );
								$current = $ID == $this->set_location ? ' current' : '';
								break;
			case 'service':		$name = BASE()->get_service_name( $ID );
								$current = $ID == $this->set_service ? ' current' : '';
								break;
			case 'worker':		$name = BASE()->get_worker_name( $ID );
								$current = $ID == $this->set_worker ? ' current' : '';
								break;
		}
		
		$desc = $sub_context = '';
		$default_img = '<img src="'. wpb_placeholder_img_src() . '" alt="'.esc_attr( $name ).'">';
		
		$s  = '<li class="app-'.$context.$current.'" data-'.$context.'="'.$ID.'" data-no="'.$this->slide_no.'" >';
		$s .= '<div class="app-gradient"></div>';
		
		if ( 'location' == $context && ! empty( $lsw->map ) ) {
			$slider_image = $lsw->map;
			$sub_context = 'map';
		} else if ( ! $page_id || apply_filters( 'app_slider_item_ignore_desc_page', false, $lsw, $context, $this ) ) {
			$slider_image = $default_img;
			
			if ( 'service' == $context ) {
				if ( $img_id = wpb_get_service_meta( $ID, 'image_id' ) ) {
					$slider_image = wp_get_attachment_image( $img_id, 'post-thumbnail', false, array( 'alt' => BASE()->get_service_name( $ID ), 'loading' => false ) );
				}
				
				if ( $maybe_desc = wpb_get_service_meta( $ID, 'description' ) ) {
					$desc = $maybe_desc;
					$sub_context = 'description';
				}
			}
			
		} else {
			$post = get_post( $page_id );
			if ( $post ) {
				$thumb = get_the_post_thumbnail( $post );
				$slider_image = $thumb ?: $default_img;
				
				if ( !empty( $post->post_excerpt ) ) {
					$desc = $post->post_excerpt;
					$sub_context = 'excerpt';
				} else {
					$desc = $post->post_content;
					$sub_context = 'content';
				}
			} else {
				$page_id = 0;
			}
		}
		
		$s .= apply_filters( 'app_slider_image', $slider_image, $lsw, $context, $this );
		$s .= '<p class="flex-caption">'. esc_html( $name );
		$s .= '<span class="flex-desc">'. wp_trim_words( apply_filters( 'app_description_text', $desc, $page_id, $ID, $context, $sub_context ) ) .'</span>';
		$s .= '</p>';
		$s .= '</li>';
		
		$this->slide_no++;
		
		return apply_filters( 'app_slider_image_html', $s, $lsw, $context, $this );
	}

	/**
	 * Utility function to check if current service lasts all day
	 * @since 3.0
	 * @return bool
	 */
	public function is_daily(){
		return $this->a->is_daily( $this->set_service );
	}

}
}