<?php
/**
 * WPB Calendar Class
 *
 * A "calendar" is a set of back-to-back time slots, usually in the range of days or weeks
 * This class handles required time range to generate HTML out of time slot availability
 *
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpB_Calendar' ) ) {

class WpB_Calendar {

	private $location,
			$service,
			$worker,
			$_time,
			$slot,
			$microtime,
			$format,
			$cl_offset;

	public $reason;

	public $cache = array();

	public $has_appointment = false;

	public $nof_free = 0;

	public $all_busy = null;

	public $has_waiting = null;

	private $disable_lazy_load,
			$disable_limit_check,
			$disable_waiting_list,
			$admin,
			$assign_worker,
			$force_assign_worker,
			$inline,
			$display,
			$hard_limit;

	private $force_min_time = 0;

	protected $a, $uniq;

	/**
	 * @param $var	mixed		WpB_Booking, WpB_Controller, WpB_Slot object, or $location string|integer
	 * @return WPB_Calendar
	 */
	public function __construct( $var = false, $service = false, $worker = false ) {
		$this->a = BASE();

		if ( $var instanceof WpB_Booking ) {
			$booking 		= $var;
			$this->location = $booking->get_location();
			$this->service 	= $booking->get_service();
			$this->worker 	= $booking->get_worker();

		} else if ( $var instanceof WpB_Controller ) {
			$controller 	= $var;
			$this->location = $controller->get_location();
			$this->service 	= $controller->get_service();
			$this->worker 	= $controller->get_worker();

		} else if ( $var instanceof WpB_Slot ) {
			$slot		 	= $var;
			$this->location = $slot->get_location();
			$this->service 	= $slot->get_service();
			$this->worker 	= $slot->get_worker();

		} else {
			$location 		= $var;
			$this->location = $location;
			$this->service 	= $service;
			$this->worker 	= $worker;

			if ( 'all' === $this->worker ) {
				$this->worker = $this->a->get_def_wid();
				$this->service = 'all';
				add_filter( 'app_get_duration', array( $this, 'force_duration' ), WPB_HUGE_NUMBER, 3 );
				add_filter( 'app_is_daily', array( $this, 'force_is_daily' ), WPB_HUGE_NUMBER, 2 );
			}

			# If no workers is defined, business rep. is the worker
			if ( ! $this->worker && ! $this->a->get_nof_workers() ) {
				$this->worker = $this->a->get_def_wid();
			}
		}

		$this->_time		= $this->a->_time;
		$this->cl_offset	= $this->is_daily() ? 0 : $this->a->get_client_offset( );

		$this->start_hard_limit_scan();	# Calendar call can last for limited time
	}

	/**
	 * Preset properties
	 * @return none
	 */
	public function setup( $args ) {
		$this->disable_lazy_load	= !empty( $args['disable_lazy_load'] );
		$this->disable_limit_check	= !empty( $args['disable_limit_check'] );	# Not used
		$this->disable_waiting_list = !empty( $args['disable_waiting_list'] );
		$this->admin				= !empty( $args['admin'] );
		$this->assign_worker		= !empty( $args['assign_worker'] );
		$this->inline				= !empty( $args['inline'] );
		$this->force_min_time		= !empty( $args['force_min_time'] ) ? (int)$args['force_min_time'] : 0;	# Forces time base as set amount (Usually 60 for 60 mins)
		$this->display				= !empty( $args['display'] ) ? $args['display'] : '';					# full, with_break or minimum
		$this->hard_limit			= !empty( $args['hard_limit'] ) ? (float)$args['hard_limit'] : null;

		if ( $this->assign_worker && $this->a->get_nof_workers() && ! $this->worker ) {
			$this->force_assign_worker = true;
		}
	}

	/**
	 * Display mode of the calendar
	 * @return string (full, with_break or minimum)
	 */
	public function get_display(){
		return apply_filters( 'app_calendar_display', $this->display, $this );
	}

	/**
	 * Set a duration for 'all' service case
	 * @return integer
	 */
	public function force_duration( $duration, $slot ) {
		if ( 'all' === $slot->get_service() ) {
			$duration = 60;
		}

		return $duration;
	}

	/**
	 * Reset is_daily for 'all' service case
	 * @return integer
	 */
	public function force_is_daily( $result, $ID ) {
		if ( 'all' === $ID ) {
			$result = false;
		}

		return $result;
	}

	/**
	 * Determine if service of calendar lasts all day
	 * @since 3.0
	 * @return bool
	 */
	public function is_daily(){
		return $this->a->is_daily( $this->service );
	}

	/**
	 * Getters
	 * @return integer
	 */
	public function get_location(){
		return $this->location;
	}

	public function get_service( ){
		return $this->service;
	}

	public function get_worker( ) {
		return $this->worker;
	}

	/**
	 * Check if inline
	 * @return bool
	 */
	public function is_inline(){
		return ! empty( $this->inline );
	}

	/**
	 * Check if used on admin side
	 * @return bool
	 */
	public function is_admin(){
		return $this->admin;
	}

	/**
	 * Check if service is a package
	 * @return bool
	 */
	public function is_package() {
		return $this->a->is_package( $this->service );
	}

	/**
	 * Check if waiting list enabled
	 * @return bool
	 */
	public function is_waiting_list_allowed(){
		return ! $this->disable_waiting_list;
	}

	/**
	 * Record microtime value to check scanning time
	 * @return none
	 */
	public function start_hard_limit_scan(){
		$this->microtime = wpb_microtime();
	}

	/**
	 * Hard limit time in seconds
	 * @return float
	 */
	private function hard_limit(){
		return apply_filters( 'app_hard_limit', $this->hard_limit ? $this->hard_limit : WPB_HARD_LIMIT );
	}

	/**
	 * Check if scanning time exceeded
	 * @return float|false		Total scan time if exceeded, false if not
	 */
	public function is_hard_limit_exceeded(){
		$scan_time = wpb_microtime() - $this->microtime;

		if ( $scan_time > $this->hard_limit() ) {
			return $scan_time;
		}

		return false;
	}

	/**
	 * Get when last scan started
	 * @return float
	 */
	public function scan_start_time() {
		return $this->microtime;
	}

	/**
	 * Create a Slot object
	 * @return object
	 */
	public function slot( $slot_start, $slot_end = false ) {
		return $this->slot = new WpB_Slot( $this, wpb_strtotime($slot_start), ($slot_end ? wpb_strtotime($slot_end) : false), false, false, $this->cl_offset );
	}

	/**
	 * Find the start and end points for the day
	 * @return array
	 */
	public function limits( $day_start ) {
		$raw_first = null;

		if ( ( 'all' === $this->service && !$this->inline ) || 'full' === $this->get_display() || $this->is_daily() ) {
			$_start = 0;
			$_end	= 24*60;
		} else {
			if ( !$this->cl_offset && !$this->is_package() ) {
				$limits	= BASE('WH')->find_limits( $this->slot( $day_start ) );
				if ( date( 'Y-m-d', $day_start ) == date( 'Y-m-d', $this->_time ) ) {
					$_start	= 0;
					$raw_first	= $limits['first'] *60 + $day_start;
				} else {
					$_start	= $limits['first'];
				}
				$_end	= $limits['last'];
			} else {
				$_start = 0;
				$_end	= 24*60;
			}
		}

		$start	= apply_filters( 'app_schedule_starting_hour', $_start, $day_start, $this );
		$end	= apply_filters( 'app_schedule_ending_hour', $_end, $day_start, $this );

		$first	= $start *60 + $day_start; 	# Timestamp of the first cell to be scanned in a day
		$last	= $end *60 + $day_start; 	# Last cell as ditto

		return array( $first, $last, $raw_first );
	}

	/**
	 * Check if service capacity has been increased, e.g. There are 3 providers but capacity set as 5
	 * @return bool
	 */
	private function is_capacity_increased(){
		$capacity	= $this->a->get_capacity( $this->service );
		$workers	= (array)$this->a->get_worker_ids_by_service( $this->service );
		$n			= $capacity - count( $workers ); # Net capacity increase or decrease

		return ( $n > 0 );
	}

	/**
	 * For services lasting 12 to 24 hours, limit duration to check as 12 hours
	 * @return integer (in seconds)
	 */
	private function instant_duration( $slot_start_current ) {
		$dur = wpb_get_duration( $this->slot( $slot_start_current ) );
		if ( $dur < 24 * 60 && $dur > 12 * 60 ) {
			return 43200;
		}

		return $dur * 60;
	}

	/**
	 * Find free or all slots in a day
	 * Several optimizations are used here for faster processing
	 * @param $day_start	Integer	Timestamp of the day we are scanning
	 * @param $find_one		Bool	Checking for just one or more free slot(s). Number entered is number of free slots we are looking for
	 * 								This is an optimization for monthly calendar.
	 * @return array				Keys are $slot_start, values are WpB_Slot object
	 */
	public function find_slots_in_day( $day_start, $find_one = false ) {
		$this->all_busy			= null;
		$this->has_appointment 	= false;
		$this->has_waiting		= false;
		$out 					= array();
		$this->nof_free			= 0;

		if ( apply_filters( 'app_skip_day', false, $day_start, $this ) ) {
			return $out;
		}

		$display		= $this->get_display();
		$is_legacy		= ('yes' == apply_filters( 'app_time_slot_calculus_legacy', wpb_setting( 'time_slot_calculus_legacy' ), $this ) );
		$apply_fix		= $this->force_min_time || $is_legacy; # Whether apply fix time
		$step_min		= $this->force_min_time ? $this->force_min_time*60 : $this->a->get_min_time()*60; # Either min time or forced time
		$step_svs		= $apply_fix ? $step_min : $this->instant_duration( $day_start );

		$assign_final	= $this->assign_worker && !$this->is_capacity_increased();

		$step = $step_svs; # First apply the service duration as step - then slow down if required

		list( $first, $last, $raw_first ) = $this->limits( $day_start ); # $raw_first > 0 shows this is today and $first is shifted down to 00:00

		if ( $this->inline && $raw_first ) {
			$first = $raw_first;
		}

		$i = 1;
		for ( $t = $first; $t < $last; $t = $t + $step ) {

			if ( ! $step ) {
				break;
			}

			$reason				= null;
			$is_longer_break	= true;
			$slot_start			= $t - $this->cl_offset;
			$dur_in_secs		= $this->instant_duration( $slot_start ); # This is here for variable durations Addon
			$step_svs			= $apply_fix ? $step_min : $dur_in_secs;

			if ( $this->inline ) {
				$out[$slot_start] = new WpB_Slot( $this, $slot_start, $slot_start + $step );
				continue;
			}

			if ( $this->slot( $slot_start )->is_past_today( ) ) {
				continue;
			}

			if ( $step_svs != $step_min ) {
				$slot = new WpB_Slot( $this, $slot_start, $slot_start + $step, false, $assign_final, $this->cl_offset );
				if ( $reason = $slot->why_not_free() ) {
					if ( $step != $step_svs ) {
						# Increase to normal service length
						$slot = new WpB_Slot( $this, $slot_start, $slot_start + $step_svs, false, $assign_final, $this->cl_offset );
						if ( $maybe_reason = $slot->why_not_free() ) {
							$reason = $maybe_reason;
						}
					}
				}

				# We are using integers here to save more in DB cache
				$is_past = 12 === $reason || 13 === $reason;
				$is_longer_break = !$this->cl_offset && (5 === $reason || 6 === $reason || 11 === $reason || 20 === $reason || $is_past ); # wpb_code2reason function for explanation of these numbers

				if ( ! $reason || $is_longer_break || 20 === $reason || ($this->cl_offset && ( $is_past || 11 === $reason ) ) ) {
					$step = $step_svs;
				} else {
					$step = $step_min;
				}
			}

			if ( null === $reason || ( false === $reason && $step_svs != $step_min ) ) {
				$slot = new WpB_Slot( $this, $slot_start, $slot_start + $step, false, $assign_final, $this->cl_offset );

				$reason = $slot->why_not_free( );

				if ( ! $reason && $step != $dur_in_secs ) {
					# Increase from $step (must be equal to $step_min) to $step_svs to find if there is a reason
					$slot = new WpB_Slot( $this, $slot_start, $slot_start + $dur_in_secs, false, $assign_final, $this->cl_offset );
					$reason = $slot->why_not_free( );
				}
			}

			if ( $this->force_assign_worker && $assign_final && !$reason && !$slot->get_worker() ) {
				$reason = 2; # Cannot assign: No workers
			}

			if ( $slot->has_appointment ) {
				$this->has_appointment = true;
			}

			if ( $reason ) {

				if ( ! $is_longer_break ) { # Prevents slowing down too early
					$step = $step_min;
				}

				if ( $this->is_hard_limit_exceeded() ) {	# Prevent too long scans
					$hard_limit = true;
					break;
				}

				if ( 0 === ( ($slot_start - $first) % $step_svs ) && ( 'full' === $display || ( 'with_break' === $display && ( empty( $raw_first ) || $slot_start >= $raw_first) ) ) ) {
					$out[ $slot_start ] = $slot;	# If we also want not available slots with reason in $slot->reason
				}

				if ( 1 === $reason && null === $this->all_busy ) {
					$this->all_busy = true;		# All slots are busy (or unavailable, but not free)
				}

				if ( 20 === $reason ) {
					$out[ $slot_start ] = $slot;
					$this->has_waiting = true;
					$slot->is_waiting = true;
				}

				if ( 9 == $reason || 10 == $reason || $this->is_daily() ){ # If full day, any reason is enough to break
					break;
				} else {
					continue;
				}
			}

			if ( ! $reason ) {
				$this->all_busy = false; # We found a free slot, so not all slots are busy
			}

			$out[ $slot_start ] = $slot;

			$this->nof_free	++;

			if ( $find_one && $this->nof_free >= $find_one ) {
				if ( ! $this->admin || $this->has_appointment ) {
					break;
				}
			}
		}

		# Get rid of the last notpossible
		if ( $out && 'full' !== $display ) {
			end( $out );
			$slot_last = current( $out );
			if ( $slot_last->reason && 1 !== $slot_last->reason && 20 !== $slot_last->reason ) {
				array_pop( $out );
			}
		}

		if ( ! empty( $reason ) ) {
			$this->reason = $reason;
		}

		return $out;
	}

	/**
	 * Find available days in a time interval
	 * @param $from		string|integer		Start time as Timestamp or date/time. Includes the day $from belongs from 00:00
	 * @param $to		string|integer		End time to check as timestamp or day/time. Includes the day $to belongs until midnight
	 * @since 3.0
	 * @return array						Values are days in Y-m-d format
	 */
	public function find_available_days( $from, $to ) {
		$out	= array();
		$start	= strtotime( date( 'Y-m-d', wpb_strtotime( $from ) ), $this->a->_time );
		$end	= strtotime( date( 'Y-m-d', wpb_strtotime( $to ) ), $this->a->_time );

		for ( $d = $start; $d < $end; $d = $d + DAY_IN_SECONDS ) {

			$maybe_found_day = $this->find_slots_in_day( $d, 1 );

			if ( ! empty( $maybe_found_day ) ) {
				$out[] = date( 'Y-m-d', $d );
			}

			if ( $this->is_hard_limit_exceeded() ) {
				break;
			}
		}

		return array_values( $out );
	}

	/**
	 * Wraps Timetable cells with required div elements
	 * @since 3.0
	 * @return string
	 */
	public function wrap_cells( $day_start, $html ) {
		$time = ! empty( $_REQUEST["app_timestamp"] ) ? wpb_clean( $_REQUEST["app_timestamp"] ) + $this->cl_offset : $this->_time + $this->cl_offset;
		$style = date( 'Ymd', $day_start ) == date( 'Ymd', $time ) ? '' : ' style="display:none"';

		$ret  = '<div class="app-timetable app_timetable_'.$day_start.'"'.$style.'>';
		$ret .= '<div class="app-timetable-title ui-state-default">';
		$ret .= apply_filters( 'app_timetable_title', date_i18n( $this->a->date_format, $day_start ), $day_start, $this );

		$ret .= '</div>';
		$ret .= '<div class="app-timetable-flex" data-nof-cells="'.substr_count( $html, 'app-timetable-cell' ).'">';

		$ret .=	$html;

		$ret .= '</div></div>';

		return $ret;
	}

	/**
	 * Check if lazy load will be executed
	 * Lazy load is not run for Monthly Calendar widget, on admin side
	 * @since 3.0
	 * @return bool
	 */
	public function is_lazy_load_allowed( ) {
		$allowed = true;

		if ( $this->disable_lazy_load ) {
			$allowed = false;
		}

		if ( 'yes' != wpb_setting( 'lazy_load' ) || is_admin() || wpb_is_account_page() ) {
			$allowed = false;
		}

		return apply_filters( 'app_is_lazy_load_allowed', $allowed, $this );
	}

	/**
	 * Check if lazy load is in progress
	 * Lazy load will run if it is allowed and not ajax
	 * @since 3.0
	 * @return bool
	 */
	public function doing_lazy_load(){
		return $this->is_lazy_load_allowed( ) && ! defined( 'WPB_AJAX' );
	}

	/**
	 * Check if caching will be used - DEPRECATED
	 * @since 3.5.6
	 * @until 3.9.1.2
	 * @return bool
	 */
	public function use_cache(){
		return false;
	}

	/**
	 * Add previous rolling and current caches and saves them - DEPRECATED
	 * Also combines separate calendar caches on the same page
	 * @since 3.0
	 * @until 3.9.1.2
	 * @return none
	 */
	public function save_cache() {
		return;
	}

}
}