<?php
/**
 * WPB Assets
 *
 * Handles loading of scripts and styles
 *
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpBAssets' ) ) {

class WpBAssets {

	private $localized;

	/**
     * WP BASE Core + Front [+Admin] instance
     */
	protected $a = null;

	/**
     * Current Post ID, Shopping Cart contents, Remaining time to finish shopping
     */
	private $current_post_id = 0;
	private $cart_values = false;
	private $remaining_time = false;

	/**
     * Constructor
     */
	public function __construct() {
		$this->a = BASE();
	}

	public function add_hooks(){
		add_action( 'admin_enqueue_scripts', array( $this, 'register' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'register' ) );

		add_action( 'admin_enqueue_scripts', array( $this, 'load_admin' ) );

		add_action( 'app_shortcode_found', array( $this, 'request_load' ) );
		add_action( 'app_load_assets', array( $this, 'request_load' ) );
		
		add_filter( 'do_shortcode_tag', array( $this, 'do_shortcode_tag' ), 10, 2 );
		
		add_action( 'template_redirect', array( $this, 'template_redirect' ) );
	}
	
	/**
     * Find post object for non-single posts
	 * Get cart values and remaining time before headers sent
     */
	public function template_redirect() {
		global $wp_the_query;
		
		if ( ! is_callable( array( $wp_the_query, 'get_queried_object' ) ) ) {
			return;
		}
		
		$current_object = $wp_the_query->get_queried_object();
		
		if ( empty( $current_object ) ) {
			return;
		}
		
		$this->current_post_id = ! empty( $current_object->ID ) ? $current_object->ID : 0;

		if ( ! headers_sent() ) {
			$this->cart_values = BASE('Multiple')->values();
			$this->remaining_time = BASE('Multiple')->remaining_time();
		}
	}

	/**
     * Request loading of front assets
     */
	public function request_load(){
		if ( has_action( 'wp_enqueue_scripts', array( $this, 'load_front' ) ) ) {
			return;
		}

		add_action( 'wp_enqueue_scripts', array( $this, 'load_front' ), 20 );
		add_action( 'wp_enqueue_scripts', array( $this, 'load_front_css' ), 20 );

		do_action( 'app_request_load' );
	}

	/**
     * Get WPB version
	 * @return string
     */
	private function ver(){
		return $this->a->version;
	}

	/**
     * Find if debug
	 * In debug "dev" files are loaded
	 * @return bool
     */
	private static function is_debug(){
		return WpBDebug::is_debug() && ! is_admin();
	}

	/**
     * Register all scripts
     */
	public function register(){
		wp_register_script( 'printomatic-js', WPB_PLUGIN_URL . '/js/admin/printomat.js', array('jquery'), $this->ver() );
		wp_register_script( 'codemirror', WPB_PLUGIN_URL . '/js/admin/codemirror.min.js', array(), $this->ver());
		wp_register_script( 'flexslider', WPB_PLUGIN_URL . '/js/dev/jquery.flexslider.js', array('jquery'), $this->ver() );
		wp_register_script( 'isotope', WPB_PLUGIN_URL . '/js/dev/isotope.pkgd.min.js', array('jquery'), $this->ver() );
		wp_register_script( 'jquery-app-qtip', WPB_PLUGIN_URL . '/js/dev/jquery.qtip.min.js', array('jquery'), $this->ver() );
		wp_register_script( 'jquery-blockui', WPB_PLUGIN_URL . '/js/dev/jquery.blockUI.js', array('jquery'), $this->ver() );
		wp_register_script( 'wp-base-countdown', WPB_PLUGIN_URL . '/js/dev/jquery.countdown.min.js', array('jquery','jquery-plugin'), $this->ver() );
		wp_register_script( 'jquery-datatables', WPB_PLUGIN_URL . '/js/dev/jquery.dataTables.min.js', array('jquery-ui-core'), $this->ver() );
		wp_register_script( 'jquery-datatables-jqueryui', WPB_PLUGIN_URL . '/js/dev/dataTables.jqueryui.min.js', array('jquery-datatables'), $this->ver() );
		wp_register_script( 'jquery-datatables-jqueryui-responsive', WPB_PLUGIN_URL . '/js/dev/responsive.jqueryui.min.js', array('jquery-datatables-jqueryui','jquery-datatables-responsive'), $this->ver() );
		wp_register_script( 'jquery-datatables-moment', WPB_PLUGIN_URL . '/js/dev/moment.min.js', array('jquery-datatables'), $this->ver() );
		wp_register_script( 'jquery-datatables-responsive', WPB_PLUGIN_URL . '/js/dev/dataTables.responsive.min.js', array('jquery-datatables'), $this->ver() );
		wp_register_script( 'jquery-plugin', WPB_PLUGIN_URL . '/js/mobile/jquery.plugin.min.js', array('jquery'), $this->ver() );
		wp_register_script( 'jquery-qtip', WPB_PLUGIN_URL . "/js/dev/jquery.qtip.min.js", array('jquery'), $this->ver() );
		wp_register_script( 'jquery-quickfit', WPB_PLUGIN_URL . '/js/dev/jquery.quickfit.js', array('jquery'), $this->ver() );
		wp_register_script( 'jquery-scrollto', WPB_PLUGIN_URL . '/js/dev/jquery.scrollTo.min.js', array('jquery'), $this->ver() );
		wp_register_script( 'jquery-ui-multidatespicker', WPB_PLUGIN_URL . '/js/dev/jquery-ui.multidatespicker.js', array('jquery-ui-datepicker'), $this->ver() );
		wp_register_script( 'signature-pad', WPB_PLUGIN_URL . '/js/dev/signature_pad.min.js', array(), $this->ver() );
		wp_register_script( 'swipe', WPB_PLUGIN_URL . '/js/mobile/swipe.js', array(), $this->ver() );
		wp_register_script( 'jstz', WPB_PLUGIN_URL . '/js/dev/jstz.min.js', array(), $this->ver() );
		wp_register_script( 'wp-base-common-scripts', WPB_PLUGIN_URL . '/js/common-scripts.js', array('jquery-ui-core','jquery-ui-widget', 'jquery-ui-position'), $this->ver(), true );
		wp_register_script( 'wp-base-libs', WPB_PLUGIN_URL . '/js/libs.js', array('jquery-ui-widget','jquery-ui-button','jquery-ui-datepicker'), $this->ver(), true );
	}

	/**
     * Register moment locale for non English websites
	 * @since 3.6.2
	 * @return string		Handle of the script to be used as dependency
     */
	private function register_moment_locale() {
		$locale 		= strtolower( $this->a->get_locale() );
		$locale_short 	= current( explode( '-', $locale ) );

		if ( $locale && 'en' != $locale && 'en-us' != $locale ) {
			foreach( array( $locale, $locale_short ) as $lcl ) {
				if ( file_exists( WPBASE_PLUGIN_DIR . '/js/locale/'.$lcl.'.js' ) ) {

					wp_register_script( 'jquery-moment-locale',
						WPB_PLUGIN_URL . '/js/locale/'.$lcl.'.js',
						(self::is_debug() ? array('jquery-datatables-moment') : array('wp-base-libs')),
						$this->ver()
					);

					return 'jquery-moment-locale';
				}
			}
		}
	}

	/**
	 * Enqueue used jQuery effects
	 * @since 3.0
	 */
	public function enqueue_effects(){
		$effects 	= array( 'drop' );
		$effects[] 	= wpb_setting( 'hide_effect' );
		$effects[] 	= wpb_setting( 'show_effect' );
		foreach ( array_unique( array_filter( $effects ) ) as $e ) {
			wp_enqueue_script( 'jquery-effects-' . $e );
		}
	}

	/**
     * Admin script dependencies
     */
	private function deps_admin() {

		$deps = array(
			'wp-base-common-scripts',
			'codemirror',
			'jquery-ui-button',
			'jquery-ui-datepicker',
			'jquery-ui-dialog',
			'jquery-ui-sortable',
			'jquery-ui-tabs',
			'wp-color-picker',
			'wp-base-libs',
		);

		# FEBM may call on front end - these are only registered for admin, so we exclude them
		if ( ! is_admin() ) {
			$deps = array_diff( $deps, array( 'wp-color-picker' ) );
		}

		return $deps;
	}

	/**
     * Front end script dependencies
     */
	private function deps_front() {
		if ( self::is_debug() ) {
			return array(
				'wp-base-common-scripts',
				'jquery-blockui',
				'wp-base-countdown',
				'jquery-datatables-jqueryui',
				'jquery-ui-button',
				'isotope',
				'jquery-datatables-jqueryui-responsive',
				'jquery-datatables-moment',
				'jquery-quickfit',
				'jquery-scrollto',
				'signature-pad',
				'flexslider',
				'jquery-effects-drop',
				'jquery-ui-button',
				'jquery-ui-dialog',
				'jstz',
			);
		} else {
			return array(
				'wp-base-common-scripts',
				'wp-base-libs',
				'jquery-effects-drop',
				'jquery-ui-button',
				'jquery-ui-dialog',
			);
		}
	}

	/**
     * Common data for javascript
     */
	public function common_data() {
		global $wp_locale;
		return array(
			'menuHeight'			=> 'size',
			'checkAll'				=> __( 'Check all', 'wp-base' ),
			'uncheckAll'			=> __( 'Clear all', 'wp-base' ),
			'ajax_url'				=> admin_url('admin-ajax.php'),
			'all'					=> $this->a->get_text( 'all' ),
			'booking'				=> $this->a->get_text( 'booking' ),
			'cancel_nonce'			=> wp_create_nonce('cancel-app'),
			'cancelled'				=> $this->a->get_text( 'cancelled' ),
			'calculating'			=> $this->a->get_text( 'calculating' ),
			'checkout'				=> $this->a->get_text( 'checkout' ),
			'choose_image'			=> $this->a->get_text( 'choose_image' ),
			'confirm_nonce'			=> wp_create_nonce('confirm-app'),
			'confirm_vendor'		=> $this->a->get_text( 'mv_become_vendor_confirm' ),
			'confirmCancel'			=> $this->a->get_text( 'confirm_cancel' ),
			'confirmConfirm'		=> $this->a->get_text( 'confirm_confirm' ),
			'confirmDelete'			=> $this->a->get_text( 'confirm_delete' ),
			'confirmed'				=> $this->a->get_text( 'confirmed' ),
			'confirmed_message'		=> $this->a->get_text( 'confirmed_message' ),
			'con_error'				=> $this->a->get_text('connection_error'),
			'curr_decimal'			=> wpb_setting('curr_decimal'),
			'daily_text'			=> __('Daily', 'wp-base'),
			'datatable_args'		=> '',
			'dayNamesMin'       	=> array_values( $wp_locale->weekday_initial ),
			'decimal_sep'			=> wpb_decimal_separator(),
			'done'					=> $this->a->get_text( 'done' ),
			'filterLabel'			=> '',						// If left empty "Filter:"
			'filterPholder'			=> '',						// If left empty "Enter keywords"			
			'js_date_format'		=> wpb_convert_dt_format( $this->a->safe_date_format() ),
			'locale'				=> strtolower( $this->a->get_locale() ),
			'length_menu'			=> $this->a->get_text( 'length_menu' ),
			'loading'				=> $this->a->get_text( 'loading' ),
			'logging_in'			=> $this->a->get_text( 'logging_in' ),
			'moment_format'			=> wpb_moment_format(),
			'monthNamesShort'   	=> array_values( $wp_locale->month_abbrev ),
			'next'					=> $this->a->get_text( 'next' ),
			'preparing_form'		=> $this->a->get_text( 'preparing_form' ),
			'preparing_timetable'	=> $this->a->get_text( 'preparing_timetable' ),
			'previous'				=> $this->a->get_text('previous'),
			'qtipDef'				=> array( 'widget' => true, 'def' => false, 'classes' => 'app-qtip' ),
			'qtipHide'				=> array( 'fixed' => true, 'delay' => 300 ),
			'qtipNarrow'			=> array( 'widget' => true, 'def' => false, 'classes' => 'app-narrow-qtip' ),
			'qtipPos'				=> is_rtl() ? array( 'my' => 'center right', 'at' => 'center left' ) : array( 'my' => 'center left', 'at' => 'center right', 'collision' => 'flipfit flipfit' ),
			'qtipPosBottom'			=> array( 'my' => 'top center', 'at' => 'bottom center' ),
			'qtipPosLeft'			=> is_rtl() ? array( 'my' => 'center left', 'at' => 'center right' ) : array( 'my' => 'center right', 'at' => 'center left' ),
			'qtipPosTop'			=> array( 'my' => 'bottom center', 'at' => 'top center' ),
			'qtipSmall'				=> array( 'widget' => true, 'def' => false, 'classes' => 'app-small-qtip' ),
			'qtipYellow'			=> array( 'widget' => false, 'def' => true, 'classes' => 'app-yellow-qtip' ),
			'reading'				=> $this->a->get_text( 'reading' ),
			'refreshing'			=> $this->a->get_text( 'refreshing' ),
			'saving'				=> $this->a->get_text( 'saving' ),
			'start_of_week'			=> $this->a->start_of_week,
			'thousands_sep'			=> wpb_thousands_separator(),
			'update_text'			=> __('Update','wp-base'),
			'updating_text'			=> $this->a->get_text( 'updating' ),
			'user_fields'			=> $this->a->get_user_fields(),
			'is'					=> array(
										'mobile' 	=> (bool)wpb_is_mobile(),
										'rtl'		=> is_rtl(),
										'eb'		=> (bool)wpb_eb(),
										'pro'		=> class_exists( 'WpBPro' ),
										'account'	=> (bool)wpb_is_account_page(),
										'msUsers'	=> true, # Use multiselect for Users dropdown
									),

		);
	}

	/**
     * Admin data for javascript
     */
	public function admin_data(){
		return array(
			'cssFile'			=>  array( WPB_PLUGIN_URL . '/css/admin.css', WPB_PLUGIN_URL . '/css/dataTables.jqueryui.css' ),
			'colorPresets'		=> wpb_get_preset(),
			'confirmDeleteLog'	=> __( 'Are you sure to clear the log file?', 'wp-base' ),
			'confirmEmpty'		=> __( 'You are about to delete at least one record. Are you sure to do this?', 'wp-base' ),
			'confirmReset'		=> __( 'WARNING!! This action will clear all existing database records (bookings, transactions, locations, services, service providers, working hours). Are you sure to do this?', 'wp-base' ),
			'confirmResetAgain'	=> __( 'Are you REALLY SURE TO DELETE the database records?', 'wp-base' ),
			'confirmRestore'	=> __( 'This action will restore all WP BASE settings to the defaults. Database records (bookings, transactions, locations, services, service providers, working hours) will not be changed. Are you sure to do this?', 'wp-base' ),
			'confirmTest'		=> __( 'Creating of test appointments may take several minutes. Do you want to continue?', 'wp-base' ),
			'confirmVendorStat' => __( 'You are about to change status of an approved vendor as pending or declined. This action will suspend the vendor.', 'wp-base' ),
			'deleteLogNonce'	=> wp_create_nonce('delete_log'),
			'dialogAppNo'		=> __( 'Booking #','wp-base' ),
			'dialogCancel'		=> __( 'You did not save the changes you have made. Are you sure?', 'wp-base' ),
			'dialogEdit'		=> __( 'Edit','wp-base' ),
			'dialogTitle'		=> __( 'Add Manual Payment','wp-base' ),
			'dialogTrNo'		=> __( 'Record #','wp-base' ),
			'dialogUpdate'		=> __( 'Update','wp-base' ),
			'eximAPlus'			=> __( 'WP BASE SERVICE TABLE records will be cleared! Existing WP BASE SERVICE PROVIDER TABLE records will be replaced with Appointments+ values. Non-matching imported records will be appended to the existing ones. Are you sure to do this?', 'wp-base' ),
			'eximDB'			=> __( 'WARNING!! This action will clear all existing database records (booked appointments, transactions, locations, services, service providers, working hours) and new values will be written to the database. Are you sure to do this?', 'wp-base' ),
			'eximNoColumn'		=> __( 'Please select at least one column', 'wp-base' ),
			'eximNoFile'		=> __( 'First select file to be imported', 'wp-base' ),
			'eximSettings'		=> __( 'This action will replace all existing settings with those inside the imported file. Database records (bookings, transactions, locations, services, service providers, working hours) will not be changed. Are you sure to do this?', 'wp-base' ),
			'gcalAuth'			=> __( 'WP BASE requires your one time consent to access to your calendar account in order to read and write events offline (even when you are not available). With this action, you will be redirected to a consent screen. Please authorize WP BASE for offline access. Then you will be redirected back to this page.', 'wp-base' ),
			'gcalReset'			=> __( 'This action will clear internal authorization token and you will need to authorize again. Use it if configuration could not be proceeded as described.', 'wp-base' ),
			'highLight'			=> 5000,
			'iedit_extend'		=> array(),
			'iedit_nonce'		=> wp_create_nonce('inline_edit'),
			'legendItems'		=> $this->a->get_legend_items(),
			'less'				=> __( 'Less', 'wp-base' ),
			'logFileCleared'	=> __( 'Log file cleared...', 'wp-base' ),
			'maybeEdited'		=> $this->a->get_text( 'maybe_edited' ),
			'more'				=> __( 'More', 'wp-base' ),
			'scheduleClick'		=> __( 'This sample is just an approximation of the actual working hours and it is not editable. To edit, use Working Hours tab.', 'wp-base' ),
			'step'				=> intval( $this->a->get_min_time()/5 ),
			'supportInfo'		=> __( 'As an attachment to your message, your current settings and plugin related database tables, plus data of the posts and users who are related to the plugin (service providers and clients) will also be sent. Transaction data will NEVER be included. All of your data will be kept confidential. Click Ok to proceed.', 'wp-base' ),
			'takeover'			=> __( 'This record is currently being edited by another user. If you take over, his work may be lost. Are you sure to take over?', 'wp-base' ),
			'timeZone'			=> __( 'You have selected a timezone other than server timezone. Are you sure to save working hours using this timezone?', 'wp-base' ),
			'whCopied'			=> __( 'Copied - Click to release', 'wp-base' ),
			'whCopy'			=> __( 'Copy to clipboard', 'wp-base' ),
			'whPaste'			=> __( 'Paste data of WHOSE', 'wp-base' ),
			'whPasted'			=> __( 'Undo', 'wp-base' ),
			'whWhose'			=> __( 'copied table', 'wp-base' ),
			'is'				=> array(
									'lockEnabled'	=> true, 	# Interlocking of service and worker records
									'useFilter'		=> false,	# User filter for Multiselect
								),
		);
	}

	/**
     * Front data for javascript
     */
	public function front_data(){
		global $current_screen, $wp_locale, $post;

		return array(
			'active_gateways'				=> count( wpb_active_gateways() ),
			'allow_focus'					=> wpb_is_ios() ? 0 : 1,	// Focus in iOS may cause issue
			'app-terms-field-entry-warning'	=> $this->a->get_text('missing_terms_check'),
			'app_select_extras-warning'		=> $this->a->get_text('missing_extra'),
			'blink_starts'					=> 60,						// Countdown warning blink starts at (secs)
			'bookTableEdge'					=> 480, 					// At which px width 4 columns of book table reduced to 2
			'cancel'						=> $this->a->get_text('cancel'),
			'cancel_app_confirm'			=> $this->a->get_text('cancel_app_confirm'),
			'cancel_confirm_no'				=> $this->a->get_text('cancel_confirm_no'),
			'cancel_confirm_text'			=> $this->a->get_text('cancel_confirm_text'),
			'cancel_confirm_yes'			=> $this->a->get_text('cancel_confirm_yes'),
			'cart_values'					=> $this->cart_values,
			'cc_legend'						=> $this->a->get_text('cc_form_legend'),
			'checkout_button_tip'			=> $this->a->get_text('checkout_button_tip'),
			'click_hint'					=> $this->a->get_text('click_to_book'),
			'click_to_remove'				=> $this->a->get_text('click_to_remove'),
			'close'							=> $this->a->get_text('close'),
			'countdown_format'				=> 'hMS',
			'countdown_pl'					=> array('Years','Months','Weeks','Days',$this->a->get_text('hours'),$this->a->get_text('minutes'),$this->a->get_text('seconds') ),
			'countdown_sin'					=> array('Year','Month','Week','Day',$this->a->get_text('hour'),$this->a->get_text('minute'),$this->a->get_text('second') ),
			'current_user_id'				=> get_current_user_id(),
			'defTS'							=> ! empty( $_GET['app_timestamp'] )
											   ? wpb_strtotime( urldecode( wpb_clean( $_GET['app_timestamp'] ) ) )
											   : 0,
			'dialog_offset'					=> 100,						// Dialog offset from window top in px (should be positive)
			'edit_nonce'					=> wp_create_nonce('edit'),
			'edited'						=> $this->a->get_text('appointment_edited'),
			'effect_speed'					=> 700,
			'error'							=> $this->a->get_text('error'),
			'error_short'					=> $this->a->get_text('error_short'),
			'facebook'						=> $this->a->get_text('login_with_facebook'),
			'flex_modes'					=> array( 'fitHeightsPerRow', 'fitHeights', 'fitRows', 'moduloColumns', 'fitColumns', 'fitColumnsTitleTop'),
			'gg_client_id'					=> wpb_setting('google-client_id'),
			'google'						=> $this->a->get_text('login_with_google'),
			'hideEffect'					=> array( 'effect' => wpb_setting( 'hide_effect' ), 'duration' => 700, 'direction' => 'up' ),
			'info'							=> $this->a->get_text('info'),
			'localhost'						=> wpb_is_localhost() ? 1 : 0,
			'logged_in'						=> $this->a->get_text('logged_in'),
			'login_debug'					=> BASE('Login') && BASE('Login')->get_debug_text(),
			'login_methods'					=> array_filter(explode( ',', wpb_setting('login_methods' ) )),
			'login_not_met'					=> 'yes' === wpb_setting('login_required') && !is_user_logged_in() ? 1 : 0,
			'login_url'						=> wp_login_url(),
			'map_width'						=> wpb_setting( 'gmap_width', 0 ),
			'nonce'							=> wp_create_nonce('front'),
			'no_gateway'					=> $this->a->get_text('payment_method_error'),
			'offset'						=> 200,											// ScrollTo offset from window top in px (should be positive)
			'opacity'						=> '0.6',										// General opacity, e.g. a button is selected. String is ok
			'password'						=> $this->a->get_text('password'),
			'pay_now'						=> $this->a->get_text('pay_now'),
			'please_wait'					=> $this->a->get_text('please_wait'),
			'post_id'						=> ! empty( $this->current_post_id ) 
											   ? $this->current_post_id
											   : (!empty( $post->ID ) ? $post->ID : 0),
			'popup_id'						=> 0,
			'received'						=> $this->a->get_text('appointment_received'),
			'redirect'						=> $this->a->get_text('redirect'),
			'refresh_url'					=> wpb_refresh_url(),
			'remaining_time'				=> $this->remaining_time,						// In secs - This is default. Can be updated by ajax
			'removed'						=> $this->a->get_text('removed'),
			'required'						=> $this->a->get_text('required'),
			'screen_base'					=> isset( $current_screen->base ) ? $current_screen->base : 0,
			'scroll_duration'				=> 1500,										// Scroll duration in msecs
			'search'						=> $this->a->get_text('search'),
			'selected'						=> $this->a->get_text('selected'),
			'showEffect'					=> array( 'effect' => wpb_setting( 'show_effect' ), 'duration' => 700, 'direction' => 'up' ),
			'sliderMargin'					=> wpb_setting( 'slider_margin', 0 ),
			'sliderCount'					=> wpb_is_mobile() ? wpb_setting( 'slider_count_mobile', 2 ) : wpb_setting( 'slider_count', 4 ),
			'sliderWidth'					=> wpb_setting( 'slider_width', 200 ),
			'submit'						=> $this->a->get_text('submit_confirm'),
			'tab'							=> isset( $_GET['tab'] ) ? wpb_clean( $_GET['tab'] ) : '',
			'too_less'						=> sprintf( $this->a->get_text('too_less'), BASE('Multiple')->get_apt_count_min() ),
			'tt_discounted_price'			=> $this->a->get_text('tt_discounted_price'),
			'tt_regular_price'				=> $this->a->get_text('tt_regular_price'),
			'twitter'						=> $this->a->get_text('login_with_twitter'),
			'username'						=> $this->a->get_text('username'),
			'use_effect'					=> 0,
			'warning_text'					=> $this->a->get_text('missing_field'),
			'wordpress'						=> $this->a->get_text('login_with_wp'),
			'is'							=> array( # Tip
												'diviActive'		=> function_exists( 'et_builder_load_framework' ),
												'lazyLoad'			=> ! wpb_is_admin() && 'yes' === wpb_setting( 'lazy_load' ) && ! wpb_is_account_page(),
												'locFilter'			=> false,	# Adds filter to location multiselect
												'serviceFilter'		=> false,	# Adds filter to service multiselect
												'slideshow'			=> true,	# Enables auto slideshow
												'workerFilter' 		=> false,	# Adds filter to worker multiselect
												'quickfit'			=> true,	# Use quickfit
												'sticky'			=> true,	# If true, selected slot cannot be removed using the same button.
												'noDialog'			=> false,	# Disables dialog
												'noMS'				=> false,	# Disable Multiselect
												'noScroll'			=> false,	# Disable auto scroll
												'autoOpenTTable'	=> true,	# Timetable opens as page loads
												'cacheTTable'		=> true,	# Timetable is cached at browser
												'focus'				=> true,	# Focus at the first field in conf form
												'modal'				=> true,	# jQuery dialog is modal or not.
											),
		);
	}

	/**
     * Call localize script once
     */
	private function localize( $data ) {

		if ( ! empty( $this->localized ) ) {
			return;
		}

		if ( wp_localize_script( 'wp-base-common-scripts', '_app_', $data ) ) {
			$this->localized = true;
		}
	}

	/**
     * Enqueue scripts for admin
     */
	public function load_admin(){

		add_action( 'admin_print_styles', array( $this, 'load_admin_css' ) );

		if ( ! wpb_is_admin_page() ) {
			return;
		}

		$deps = $this->deps_admin();
		$data = apply_filters( 'app_js_data', wpb_array_merge( $this->common_data(), $this->admin_data() ), 'admin' );

		if ( wpb_requires_front() ) {
			wp_register_script( 'wp-base-front-scripts', WPB_PLUGIN_URL . '/js/front-scripts.js', array_merge( $deps, $this->deps_front() ), $this->ver(), true );
			$deps[] = 'wp-base-front-scripts';
			$data = wpb_array_merge( $data, $this->front_data() );
		}

		if ( $handle = $this->register_moment_locale() ) {
			$deps[] = $handle;
		}

		wp_register_script( 'wp-base-admin-scripts', WPB_PLUGIN_URL . '/js/admin/admin-scripts.js', $deps, $this->ver() );
		$this->localize( $data );

		wp_enqueue_script( 'wp-base-admin-scripts' );
		wp_enqueue_script( 'heartbeat' ); # Called separately because some Internet providers may block its usage
	}

	/**
     * Enqueue scripts for front
     */
	public function load_front(){
		$data 	= apply_filters( 'app_js_data', wpb_array_merge( $this->common_data(), $this->front_data() ), 'front' );
		$deps 	= $this->deps_front();

		if ( wpb_is_mobile() ) {
			$deps = array_merge( $deps, array( 'jquery-plugin', 'swipe' ) );
		} else {
			$deps = array_merge( $deps, array( 'jquery-ui-datepicker' ) );
			if ( self::is_debug() ) {
				$deps = array_merge( $deps, array( 'jquery-app-qtip' ) );
			}
		}

		if ( $handle = $this->register_moment_locale() ) {
			$deps[] = $handle;
		}

		wp_register_script( 'wp-base-front-scripts', WPB_PLUGIN_URL . '/js/front-scripts.js', $deps, $this->ver(), true );
		$this->localize( $data );
		$this->enqueue_effects();

		wp_enqueue_script( 'wp-base-front-scripts' );

		do_action( 'app_scripts_enqueued' );
	}

	/**
	 * Load common styles
	 */
	private function load_common_css() {
		wp_enqueue_style( 'wp-base-updating', WPB_PLUGIN_URL . '/css/updating.css', array(), $this->ver() );
		wp_enqueue_style( 'jquery-ui-'.sanitize_file_name( $this->a->selected_theme() ), $this->a->get_theme_file(), array(), $this->ver() );

		if ( WpBDebug::is_debug() ) {
			wp_enqueue_style( 'jquery-ui-structure', WPB_PLUGIN_URL . '/css/jquery-ui.structure.min.css', array(), $this->ver() );
			wp_enqueue_style( 'jquery-qtip', WPB_PLUGIN_URL . '/css/jquery.qtip.css', array(), $this->ver() );
			wp_enqueue_style( 'jquery-multiselect', WPB_PLUGIN_URL . '/css/jquery.multiselect.css', array(), $this->ver() );
			wp_enqueue_style( 'wp-base-countdown', WPB_PLUGIN_URL . '/css/jquery.countdown.css', array(), $this->ver() );
			wp_enqueue_style( 'jquery-datatables-responsive', WPB_PLUGIN_URL . '/css/responsive.dataTables.css', array(), $this->ver() );
			wp_enqueue_style( 'jquery-datatables-responsive-ui', WPB_PLUGIN_URL . '/css/responsive.jqueryui.css', array(), $this->ver() );
			wp_enqueue_style( 'jquery-datatables-jqueryui', WPB_PLUGIN_URL . '/css/dataTables.jqueryui.css', array(), $this->ver() );
		} else {
			wp_enqueue_style( "wp-base-libs-min", WPB_PLUGIN_URL . "/css/libs.min.css", array(), $this->ver() );
		}

		if ( is_rtl() ) {
			wp_enqueue_style( 'wp-base-rtl', WPB_PLUGIN_URL . '/css/common-rtl.css', array(), $this->ver() );
		}
	}

	/**
	 * Load styles for admin
	 */
	public function load_admin_css() {
		if ( 'yes' == wpb_setting( 'disable_css_admin' ) )
			return;

		if ( ! wpb_is_admin_page() ) {
			return;
		}

		wp_enqueue_style( 'wp-base-admin', WPB_PLUGIN_URL . '/css/admin.css', array(), $this->ver() );
		wp_enqueue_style( 'wp-color-picker' );
		wp_enqueue_style( 'editor-buttons' );	// Fix for: wp_editor call does not load editor.min.css on emails page

		if ( wpb_requires_front() ) {
			wp_enqueue_style( 'wp-base', $this->a->get_front_css_file(), array(), $this->ver() );
		}

		$this->load_common_css();

		if ( is_rtl() ) {
			wp_enqueue_style( 'wp-base-admin-rtl', WPB_PLUGIN_URL . "/css/admin-rtl.css", array('wp-base-admin'), $this->ver() );
		}

		do_action( 'app_admin_styles_enqueued' );
	}

	/**
	 * Load styles for front
	 */
	public function load_front_css() {
		if ( 'yes' == wpb_setting( 'disable_css' ) ) {
			return;
		}

		$this->load_common_css();

		wp_enqueue_style( 'wp-base', $this->a->get_front_css_file(), array(), $this->ver() );

		do_action( 'app_styles_enqueued' );
	}
	
	/**
	 * Load assets for unsupported page builders upon WP BASE shortcode usage
	 * @uses do_shortcode_tag filter hook
	 * @since 3.8.0.4
	 * @return string
	 */
	public function do_shortcode_tag( $output, $tag ) {
		if ( ! did_action( 'app_scripts_enqueued' )  && in_array( $tag, wpb_shortcodes() ) ) {
			$this->load_front();
			$this->load_front_css();
		}
		
		return $output;
	}

}
	BASE('Assets')->add_hooks();
}
