<?php
/**
 * WPB Welcome
 *
 * Opens an informative popup on first installation or after an update
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( !class_exists( 'WpBWelcome' ) ) {

class WpBWelcome {

	/**
     * WP BASE Core + Front [+Admin] instance
     */
	protected $a = null;

	/**
     * Constructor
     */
	public function __construct() {
		$this->a = BASE();
	}

	/**
     * Add hooks
     */
	public function add_hooks(){
		add_action( 'admin_init', array( $this, 'welcome_init' ), 8, 2 );		// Welcome popup
		add_action( 'app_installed', array( $this, 'installed' ), 10, 2 );
	}
	
	/**
	 * Mark installation + Upgrade
	 * @since 3.6.0
	 */
	public function installed( $installed_version, $previous_version ) {
		if ( '0' === (string)intval($previous_version) ) {
			update_user_meta( get_current_user_id(), 'app_welcome', WPB_VERSION );
		}
	}

	/**
	 * Triggers welcome popup
	 * @since 2.0
	 */
	public function welcome_init( ) {
		if ( !get_user_meta( get_current_user_id(), 'app_welcome', true ) )
			return;

		if ( is_network_admin() || isset( $_GET['activate-multi'] ) || wpb_is_mobile() )
			return;

		// Import Suggestion from A+ or ApB
		if ( function_exists('wpb_import_from_a_plus') && wpb_admin_access_check( 'manage_tools', false ) && !get_option( 'app_imported_from_a_plus' ) && wpb_import_from_a_plus( true ) ) {
			add_action( 'admin_notices', '_wpb_suggest_a_plus_import' );
		}

		add_filter( 'app_js_data', array( $this, 'js_data' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'welcome_script' ) );
		add_action( 'admin_footer', array( $this, 'welcome' ) );
	}

	/**
	 * Add flag to open dialog
	 * @since 2.0
	 */
	public function js_data( $args ) {
		if ( is_admin() ) {
			$args['is']['welcome'] = (bool)get_user_meta( get_current_user_id(), 'app_welcome' );
		}

		return $args;
	}

	/**
	 * Loads jQuery UI dialog and css files
	 * @since 2.0
	 */
	public function welcome_script() {
		wp_enqueue_style( 'wp-base-libs-min', WPB_PLUGIN_URL . "/css/libs.min.css", array(), WPB_VERSION );
		wp_enqueue_style( 'jquery-ui-'.$this->a->selected_theme( ), $this->a->get_theme_file( ), array(), WPB_VERSION );
		wp_enqueue_script( 'jquery-ui-dialog' );
		wp_enqueue_script( 'jquery-ui-effects-drop' );
	}

	 /**
	 * Displays welcome popup
	 * @since 2.0
	 */
	public function welcome(  ) {
		if ( is_network_admin() || isset( $_GET['activate-multi'] ) || wpb_is_mobile() )
			return;

		delete_user_meta( get_current_user_id(), 'app_welcome' );

?>
<div id="app-welcome" data-title="<?php echo esc_attr( __( 'Welcome to WP BASE - Bookings for Appointments, Services and Events', 'wp-base' ) ) ?>" style="display:none">
	<p><?php _e( 'Thank you for choosing WP BASE.', 'wp-base' ) ?></p>
	<p><?php _e( 'WP BASE is the most advanced Services and Appointment bookings WordPress plugin available on the market. It can be easily set up for most business models, however some sophisticated applications may require a little bit more effort. We have several tools and documentation to help you set up WP BASE to fulfill your business requirements.', 'wp-base' ) ?></p>
	<p><?php _e( 'If you are a first time user, we recommend you to have a look at these links:', 'wp-base' ) ?></p>
	<ul>
		<li><a href="<?php echo WPB_DEMO_WEBSITE ?>" target="_blank"><?php _e( 'Our demo website includes several application examples','wp-base' )?></a></li>
		<li><a href="<?php echo WPBASE_URL .'knowledge-base/' ?>" target="_blank"><?php _e( 'Our Knowledge Base provides reference information','wp-base' )?></a></li>
		<li><a href="<?php echo admin_url('admin.php?page=app_settings') ?>" target="_blank"><?php _e( 'You can visit Global Settings page to create a booking page and adjust other settings','wp-base' )?></a></li>
		<li><a href="javascript:void(0)" class="app_close_welcome"><?php _e( 'You can close this popup if you are already familiar with WP BASE','wp-base' )?></a></li>
	</ul>
</div>
<?php
	}

}
	BASE('Welcome')->add_hooks();
}