<?php
/**
 * WPB Admin Toolbar
 *
 * Adds WP BASE items to admin toolbar
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpBToolbar' ) ) {

class WpBToolbar {

	/**
     * WP BASE Core + Front [+Admin] instance
     */
	protected $a = null;

	/**
     * WP_Admin_Bar object pointer
     */
	private $admin_bar;
	
	/**
     * One time checker
     */
	public $scripts_printed;

	/**
     * Constructor
     */
	public function __construct() {
		$this->a = BASE();
	}

	/**
     * Add hooks
     */
	public function add_hooks(){
		add_action( 'admin_bar_menu', array( $this, 'admin_bar_render'), 101 );				// Add items to admin toolbar
		add_action( 'permalink_structure_changed', array( $this, 'rebuild_menu' ) );		// Rebuild WP BASE admin menu
		add_action( 'save_post', array( $this, 'save_post' ), 10, 2 ); 						// Save post ids if it has shortcodes, also clear cache
		add_action( 'delete_post', array( $this, 'delete_post' ) ); 						// Check if deleted post has shortcodes
		add_action( 'trash_post', array( $this, 'delete_post' ) ); 							// Check if deleted post has shortcodes
		if ( self::is_active() ) {
			add_action( 'admin_print_scripts', array( $this, 'print_scripts' ) );
			add_action( 'wp_print_scripts', array( $this, 'print_scripts' ) );
		}
	}

	/**
	 * Check if admin toolbar is active
	 * @since 3.0
	 * @return bool
	 */
	public static function is_active(){
		return 'yes' === wpb_setting( 'admin_toolbar' );
	}

	/**
	 * Helper function to add more than one item to a parent
	 * @since 3.5.10
	 * @return none
	 */
	private function add_items( $items, $parent = 'wpbase' ) {
		global $wp_admin_bar;

		foreach( $items as $page => $title ) {
			$pid = 'wpbase-'.$page;

			$wp_admin_bar->add_node( array(
				'id'		=> $pid,
				'title' 	=> $title,
				'parent'	=> $parent,
				'href' 		=> admin_url( 'admin.php?page='.$page )
			) );

			$tabs = wpb_admin_tabs( substr( $page, 4 ) );

			if ( !empty( $tabs ) ) {
				foreach( $tabs as $tab => $tab_title ) {
					$wp_admin_bar->add_node( array(
						'id'		=> 'wpbase-'.$page.'-'.$tab,
						'title' 	=> $tab_title,
						'parent'	=> $pid,
						'href' 		=> admin_url( 'admin.php?page='.$page.'&tab='.$tab )
					) );
				}
			}
		}
	}

	 /**
	 * Add items to admin toolbar
	 * @since 2.0
	 * http://codex.wordpress.org/Function_Reference/add_node
	 */
	public function admin_bar_render( $wp_admin_bar ) {
		if ( is_network_admin() || ! self::is_active() || ! current_user_can( WPB_ADMIN_CAP ) ) {
			return;
		}

		if ( ! function_exists('is_admin_bar_showing') || ! is_admin_bar_showing() ) {
			return;
		}

		if ( $count = wpb_pending_count() ) {
			/* translators: %s: number of notifications */
			$text		= sprintf( _n( '%s notification', '%s notifications', $count, 'wp-base' ), number_format_i18n( $count ) );
			$counter	= sprintf( ' <div class="wp-core-ui wp-ui-notification app-issue-counter"><span class="app-issues-count" aria-hidden="true">%d</span><span class="screen-reader-text">%s</span></div>', $count, $text );
			
		} else {
			$counter = '';
		}

		$wp_admin_bar->add_node( array(
			'id'		=> 'wpbase',
			'title'		=> __('WP BASE','wp-base'). $counter,
		) );

		$items = apply_filters( 'app_toolbar_items', array(
			'appointments'		=> __('Dashboard','wp-base'),
			'app_bookings'		=> __('Bookings','wp-base'),
			'app_clients'		=> __('Clients','wp-base'),
			'app_schedules'		=> __('Schedules','wp-base'),
			'app_transactions'	=> __('Payment History','wp-base'),
		), $wp_admin_bar );

		$this->add_items( $items );

		$wp_admin_bar->add_node( array(
			'id'		=> 'wpbase-settings',
			'title'		=> __( 'Settings', 'wp-base' ),
			'parent'	=> 'wpbase',
		) );

		$items = apply_filters( 'app_toolbar_setting_items', array(
			'app_business'		=> __('Business','wp-base'),
			'app_monetary'		=> __('Monetary','wp-base'),
			'app_display'		=> __('Display','wp-base'),
			'app_settings'		=> __('Global','wp-base'),
		), $wp_admin_bar );

		$this->add_items( $items, 'wpbase-settings' );

		$items = array(
			'app_tools'			=> __('Tools','wp-base'),
			'app_addons'		=> __('Addons','wp-base'),
			'app_help'			=> __('Help','wp-base'),
		);

		$this->add_items( $items );

		$wp_admin_bar->add_group( array(
			'id'		=> 'wpbase-admin',
			'parent'	=> 'wpbase',
		) );

		$wp_admin_bar->add_node( array(
			'id'		=> 'wpbase-your-bookings',
			'title' 	=> __('Your Bookings','wp-base'),
			'parent'	=> 'wpbase-admin',
			'href' 		=> admin_url( 'users.php?page=your_bookings' ),
		) );

		$wp_admin_bar->add_group( array(
			'id'		=> 'wpbase-front-end',
			'parent'	=> 'wpbase',
		) );

		if ( $make_pages = wpb_setting('make_page') ) {

			$count = 0;
			$items = array();

			foreach ( (array)$make_pages as $page_id ) {
				$page = get_post( $page_id );
				if ( is_object( $page ) && ( 'post' == $page->post_type || 'page' == $page->post_type ) ) {
					$count++;
					$items[ $page->ID ] = $page->post_title;
					if ( $count >= 12 ) {
						break;
					}
				}
			}

			if ( $count == 1 ) {

				$wp_admin_bar->add_node( array(
					'id'		=> 'wpbase-'.$page->ID,
					'title' 	=> $page->post_title,
					'parent'	=> 'wpbase-front-end',
					'href' 		=> get_permalink( $page->ID ),
				) );

			} else if ( $count > 1 ) {

				$wp_admin_bar->add_node( array(
					'id'		=> 'wpbase-booking-pages',
					'title' 	=> __( 'Booking Pages','wp-base'),
					'parent'	=> 'wpbase-front-end',
				) );

				asort( $items );

				foreach ( $items as $page_id => $title ) {

					$wp_admin_bar->add_node( array(
						'id'		=> 'wpbase-'.$page_id,
						'title' 	=> $title,
						'parent'	=> 'wpbase-booking-pages',
						'href' 		=> get_permalink( $page_id ),
					) );
				}
			}

		}

		$items = apply_filters( 'app_admin_toolbar_frontend_items', array(
			'account_page'		=> __('Account','wp-base'),
			'list_page'			=> __('List of Bookings','wp-base'),
			'manage_page'		=> __('Manage Bookings','wp-base'),
			'mv_register_page'	=> __('Registration','wp-base'),
		) );

		foreach( $items as $stg => $title ) {
			if ( $page_id = wpb_setting( $stg ) ) {
				if ( $permalink = get_permalink( $page_id ) ) {
					$wp_admin_bar->add_node( array(
						'id'		=> 'wpbase-'.$page_id,
						'title' 	=> $title,
						'parent'	=> 'wpbase-front-end',
						'href' 		=> $permalink,
					) );
				}
			}
		}
	}
	
	/**
	 * Print codes to header
	 * @since 2.0
	 */
	public function print_scripts() {
		if ( ! empty( $this->scripts_printed ) ) {
			return;
		}
?>
<style type="text/css">
#wpadminbar .app-issue-counter {
    border-radius: 9px;
    color: #fff;
    display: inline;
    padding: 1px 7px 1px 6px!important;
}
.app-issue-counter.wp-ui-notification {
    color: #fff;
    background-color: #d63638;
}
</style>
<?php
		$this->scripts_printed = true;
	}

	/**
	 * Check if saved post has shortcodes, clear cache and save post ids
	 * @since 2.0
	 * @return null
	 */
	public function save_post( $post_id, $post, $options = false ) {

		if ( ! is_object( $post ) ) {
			$post = get_post( $post_id );
		}

		if ( 'revision' == $post->post_type || 'attachment' == $post->post_type ) {
			return;
		}

		$options = wpb_setting();

		$changed = false;
		$content = $post->post_content;
		$status = $post->post_status;

		// Check if we have the shortcodes
		if ( strpos( $content, '[app_manage' ) !== false ) {
			$options['manage_page']	= $post_id;
			$changed = true;
		} else if ( strpos( $content, '[app_confirmation' ) !== false || strpos( $content, '[app_book' ) !== false ) {
			$options['make_page'][] = $post_id;
			$options['make_page'] = array_unique( array_filter( $options['make_page'] ) );
			$changed = true;
		} else if ( strpos( $content, '[app_account' ) !== false ) {
			$options['account_page'] = $post_id;
			$changed = true;
		} else if ( has_shortcode( $content, '[app_list' ) ) {
			$options['list_page'] = $post_id;
			$changed = true;
		} else if ( has_shortcode( $content, '[app_register' ) ) {
			$options['mv_register_page'] = $post_id;
			$changed = true;
		}

		// If shortcode cleared

		$key = isset( $options['make_page'] ) ? array_search( $post_id, (array)$options['make_page'] ) : false;
		if ( $key !== false && ('trash' == $status || (strpos( $content, '[app_confirmation' ) === false && strpos( $content, '[app_book' ) === false ) ) ) {
			unset( $options['make_page'][$key] );
			$changed = true;
		}

		if ( wpb_setting('account_page') == $post_id && ( strpos( $content, '[app_account' ) === false || 'trash' == $status ) ) {
			unset( $options['account_page'] );
			$changed = true;
		}

		if ( wpb_setting('list_page') == $post_id && ( strpos( $content, '[app_list' ) === false || 'trash' == $status ) ) {
			unset( $options['list_page'] );
			$changed = true;
		}

		if ( wpb_setting('manage_page') == $post_id && ( strpos( $content, '[app_manage' ) === false || 'trash' == $status ) ) {
			unset( $options['manage_page'] );
			$changed = true;
		}

		if ( wpb_setting('mv_register_page') == $post_id && ( strpos( $content, '[app_register' ) === false || 'trash' == $status ) ) {
			unset( $options['mv_register_page'] );
			$changed = true;
		}

		if ( $changed ) {
			$this->a->pause_log = true;
			$this->a->update_options( $options );
			wpb_flush_cache();
			unset( $this->a->pause_log );
		}

	}

	/**
	 * Check if deleted post was in the menu
	 * @since 2.0
	 * @return none
	 */
	public function delete_post( $post_id ) {
		$options	= wpb_setting();
		$changed	= false;
		$make_page	= wpb_setting('make_page');

		if ( ! is_array( $make_page ) ) {
			if ( $make_page == $post_id ) {
				unset( $options['make_page'] );
				$changed = true;
			}
		} else {
			foreach ( $make_page as $key => $page ) {
				if ( $page == $post_id ) {
					unset( $options['make_page'][$key] );
					$changed = true;
				}
			}
		}

		foreach ( array( 'all_page', 'account_page', 'list_page', 'manage_page', 'mv_register_page' ) as $_page ) {
			if ( wpb_setting( $_page ) == $post_id ) {
				unset( $options[ $_page ] );
				$changed = true;
			}
		}

		if ( $changed ) {
			$this->a->pause_log = true;
			$this->a->update_options( $options );
			wpb_flush_cache();
			unset( $this->a->pause_log );
		}
	}

	/**
	 * Rebuild menu items
	 * @since 2.0
	 * @return none
	 */
	public function rebuild_menu( ) {
		if ( ! $posts = $this->a->db->get_results( "SELECT * FROM ". $this->a->db->posts.
				" WHERE post_type='page' AND post_status='publish'
				AND post_content LIKE '%[app_%' ORDER BY post_date DESC LIMIT 100" ) ) {
			return;
		}

		foreach ( $posts as $_post ) {
			if ( is_object( $_post ) && !empty( $_post->ID ) && !empty( $_post->post_type ) ) {
				$this->save_post( $_post->ID, $_post );
			}
		}
	}

}
	BASE('Toolbar')->add_hooks();
}