<?php
/**
 * WPB Admin Services List
 *
 * Handles display and creation of services on admin side
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.8.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpBAdminServices' ) ) {

class WpBAdminServices {

	public $row_number = 0;

	private $colspan = 7;

	private $by_worker, $sel_location;

	/**
     * WP BASE instance
     */
	protected $a = null;

	/**
     * Reference to services_table
     */
	private $table;

	/**
     * Last inserted id saved from $wpdb->insert_id
     */
	private $last_insert_id = 0;

	/**
     * Name of the tab/title
     */
	private static $name;

	/**
     * Name of user option for Hidden Columns
     */
	const HIDDEN_COLUMNS = 'app_manage_services_columnshidden';

	/**
     * Name of user option for Hidden Navbar
     */
	const HIDDEN_NAVBAR = 'app_manage_services_navbarhidden';

	/**
     * Constructor
     */
	public function __construct() {
		$this->a = BASE();
		self::$name = array( 'services' => __('Services', 'wp-base') );
		$this->table = $this->a->services_table;
		include_once( WPBASE_PLUGIN_DIR . '/includes/admin/service-props.php' );
	}

	/**
     * Add actions and filters
     */
	public function add_hooks() {
		add_filter( 'admin_title', array( $this, 'admin_title' ), 10, 2 );
		add_filter( 'appointments_business_tabs', array( $this, 'add_services_tab' ), 4 );
		add_action( 'app_business_services_tab', array( $this, 'listing' ) );
		add_action( 'app_save_settings', array( $this, 'change_status_bulk' ) );
		add_action( 'app_account_load_assets', array( $this, 'change_status_bulk' ) );
		add_filter( 'heartbeat_received', array( $this, 'refresh_lock' ), 10, 3 );
		add_action( 'wp_ajax_app_delete_lock_service', array( $this, 'delete_lock' ) );
		add_action( 'wp_ajax_app_start_lock_service', array( $this, 'start_lock' ) );
		add_action( 'wp_ajax_app_inline_edit_service', array( $this, 'inline_edit' ) );
		add_action( 'wp_ajax_app_inline_edit_save_service', array( $this, 'inline_edit_save' ) );
		add_action( 'wp_ajax_app_save_hidden_columns_service', array( $this, 'save_hidden_columns' ) );
		add_action( 'wp_ajax_app_save_navbar_service', array( $this, 'save_navbar' ) );
		add_action( 'wp_ajax_app_save_sort_service', array( $this, 'save_sort' ) );

	}

	/**
	 * Whether user can edit this service
	 * @since 3.8.0
	 * @return bool
	 */
	public static function can_edit( $service ) {
		$service = $service instanceof WpB_Service ? $service : new WpB_Service( $service );

		if ( current_user_can( WPB_ADMIN_CAP ) || $service->get_owner() == get_current_user_id() ) {
			return true;
		}
	}

	/**
	 * Change admin SEO title
	 * @since 3.8.0
	 */
	public function admin_title( $admin_title, $title ) {
		if ( ( ! empty( $_GET['tab'] ) && key( self::$name ) == $_GET['tab'] ) ||  ( empty( $_GET['tab'] ) &&  ! empty( $_GET['page'] ) && 'app_business' == $_GET['page'] ) ) {
			return str_replace( $title, current( self::$name ), $admin_title );
		} else {
			return $admin_title;
		}
	}

	/**
	 * Add "services" tab
	 */
	public function add_services_tab( $tabs ) {
		if ( wpb_admin_access_check( 'manage_services', false ) ) {
			$tabs = array_merge( $tabs, self::$name );
		}

		return $tabs;
	}

	/**
	 * Define an array of allowed columns in services page
	 * Called after addons loaded
	 */
	public static function get_allowed_columns() {
		$allowed = array( 'delete','id','image','sort_order','sname','internal','owner','location','category','provider','capacity','duration','padding_before','padding_after','price','deposit','description_page' );
		return apply_filters( 'app_services_allowed_columns', $allowed, '' );
	}

	/**
     * Add some columns as hidden to Bookings page if there is no selection before
     */
	private static function default_hidden_columns( ) {
		return apply_filters( 'app_services_hidden_columns', array( 'owner','location','category','provider','padding_before','padding_after','deposit' ) );
	}

	/**
	 * Return number of columns
	 */
	public function nof_columns(){

		$tabs	= apply_filters( 'app_admin_services_add_more_tab', array() );
		$count	= $this->colspan + 2;

		if ( $tabs && count( $tabs ) == 1 ) {
			return $count + 2;
		} else {
			return $count;
		}
	}

	/**
	 * Get hidden columns for the table
	 * @since 3.9
	 * @return array
	 */
	private static function get_hidden_columns(){
		$hidden_columns = get_user_option( self::HIDDEN_COLUMNS );

		if ( empty( $hidden_columns['hidden_check'] ) ) {
			$hidden_columns = self::default_hidden_columns();
		}

		return $hidden_columns;
	}

	/**
	 * Select element to display columns on the table
	 * @since 3.9
	 * @return string
	 */
	private function displayed_columns_selection(){
		$cols	= self::get_allowed_columns();
		$hidden = self::get_hidden_columns();

		$html = '<select size="16" multiple data-scope="service" data-table="app-services" class="app_displayed_columns app-ms-small" data-selectedtext="'.esc_attr( __( 'Columns', 'wp-base' ) ).'">';
		foreach ( $cols as $col ) {
			if ( 'delete' == $col || 'name' == $col ) {
				continue;
			}

			if ( 'id' == $col ) {
				$text = __( 'ID', 'wp-base' );
			} else if ( 'sname' == $col ) {
				$text = $this->a->get_text( 'name' );
			} else {
				$text = $this->a->get_text( $col );
			}

			$html .= '<option value="'.$col.'" '. selected( in_array( $col, $hidden ), false, false).'>'.$text.'</option>';
		}
		$html .= '</select>';

		return $html;
	}

	/**
	 * Ajax save hidden columns
	 * @since 3.8
	 * @return json
	 */
	public function save_hidden_columns(){
		if ( ! check_ajax_referer( 'inline_edit', 'ajax_nonce', false ) ) {
			die( json_encode( array('error' => $this->a->get_text('unauthorised') ) ) );
		}

		if ( wpb_is_demo() ) {
			die( json_encode( array( 'error' => __('Changes cannot be saved in DEMO mode!', 'wp-base' ) ) ) );
		}

		if ( ! empty( $_POST['hidden'] ) ) {
			update_user_option( get_current_user_id(), self::HIDDEN_COLUMNS, array_merge( $_POST['hidden'], array( 'hidden_check' => 1 ) ) );
		}

		wp_send_json_success();
	}

	/**
	 * Ajax save navbar
	 * @since 3.8
	 * @return json
	 */
	public function save_navbar(){
		if ( ! check_ajax_referer( 'inline_edit', 'ajax_nonce', false ) ) {
			die( json_encode( array('error' => $this->a->get_text('unauthorised') ) ) );
		}

		if ( wpb_is_demo() ) {
			die( json_encode( array( 'error' => __('Changes cannot be saved in DEMO mode!', 'wp-base' ) ) ) );
		}

		update_user_option( get_current_user_id(), self::HIDDEN_NAVBAR, ! empty( $_POST['hidden'] ) );

		wp_send_json_success();
	}

	/**
	 * Ajax save sort
	 * @since 3.8
	 * @return json
	 */
	public function save_sort(){
		if ( ! check_ajax_referer( 'inline_edit', 'ajax_nonce', false ) ) {
			die( json_encode( array('error' => $this->a->get_text('unauthorised') ) ) );
		}

		if ( wpb_is_demo() ) {
			die( json_encode( array( 'error' => __('Changes cannot be saved in DEMO mode!', 'wp-base' ) ) ) );
		}

		$_sorts = json_decode( wp_unslash( $_POST['sort_data'] ), true );
		$sorts = array();

		foreach ( $_sorts as $i => $v ) {
			$sorts[ trim( $i ) ] = $v;
		}

		$max	= count( $sorts ) > 0 ? max( array_keys( $sorts ) ) : 0;
		$paged	= ! empty( $_POST['app_paged'] ) ? $_POST['app_paged'] : 0;
		$i		= 0 + $paged*100;
		$data	= array();

		foreach ( $sorts as $id => $sort ) {
			$_service = new WpB_Service( $id );

			if ( $_service->get_sort_order() != $i ) {
				$_service->set_sort_order( $i );
				$_service->save();
			}

			$data[] = array( 'sid' => $_service->get_ID(), 'order' => $i );

			$i++;
		}

		wp_send_json_success( $data );
	}

	/**
	 * Helper to pass page and tab variables
	 */
	public static function print_fields(){
		if ( ! empty( $_GET['tab'] ) ) { ?>
			<input type="hidden" name="tab" value="<?php echo wpb_clean( $_GET['tab'] ) ?>"/>
		<?php }
		if ( is_admin() ) { ?>
			<input type="hidden" name="page" value="app_business"/>
		<?php }
	}

	/**
	 * Logged in worker ID
	 * @return false|integer
	 */
	private function get_by_worker() {
		return $this->by_worker;
	}

	/**
	 * Selected Location, e.g. BP groups
	 * @return false|integer
	 */
	private function get_sel_location() {
		return $this->sel_location;
	}

	/**
	 * Whether admin or worker has authority
	 * @return bool
	 */
	private static function check_by_worker( $by_worker ) {
		if ( ( $by_worker && $by_worker != get_current_user_id() ) || ( ! $by_worker && ! current_user_can( WPB_ADMIN_CAP ) ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Creates the list for Services admin page
	 * Also used by Account page
	 */
	public function listing( $by_worker = false, $sel_location = 0 ) {

		if ( ! self::check_by_worker( $by_worker ) ) {
			wpb_notice( 'unauthorised', 'error' );
			return;
		}

		wpb_admin_access_check( ($by_worker ? 'manage_own_services' : 'manage_services') );

		do_action( 'app_admin_services_start_render', $by_worker );

		wpb_add_action_footer( $this );

		# Load descriptions to the footer
		# Compatibility for older Addons
		foreach( array( 'ESF', 'GroupBookings', 'Packages', 'Quotas', 'Recurring', 'SelectableDurations', 'TimeVariantDurations', 'WaitingList' ) as $id ) {
			wpb_add_action_footer( BASE( $id ) );
		}

		$this->by_worker	= $by_worker;
		$this->sel_location = $sel_location;

		$is_hidden = (bool)get_user_option( self::HIDDEN_NAVBAR );

	?>
	<div class="wrap">
		<div id="app-open-navbar"<?php echo(! $is_hidden ? ' style="display:none"' : '')?>>
			<button title="<?php echo esc_attr( __( 'Open toolbar', 'wp-base' ) ) ?>"></button>
		</div>

		<div id="app-navbar" class="app-control-wrap metabox-holder"<?php echo ($is_hidden ? ' style="display:none"' : '')?>>

			<div id="app-close-navbar">
				<button title="<?php echo esc_attr( __( 'Close toolbar', 'wp-base' ) ) ?>"></button>
			</div>

			<div class="postbox app-controls">
				<div id="services-filter" class="tablenav top">
					<div class="app-actions actions">

						<div class="app-dash-title"><span><?php echo (self::get_by_worker() ? BASE()->get_text('bp_services') : sprintf( __('%s Services','wp-base'), WPB_NAME ) ); ?></span></div>

					<?php if ( apply_filters( 'app_service_inline_edit_show_add_new', true, $this ) ) { ?>
						<a href="javascript:void(0)" class="add-new-h2 add-new-service"><?php _e('Add New Service', 'wp-base') ?></a>
					<?php } ?>

						<div class="app-manage-second-column"><?php $this->search_form() ?></div>

					</div>

					<div class="app-actions actions"><?php
					self::sort_form();
					$this->filter_form();
					self::clear_filters_form(); ?></div>

				</div>
			</div>
		</div>

		<?php // do_action( 'app_admin_services_after_info' ); ?>

		<div class="app-manage-row third-row app-mt">
			<div class="app-manage-first-column">
				<?php self::status_change_form() ?>
			</div>

			<?php do_action( 'app_admin_services_before_displayed_columns' ); ?>

			<div class="app-manage-second-column app-crowded">
				<?php echo $this->displayed_columns_selection(); ?>
			</div>

			<?php do_action( 'app_admin_services_after_displayed_columns' ); ?>
		<?php
		# Pagination
		if ( is_admin() ) {
			$paged = empty($_GET['paged']) ? 1 : absint( $_GET['paged'] );
		} else {
			$paged = get_query_var( 'paged' ) ? absint( get_query_var( 'paged' ) ) : 1;
		}

		$rpp 		= wpb_setting( 'records_per_page', 30 ); # Records per page
		$startat	= ($paged - 1) * $rpp;
		$services	= $this->get_admin_services( $startat, $rpp );
		$total		= $this->get_services_total( );

		wpb_paginate_links( $total, $paged, $rpp );

		?>
		</div>
		<?php
		# Preload meta cache as a whole instead of calling one by one
		WpBMeta::update_meta_cache( 'service', wp_list_pluck( $services, 'ID' ) );

		$this->display_table( $services );
?> </div> <?php
	}

	/**
	 * Build mysql query and return results for services
	 */
	private function get_admin_services( $startat, $num ) {

		$sel_location	= $this->get_sel_location();

		if ( ! empty( $_GET['app_s'] ) && ! empty( $_GET['stype'] ) ) {
			$stype 	= esc_sql( $_GET['stype'] );
			$s 		= apply_filters( 'app_search', esc_sql( $_GET['app_s'] ), $stype );

			switch ( $stype ) {
				case 'service_id':
								$sarr = array_filter( array_map( 'esc_sql', array_map( 'trim', explode( ' ', str_replace( ',', ' ', $s ) ) ) ) );
								$add = ! empty( $sarr ) ? "(ID='". implode( "' OR ID='", $sarr ) . "')" : ""; break;
				case 'name':	$add = "LOWER(name) LIKE '%{$s}%' ";
								break;

				default:		$add = apply_filters( 'app_services_search_switch', ' 1=1 ', $stype, $type, $s );
				break;
			}
		} else {
			$add = '1=1';
		}

		# Get sanitized filter params
		$filt = wpb_sanitize_search();

		if ( ! isset( $_GET['app_or_fltr'] ) ) {
			# Filters
			if ( $filt['owner'] ) {
				$add .= $this->a->db->prepare( " AND ID IN (SELECT object_id FROM ".$this->a->meta_table.
						" WHERE meta_type='service' AND meta_key='owner' AND meta_value=%d)", $filt['owner'] );
			} else if ( "0" === (string)$filt['owner'] ) {
				$add .= " AND (ID IN (SELECT object_id FROM ".$this->a->meta_table.
						" WHERE meta_type='service' AND meta_key='owner' AND meta_value=0)
						  OR ID NOT IN (SELECT object_id FROM ".$this->a->meta_table.
						" WHERE meta_type='service' AND meta_key='owner' AND meta_value<>''))";
			}

			if ( $filt['location_id'] ) {
				$add .= $this->a->db->prepare( " AND locations LIKE %s ", '%:'. $this->a->db->esc_like( $filt['location_id'] ) .':%' );
			}

			if ( $filt['category_id'] ) {
				$add .= $this->a->db->prepare( " AND categories LIKE %s ", '%:'. $this->a->db->esc_like( $filt['category_id'] ) .':%' );
			}

			if ( $filt['duration'] ) {
				$add .= $this->a->db->prepare( " AND duration=%d ", $filt['duration'] );
			}
		}

		if ( $this->get_by_worker() ) {
			$_worker = get_current_user_id();
		} else if ( $filt['worker_id'] && ! isset( $_GET['app_or_fltr'] ) ) {
			$_worker = $filt['worker_id'];
		} else {
			$_worker = false;
		}

		if ( $_worker ) {
			$provided = $this->a->get_services_by_worker( $_worker );
			$add .= $provided ? ' AND ID IN(' . implode( ',', array_keys( $provided ) ) .') ' : ' AND 1=2 ';
		}

		if ( $sel_location ) {
			$add .= $this->a->db->prepare( " AND locations LIKE %s ", '%:'. $this->a->db->esc_like( $sel_location ) .':%' );
		}

		# Sanitize Order by
		$test = str_replace( array(' desc', ' inc', ' ' ), '', strtolower( $filt['order_by'] ) );
		$meta = 'app_admin_services_preferred_order';

		if ( $test && in_array( $test, array( 'name', 'sort_order', 'id', 'id-desc', 'name-desc', 'sort_order-desc' ) ) ) {
			$order = str_replace( array('-'), array(' '), $filt['order_by'] );
			update_user_meta( get_current_user_id(), $meta, $order );
		} else {
			$order = "sort_order";
		}

		$select 	= "SELECT SQL_CALC_FOUND_ROWS * FROM {$this->table} AS services ";
		$order_by 	= "ORDER BY {$order}";
		$limit 		= $this->a->db->prepare( "LIMIT %d,%d", $startat, $num );
		$add		= apply_filters( 'app_admin_services_sql_add', $add, $order_by, $limit );

		$sql = $select. "WHERE {$add} {$order_by} {$limit}";
		$sql = apply_filters( 'app_admin_services_sql', $sql, $add, $order_by, $limit );

		return $this->a->db->get_results( $sql );
	}

	/**
	 * Get total number from previous query
	 * @return integer
	 */
	private function get_services_total( ) {
		return $this->a->db->get_var( "SELECT FOUND_ROWS();" );
	}

	/**
	 * Helper to produce search form
	 * @return string
	 */
	private function search_form(){
		$stype	= ! empty( $_GET['stype'] ) ? wpb_clean( $_GET['stype'] ) : '';
		$s		= apply_filters( 'app_services_search', (! empty( $_GET['app_s'] ) ? wpb_clean( $_GET['app_s'] ) : ''), $stype );
	?>
	<form id="app-search-form" method="get" action="" class="search-form">
		<?php self::print_fields() ?>
		<input type="hidden" value="1" name="app_or_fltr" />
		<input type="search" value="<?php echo esc_attr($s); ?>" name="app_s" />

		<?php $add_class = $stype === 'service_id' ? 'class="app-option-selected"' : ''; ?>
		<select name="stype" <?php echo $add_class ?>>
			<option value="name" <?php selected( $stype, 'name' ); ?>><?php _e('Service Name','wp-base'); ?></option>
			<option value="service_id" <?php selected( $stype, 'service_id' ); ?>><?php _e('Service ID','wp-base'); ?></option>
			<?php do_action( 'app_services_search_options', $stype ) ?>
		</select>

		<input type="submit" class="button app-search-button" value="<?php _e('Search','wp-base'); ?>" />
	</form>
	<?php
	}

	/**
	 * Helper to produce filter form
	 * @return string
	 */
	public function filter_form() {
		# Get sanitized filter params
		$filt = wpb_sanitize_search();
	?>
	<form id="app-filter-form" class="app-filter-form" method="get" action="">
		<?php
		do_action( 'app_admin_services_form_filter_pre' );

		if ( ! $this->get_by_worker() ) {
			$owners = $this->a->db->get_results( "SELECT DISTINCT meta_value FROM ".$this->a->meta_table." WHERE meta_type='service' AND meta_key='owner'" );

			$add_class = $filt['owner'] ? 'class="app-option-selected"' : '';
			?>
			<select name="app_owner" <?php echo $add_class ?>>
				<option value=""><?php _e('Filter by owner','wp-base'); ?></option>
			<?php if ( $owners ) {  ?>
				<option value="0" <?php selected( $filt['owner'], 0 ) ?>><?php _e('Admin','wp-base'); ?></option>
			<?php
				foreach ( $owners as $owner ) {
					$selected = $filt['owner'] == $owner->meta_value ? " selected='selected' " : '';
					echo '<option '.$selected.' value="'.$owner->meta_value.'">'. $this->a->get_worker_name( $owner->meta_value ) .'</option>';
				}
			}
			?>
			</select>
		<?php }

		if ( ! $this->get_sel_location() && $locations = $this->a->get_locations( 'name' ) ) {
			$add_class = $filt['location_id'] ? 'class="app-option-selected"' : '';
		?>
			<select name="app_location_id" <?php echo $add_class ?>>
				<option value=""><?php _e('Filter by location','wp-base'); ?></option>
		<?php
			foreach ( $locations as $loc) {
				echo '<option '.selected( $filt['location_id'], $loc->ID, false ).' value="'.$loc->ID.'">'. $this->a->get_location_name( $loc->ID ) .'</option>';
			}
		?>
			</select>
		<?php }

		if ( $cats = $this->a->get_categories( 'name' ) ) {
			$add_class = $filt['category_id'] ? 'class="app-option-selected"' : '';
		?>
			<select name="app_category_id" <?php echo $add_class ?>>
				<option value=""><?php _e('Filter by category','wp-base'); ?></option>
		<?php
			foreach ( $cats as $id => $cat ) {
				echo '<option '.selected( $filt['category_id'], $id, false ).' value="'.$id.'">'. $this->a->get_category_name( $id ) .'</option>';
			}
		?>
			</select>
		<?php }

		if ( ! $this->get_by_worker() && $this->a->get_nof_workers() ) {
			$workers = (array)$this->a->get_workers( 'name' );
			$add_class = $filt['worker_id'] || "0" === (string)$filt['worker_id'] ? 'class="app-option-selected"' : '';
			?>
			<select name="app_worker_id" <?php echo $add_class ?>>
				<option value=""><?php _e('Filter by provider','wp-base'); ?></option>
			<?php
				foreach ( $workers as $worker ) {
					$selected = $filt['worker_id'] == $worker->ID ? " selected='selected' " : '';
					echo '<option '.$selected.' value="'.$worker->ID.'">'. $this->a->get_worker_name( $worker->ID ) .'</option>';
				}
			?>
			</select>
		<?php }

		$durations = $this->a->db->get_results( "SELECT DISTINCT duration FROM ".$this->table. " ORDER BY duration" );

		$add_class = $filt['duration'] ? 'class="app-option-selected"' : '';
		?>
		<select name="app_duration" <?php echo $add_class ?>>
			<option value=""><?php _e('Filter by duration','wp-base'); ?></option>
		<?php if ( $durations ) {
			foreach ( $durations as $s ) {
				$selected = $filt['duration'] == $s->duration ? " selected='selected' " : '';
				echo '<option '.$selected.' value="'.$s->duration.'">'. wpb_format_duration( $s->duration ) .'</option>';
			}
		}
		?>
		</select>

		<?php do_action( 'app_admin_services_form_filter' )  ?>

		<?php self::print_fields() ?>
		<input type="submit" class="button" value="<?php _e('Filter','wp-base'); ?>" />
	</form>
	<?php
	}

	/**
	 * Helper to produce clear filters form
	 * @return string
	 */
	private static function clear_filters_form() {
?>
		<form id="app-service-reset-form" method="get" action="<?php echo wpb_add_query_arg(null, null); ?>" >
			<?php self::print_fields() ?>
			<input type="hidden" value="1" name="app_filter_reset" />
			<input type="submit" class="button" value="<?php _e('Clear Filters','wp-base'); ?>" />
		</form>
<?php
	}

	/**
	 * Helper to produce sort form
	 * @return string
	 */
	private static function sort_form() {
		$filt = wpb_sanitize_search();
		$add_class = $filt['order_by'] ? 'class="app-option-selected"' : '';
	?>
	<form id="app-sort-form" method="get" action="" >
		<select name="app_order_by" <?php echo $add_class ?>>
			<option value=""><?php _e('Sort by','wp-base'); ?></option>
			<option value="sort_order" <?php selected( $filt['order_by'], 'sort_order' ); ?>><?php _e('Default order','wp-base'); ?></option>
			<option value="sort_order-DESC" <?php selected( $filt['order_by'], 'sort_order-DESC' ); ?>><?php _e('Reverse order','wp-base'); ?></option>
			<option value="ID" <?php selected( $filt['order_by'], 'ID' ); ?>><?php _e('ID (1 &rarr; 99)','wp-base'); ?></option>
			<option value="ID-DESC" <?php selected( $filt['order_by'], 'ID-DESC' ); ?>><?php _e('ID (99 &rarr; 1)','wp-base'); ?></option>
			<option value="name" <?php selected( $filt['order_by'], 'name' ); ?>><?php _e('Name (A &rarr; Z)','wp-base'); ?></option>
			<option value="name-DESC" <?php selected( $filt['order_by'], 'name-DESC' ); ?>><?php _e('Name (Z &rarr; A)','wp-base'); ?></option>
		</select>
		<?php self::print_fields() ?>
		<input type="submit" class="button" value="<?php _e('Sort','wp-base'); ?>" />
	</form>
	<?php
	}

	/**
	 * Helper to produce status change form
	 * @return string
	 */
	private static function status_change_form(){
		global $post;
	?>
	<form id="app-bulk-service-change-form" method="post" action="" >
		<select name="app_new_status">
			<option value=""><?php _e('Bulk status change','wp-base'); ?></option>
			<option value="internal"><?php _e('Set as Internal (private)','wp-base'); ?></option>
			<option value="public"><?php _e('Set as public','wp-base'); ?></option>
			<option value="delete"><?php _e('Delete permanently','wp-base'); ?></option>
		</select>
		<?php self::print_fields() ?>
		<input type="hidden" value="<?php if ( isset( $post->ID ) ) echo $post->ID; else echo 0; ?>" name="page_id" />
		<input type="hidden" value="app_service_status_change" name="action_app" />
		<input type="hidden" value="1" name="app_service_status_change" />
		<?php wp_nonce_field( 'update_app_settings', 'app_nonce' ); ?>
		<input type="submit" id="change_status_service" class="button app-change-status-btn" value="<?php _e('Change','wp-base'); ?>" />
	</form>
	<?php
	}

	/**
	 * Change status of services from admin page
	 */
	public function change_status_bulk(){

		if ( !( isset( $_POST["app_service_status_change"] ) && $_POST["app_new_status"] && isset( $_POST["app"] ) && is_array( $_POST["app"] ) ) ) {
			return;
		}

		$changed	= $deleted = 0;
		$stat		= $_POST["app_new_status"];
		$ids		= implode( ',', array_map( 'esc_sql', $_POST['app'] ) );

		if ( 'delete' == $stat ) {

			$can_delete = wpb_setting('allow_worker_delete_service'); # yes, no, if_empty
			$is_admin	= current_user_can( WPB_ADMIN_CAP );

			if ( ! $is_admin && ! in_array( $can_delete, array( 'yes', 'if_empty' ) ) ) {
				wpb_notice( __('Deletion is turned off by admin.','wp-base'), 'error' );
				return false;
			}

			foreach ( $_POST["app"] as $id ) {
				if ( apply_filters( 'app_service_skip_delete', false, $id ) ) {
					continue;
				}

				$s	= new WpB_Service( $id );

				if ( ! self::can_edit( $s ) ) {
					continue;
				}

				if ( ! $is_admin && 'if_empty' == $can_delete && $s->has_booking() ) {
					continue;
				}

				if ( wpb_delete_service( $id ) ) {
					# Update related bookings
					$affected = $this->a->db->query( $this->a->db->prepare(
								"UPDATE {$this->a->app_table} SET service=0, status='removed' WHERE ID=%d", $id ));

					wpb_flush_cache();

					do_action( 'app_service_deleted', $id, $affected );

					$deleted++;
				}
			}
		} else if ( 'internal' == $stat ) {
			$changed = $this->a->db->query( "UPDATE {$this->table} SET internal = REPLACE(internal, '0', '1' ) WHERE ID IN ({$ids})" );
		} else if ( 'public' == $stat ) {
			$changed = $this->a->db->query( "UPDATE {$this->table} SET internal = REPLACE(internal, '1', '0' ) WHERE ID IN ({$ids})" );
		}

		if ( $deleted ) {
			wpb_notice( sprintf( __( '%d records have been deleted', 'wp-base' ), $deleted ) );
		}

		if ( $changed ) {
			wpb_notice( sprintf( __( '%d records have been changed', 'wp-base' ), $changed ) );
		}
	}

	/**
	 * Helper function for render services table
	 */
	public function display_table( $services ) {

		$cols = apply_filters( 'app_services_allowed_columns',
				array( 'delete','id','sname','image','sort_order','internal','owner','location','category',
				'provider','capacity','duration','padding_before','padding_after','price','deposit','description_page' ) );

		$ret  = apply_filters( 'app_services_before_table', '<form id="add_services_form" class="app-form" method="post" >', $cols );
		$ret .= '<div style="overflow-x:auto;">';
		$ret .= '<table %HIDDENCOLUMNS%	id="services-table" data-by_worker="'.esc_attr( $this->get_by_worker() ).'" data-sel_location="'.esc_attr( $this->get_sel_location() ).'" data-app_paged="'.esc_attr( get_query_var( 'paged' ) ).'" class="app-list-table wp-list-table widefat app-services striped">';
		$ret .= '<thead>';

		/* hf : Common for header-footer */
		$this->colspan = 0;
		$hf = $this->table_head( $cols );

		$ret .= $hf. '</thead>';
		// Remove id from foot
		$ret .= '<tfoot>' . preg_replace( '/<th id="(.*?)"/is','<th ', $hf ). '</tfoot>';
		$ret = str_replace(
				'%HIDDENCOLUMNS%',
				'data-hide_boxes="'.implode( ',', array_diff( $this->get_allowed_columns(), $this->used_cols ) ).'"',
				$ret
			);

		$ret .= '<tbody>';
		$ret  = apply_filters( 'app_services_after_table', $ret, $cols );
		echo $ret;

		$ret = '';

		if ( $services ) {
			remove_filter( 'gettext', array( 'WpBCustomTexts', 'global_text_replace' ) );

			foreach ( $services as $r ) {

				$classes = array( 'app-tr' );

				$ret .= '<tr id="app-tr-'.$r->ID.'" class="'.implode( ' ', $classes ).'">';

				foreach( $cols as $col ) {

					if ( ! in_array( $col, $this->get_allowed_columns() ) ) {
						continue;
					}

					$hidden			= in_array( $col, $this->get_hidden_columns() ) ? " hidden" : "";
					$col_primary	= in_array( $col, array('sname' ) ) ? " column-primary" : "";
					$col_check		= in_array( $col, array( 'delete' ) ) ? " check-column" : "";

					$ret .= '<td class="column-'.$col. $hidden. $col_primary. $col_check. '">';
					$ret .= $this->table_cell( $col, $r, $cols );
					$ret .= '</td>';
				}
				$ret .= '</tr>';
			}
			echo $ret;

		} else {
			?>
			<tr class="alternate app-tr">
				<td colspan="<?php echo $this->colspan; ?>" scope="row"><?php _e( 'No matching records have been found.','wp-base' ); ?></td>
			</tr>
	<?php }	?>
		</tbody>
	</table>
</div>
	<?php
		do_action( 'app_admin_services_form' );
	?>
</form>
	<?php
	}

	/**
	 * Prepare header for services table
	 * @since 3.9
	 */
	private function table_head( $cols ) {

		$hf = '<tr>';
		foreach( $cols as $col ) {
			if ( ! in_array( $col, $this->get_allowed_columns() ) ) {
				continue;
			}

			$this->used_cols[] 	= $col;
			$hidden 			= in_array( $col, $this->get_hidden_columns() ) ? " hidden" : "";
			$col_primary 		= in_array( $col, array( 'sname' ) ) ? " column-primary" : "";
			$col_check 			= in_array( $col, array( 'delete' ) ) ? " check-column" : "";

			if ( ! $hidden ) {
				$this->colspan++;
			}

			$hf .= '<th id="'.$col.'" class="manage-column column-'.$col. $hidden. $col_primary. $col_check. '">';

			switch ($col) {
				case 'delete':		$hf .= '<input type="checkbox" class="app-no-save-alert" />';
									break;
				case 'id':			$hf .= __( 'ID', 'wp-base' );
									break;

				case 'sname':		$hf .= $this->a->get_text( 'name' );
									break;

				case $col:			$hf .= '<span>'. apply_filters( 'app_services_column_title', str_replace( ':', '', $this->a->get_text($col)), $col) .'</span>';
									break;
				default:			break;
			}
			$hf .= '</th>';

		}
		$hf .= '</tr>';

		return $hf;
	}

	/**
	 * Prepare a single cell in services table
	 * @since 3.9
	 */
	private function table_cell( $col, $r, $cols ) {
		$ret		= '';
		$service	= new WpB_Service( $r->ID );

		if ( ! $service->get_ID() ) {
			return '';
		}

		switch ( $col ) {
			case 'delete':			$ret .= self::can_edit( $service )
											?'<input type="checkbox" class="app-no-save-alert" name="app[]" value="'. $r->ID .'" />'
											: '';
									$ret .= '<div class="locked-indicator"><span class="locked-indicator-icon" aria-hidden="true"></span></div>';
									break;

			case 'created_by':		$ret .= self::created_by( $service ); break;
			case 'created_at':		$ret .= sprintf( __( 'Created at: %s', 'wp-base'), $service->get_created_at() ); break;
			case 'owner':			$ret .= $service->get_owner()
											? BASE()->get_worker_name( $service->get_owner() )
											: $this->a->get_text('admin');
											break;
			case 'id': 				$ret .= self::table_cell_ID( $service ); break;
			case 'image':			$ret .= self::table_cell_image( $service ); break;
			case 'sort_order':		$ret .= self::table_cell_sort_order( $service ); break;
			case 'sname':			$ret .= self::table_cell_service_name( $service ); break;
			case 'internal':		$ret .= $service->is_internal() ? __( 'Yes', 'wp-base' ) : __( 'No', 'wp-base' ); break;
			case 'location':		$ret .= self::table_cell_locations( $service ); break;
			case 'category': 		$ret .= self::table_cell_categories( $service ); break;
			case 'provider': 		$ret .= self::table_cell_providers( $service ); break;
			case 'capacity':		$ret .= $service->get_capacity(); break;
			case 'duration':		$ret .= wpb_format_duration( $service->get_duration() ); break;
			case 'padding_before':	$ret .= wpb_format_duration( $service->get_padding_before() ); break;
			case 'padding_after':	$ret .= wpb_format_duration( $service->get_padding_after() ); break;
			case 'price':			$ret .= wpb_format_currency( $service->get_price(), '', true ); break;
			case 'deposit':			$ret .= wpb_format_currency( $service->get_deposit(), '', true ); break;
			case 'description_page':$ret .= self::table_cell_page( $service ); break;

			default:				$ret .= apply_filters( 'app_services_add_cell', '', $col, $r, $cols ); break;
		}

		return $ret;
	}

	/**
	 * Prepare ID cell content
	 * @return string
	 */
	private static function table_cell_ID( $service ) {
		$marks	= apply_filters( 'app_admin_services_after_id', '', $service->get_ID(), $service );

		return '<span class="span_service_id">'.$service->get_ID() .'</span>'. $marks;
	}

	/**
	 * Prepare image cell content
	 * @return string
	 */
	private static function table_cell_image( $s ) {

		$ID			= $s->get_ID();
		$img_id		= $s->get_image_id() ?: 0;
		$pholder	= wpb_placeholder_img_src();
		$img_url	= $s->get_image_url() ?: '';
		$img_html	= $img_id
					  ? wp_get_attachment_image( $img_id, array(40,40), false, array( 'class' => 'app-main-avatar', 'srcset' => ' ', 'loading' => false, 'alt' => BASE()->get_service_name( $ID ) ) )
					  : '<img class="app-main-avatar" src="'.esc_attr($pholder).'" alt="Service" />';

		return $img_html;
	}

	/**
	 * Prepare sort order content
	 * @return string
	 */
	private static function table_cell_sort_order( $service ) {
		$filters	= array_filter( array_values( wpb_sanitize_search() ) );

		$cl			= ! empty( $_GET['app_filter_reset'] ) || ( empty( $filters ) && empty( $_GET['app_s'] ) && empty( $_GET['app_order_by'] ) )
					  ? 'app-sortable'
					  : '';
		$sort		= $service->get_sort_order();

		return '<span class="'.$cl.'" data-service_id="'.$service->get_ID().'" data-sort_order="'.$sort.'">'.$sort.'</span>';
	}

	/**
	 * Prepare service name cell content
	 * @return string
	 */
	private static function table_cell_service_name( $service ) {
		$lock_holder = self::check_lock( $service->get_ID() );

		if ( $lock_holder ) {
			$lock_holder   = get_userdata( $lock_holder );
			$locked_avatar = get_avatar( $lock_holder->ID, 18 );
			/* translators: %s: User's display name. */
			$locked_text = esc_html( sprintf( __( '%s is editing', 'wp-base' ), $lock_holder->display_name ) );
		} else {
			$locked_avatar = '';
			$locked_text   = '';
		}

		$ret = '<div class="locked-info"><span class="locked-avatar">' . $locked_avatar . '</span> <span class="locked-text">' . $locked_text . "</span></div>\n";

		$ret .= self::table_cell_service_inner( $service );

		return $ret;
	}

	/**
	 * Prepare service name cell content
	 * @return string
	 */
	private static function table_cell_service_inner( $service ) {

		$ret  = '<div class="service-inner">';
		$ret .= '<span class="app-client-name">';
		$ret .= '<a href="javascript:void(0)" class="app-inline-edit-service" title="'.esc_attr(__('Click to edit service','wp-base')).'">';
		$ret .= BASE()->get_service_name( $service->get_ID() );
		$ret .= '</a>';
		$ret .= '</span>';

		$ret = apply_filters( 'app_services_after_service_name', $ret, $service );

		$ret .= '</div>';

		return $ret;
	}

	/**
	 * Prepare locations cell content
	 * @return string
	 */
	private static function table_cell_locations( $service ) {
		$locs	= $service->get_locations();
		$count	= count( $locs );

		if ( $count > 6 ) {
			$slice	= array_slice( $locs, 0, 5 );
			$more	= sprintf( __( ' and %d more', 'wp-base' ), ($count - 5) );
		} else {
			$slice	= $locs;
			$more	= '';
		}

		return implode( ', ', array_map( array( BASE(), 'get_location_name' ), $slice ) ) . $more;
	}

	/**
	 * Prepare categories cell content
	 * @return string
	 */
	private static function table_cell_categories( $service ) {
		$cats	= $service->get_categories();
		$count	= count( $cats );

		if ( $count > 6 ) {
			$slice	= array_slice( $cats, 0, 5 );
			$more	= sprintf( __( ' and %d more','' ), ($count - 5) );
		} else {
			$slice	= $cats;
			$more	= '';
		}

		return implode( ', ', array_map( array( BASE(), 'get_category_name' ), $slice ) ) . $more;
	}

	/**
	 * Prepare providers cell content
	 * @return string
	 */
	private static function table_cell_providers( $service ) {
		$workers	= $service->get_workers();
		$count		= count( $workers );

		if ( $count > 6 ) {
			$slice	= array_slice( $workers, 0, 5 );
			$more	= sprintf( __( ' and %d more','' ), ($count - 5) );
		} else {
			$slice	= $workers;
			$more	= '';
		}

		return implode( ', ', array_map( array( BASE(), 'get_worker_name' ), $slice ) ) . $more;
	}

	/**
	 * Prepare page cell content
	 * @return string (URL)
	 */
	private static function table_cell_page( $service ) {
		$link = $service->get_page_permalink();
		$post = get_post( $service->get_page() );
		return (! empty( $post->post_title ) && $link ? "<a href='{$link}' target='_blank'>". $post->post_title ."</a>" : '');
	}

	/**
	 * Edit or create appointments on admin side
	 */
	public function inline_edit( $echo = false, $colspan = 0 ) {

		if ( ! $echo ) {
			if ( ! check_ajax_referer( 'inline_edit', 'ajax_nonce', false ) ) {
				die( json_encode( array( 'error' => $this->a->get_text('unauthorised') ) ) );
			}
		}

		$by_worker		= ! empty( $_POST['by_worker'] ) ? $_POST['by_worker'] : false;
		$sel_location	= ! empty( $_POST['sel_location'] ) ? $_POST['sel_location'] : false;

		# What if such a user does not exist?
		$service_id	= isset( $_REQUEST['service_id'] ) ? wpb_clean( $_REQUEST['service_id'] ) : 0;
		$service	= new WpB_Service( $service_id );
		$js_id		= $service_id ?: uniqid();
		$_colspan 	= ! empty( $_POST['col_len'] ) ? wpb_clean( $_POST['col_len'] ) : ( $colspan ? $colspan : 7 );

		$html  = '';
		$html .= '<tr class="inline-edit-row inline-edit-row-post quick-edit-row-post '.($service_id ? "" : "inline-edit-row-add-new").'">';
		$html .= '<td colspan="'.$_colspan.'" class="colspanchange">';

		$html .= '<fieldset>';
		$html .= '<div class="inline-edit-col">';

		$html .= $this->service_info_html( $service );

		$html .= '</fieldset>';
		$html .= '</div>';

		$html .= '<fieldset class="">';
		$html .= '<div class="inline-edit-col">';

		/* Name */
		$html .= wpb_wrap_field( 'service_name', __( 'Name', 'wp-base' ), apply_filters( 'app_admin_add_inline_field',
				'<input type="text" name="service_name" class="ptitle" value="'.esc_attr( $service->get_name() ).'" placeholder="'.esc_attr(__('Required','wp-base') ) .'"/>',
				$service->get_ID(), 'service_name' )
		);

		/* Locations */
		$html .= wpb_wrap_field( 'locations', __( 'Locations', 'wp-base' ),	$this->location_selection( $service ) );

		/* Categories */
		$html .= wpb_wrap_field( 'categories', __( 'Categories', 'wp-base' ), $this->category_selection( $service ) );

		/* Capacity */
		$html .= wpb_wrap_field( 'capacity', __( 'Capacity', 'wp-base' ),
				'<input type="text" name="capacity" class="ptitle" value="'.esc_attr( $service->get_capacity() ).'"/>',
				__( 'You can enter a value to increase number of time slots that can booked for any given time. If not set, or set lower than providers of the service, capacity is determined by the number of providers giving this service.', 'wp-base' )
		);

		/* Duration */
		$html .= wpb_wrap_field( 'duration', __( 'Duration', 'wp-base' ), $this->duration_selection( $service ) );

		/* Padding Before */
		$html .= wpb_wrap_field( 'padding_before', __( 'Padding Before', 'wp-base' ),
				$this->padding_selection( $service, 'before' ),
				__( 'Time needed to pass before another booking can be made for the service.', 'wp-base' )
		);

		/* Padding After */
		$html .= wpb_wrap_field( 'padding_after', __( 'Padding After', 'wp-base' ),
				$this->padding_selection( $service, 'after' ),
				__( 'Time needed to pass after another booking can be made for the service.', 'wp-base' )
		);

		/* Price */
		$html .= wpb_wrap_field( 'service_price', sprintf( __( 'Price (%s)', 'wp-base' ), wpb_format_currency( false ) ),
				'<input type="text" name="price" class="ptitle" value="'.esc_attr( $service->get_price() ).'"/>',
				__( 'Price for the service per time slot per person.', 'wp-base' )
		);

		/* Deposit */
		$html .= wpb_wrap_field( 'service_deposit', sprintf( __( 'Security Deposit (%s)', 'wp-base' ), wpb_format_currency( false ) ),
				'<input type="text" name="deposit" class="ptitle" value="'.esc_attr( $service->get_deposit() ).'"/>'.
				'<input type="hidden" name="deposit_check" value="1" />',
				__( 'Refundable security deposit. Not to be confused with advance payment/down payment which is deducted from the price.', 'wp-base' )
		);

		/* Description Page */
		$html .= wpb_wrap_field( 'description_page', __( 'Description Page', 'wp-base' ),
				$this->description_page_selection( $service ),
				__( 'Optional description post/page of the service. If selected, description and feature image will be taken from the post/page instead of the Description and Featured Image fields here.', 'wp-base' )
		);

		$html .= '</div>';
		$html .= '</fieldset>';

		/* Add More tabs */
		if ( $service->get_ID() ) {

			$tabs = apply_filters( 'app_admin_services_add_more_tab', array() );

			if ( ! empty( $tabs ) ) {
				$html .= '<fieldset><table class="app-service-addon-settings"><tbody class="service-tabs-container">';
				if ( count($tabs) > 1 ) {
					$html .= "<tr class='app-service-tab-head' data-service-id='service-{$service_id}' >";
					$html .= '<td colspan="'.($this->nof_columns()-2).'">';
					$html .= '<ul>';
					foreach ( $tabs as $tab => $name ) {
						# $tab is class_name and $n is service ID
						$html .= "<li><a href='#{$tab}_{$service_id}'><span class='tab-title'>{$name}</span></a></li>";
					}
					$html .= '</ul>';
					$html .= '</td>';
					$html .= '</tr>';
				}
			}

			$html = apply_filters( 'app_admin_services_after_tr', $html, $service_id, $service );

			if ( ! empty( $tabs ) ) {
				$html .= '</tbody></table></fieldset>';
			}
		}

	/* SAVE and OTHER ACTIONS */
		$html .= '<fieldset><div class="inline-edit-save app-clearfix">';
		$html .= '<input type="hidden" name="by_worker" value="'.$by_worker.'" />';
		$html .= '<input type="hidden" name="sel_location" value="'.$sel_location.'" />';
		$html .= '<input type="hidden" name="service_id" value="'.$service_id.'" />';
		$html .= '<a href="javascript:void(0)" title="'.__('Cancel', 'wp-base').'" class="button-secondary cancel alignleft">'.__('Cancel','wp-base').'</a>';
		$html .= '<img class="waiting alignleft" style="display:none;" src="'.admin_url('images/wpspin_light.gif').'" alt="">';
		$html .= '<span class="error alignleft" style="display:none"></span>';
		
		$html  = apply_filters( 'app_admin_services_before_save_button', $html, $service_id, $service );
		
		$html .= '<button class="button-primary save alignright'. (! $service_id || self::can_edit( $service ) ? '' : ' disabled') .'">';
		$html .= $service_id
				 ? ( self::can_edit( $service ) ? __('Update','wp-base') : __('Cannot Update','wp-base') )
				 : __('Add New Service','wp-base');
		$html .= '</button>';
		
		$html  = apply_filters( 'app_admin_services_after_save_button', $html, $service_id, $service );		
		
		$html .= '</div></fieldset>';

		$html .= '</td>';
		$html .= '</tr>';

		if ( $echo ) {
			echo $html;
		} else {
			die( json_encode( array(
				'result'	=> $html,
				'id'		=> $js_id,
			) ) );
		}
	}

	/**
	 * Add Owner select element
	 * @param $s	object		WpB_Service object
	 * @return string
	 */
	public static function select_owner( $s ) {
		$ID	 = $s->get_ID();

		$h = '<select name="owner">';
		$h .= '<option value="">'. esc_attr( BASE()->get_text('admin') ) .'</option>';

		foreach( BASE()->get_workers( 'name' ) as $worker ) {
			$name = BASE()->get_worker_name( $worker->ID );
			$h .= '<option value="'.$worker->ID.'" '.selected( $worker->ID, $s->get_owner(), false ).'>'. esc_attr( $name )	.'</option>';
		}

		$h .= '</select>';
		$h .= '<input type="hidden" name="owner_check" value="1">';

		return $h;
	}

	/**
	 * Add locations select/option element
	 */
	public function location_selection( $service ) {
		$html = '';
		if ( $locations = $this->a->get_locations() ) {

			$html .= '<select multiple class="add_location_multiple app-ms-small app_ms_tab" data-buttonWidth="100%" name="app_locations[]">';
			foreach ( $locations as $location ) {
				$s = in_array( $location->ID, $service->get_locations() ) ? 'selected="selected"' : '';
				$html .= '<option value="'. $location->ID . '" '.$s.'>'. esc_attr( $this->a->get_location_name( $location->ID ) ). '</option>';
			}
			$html .= '</select>';
			$html .= '<input type="hidden" name="app_locations_check" value="1" />';
		} else {
			$html .= __( 'None defined', 'wp-base' );
		}

		return $html;
	}

	/**
	 * Add categories cell to add service function
	 * @return string
	 */
	public function category_selection( $service ) {
		$html = '';
		if ( $categories = $this->a->get_categories() ) {

			$html .= '<select multiple class="add_category_multiple app-ms-small app_ms_tab" data-buttonWidth="100%" name="app_categories[]">';

			foreach ( $categories as $cat_id => $category ) {
				$s = in_array( $cat_id, $service->get_categories() ) ? 'selected="selected"' : '';
				$html .= '<option value="'. $cat_id . '" '.$s.'>'. esc_html( stripslashes( $category['name'] ) ) . '</option>';
			}

			$html .= '</select>';
			$html .= '<input type="hidden" name="app_categories_check" value="1" />';
		} else {
			$html .= __( 'None defined', 'wp-base' );
		}

		return $html;
	}

	/**
     * Helper to create duration select menu
	 * @since 3.9
     */
	private function duration_selection( $service ) {
		$min_time	= $this->a->get_min_time();
		$selections	= wpb_service_duration_selections();
		$duration	= $service->get_duration();
		$class		= ( $duration % $min_time != 0 || $duration > max( $selections ) ) ? 'class="error"': '';

		$html = '<select '.$class.' name="duration" >';

		foreach ( $selections as $val ) {
			$text = $val == 1440 ? __( 'All day', 'wp-base' ) : wpb_format_duration( $val );

			$html .= '<option value="'.$val.'"'. selected( $val, $duration, false ).'>'. esc_attr( $text ) . '</option>';
		}
		$html .= '</select>';

		return $html;
	}

	/**
     * Helper to create padding select menu
	 * @param $context		string		before or after
     */
	private function padding_selection( $service, $context ) {
		if ( 'before' === $context ) {
			$padding = $service->get_padding_before();
		} else if ( 'after' === $context ) {
			$padding = $service->get_padding_after();
		}

		$min_time	= $this->a->get_min_time();
		$k_max		= apply_filters( 'app_padding_selection_max', (int)(1440/$min_time), $service, $context );
		$class		= ($padding % $min_time != 0) ? 'class="error"': '';

		$html = '<select '.$class.' name="padding_'.$context.'">';

		for ( $k = 0; $k <= $k_max; $k++ ) {
			$val = $k * $min_time;
			$html .= '<option value="'.$val.'"'.selected( $padding, $val, false ).'>'. wpb_format_duration( $val ) . '</option>';
		}
		$html .= '</select>';

		# Add only once
		if ( 'before' == $context ) {
			$html .= '<input type="hidden" name="padding_check" value="1" />';
		}

		return $html;
	}

	private function description_page_selection( $service ) {
		$pages = wpb_get_description_pages( wpb_setting( 'description_post_type', 'page' ) );

		$html  = '<select class="app-page-selection" name="description_page">';
		$html .= '<option value="0">'. __('None','wp-base') .'</option>';

		foreach( (array)$pages as $page ) {
			$s = $service->get_page() == $page->ID ? ' selected="selected"' : '';
			$html .= '<option value="'.$page->ID.'"'.$s.'>'. esc_html( $page->post_title ) . '</option>';
		}

		$html  = apply_filters( 'app_admin_services_after_description_page_options', $html, $service->get_ID(), $service );
		$html .= '</select>';
		$html .= '<input type="hidden" name="description_page_check" value="1" />';

		return $html;
	}

	/**
	 * Vendor avatar, ID, etc info
	 */
	private function service_info_html( $service ){
		$info = self::service_info( $service );
		ob_start();
?>
<div class="app-service-info">

	<div class="app-service-main left">
		<span class="app-service-item app-service-name"><?php echo ($service->get_ID() ? $info['name'] : __( 'NEW SERVICE', 'wp-base' )); ?></span>

	<?php if ( $service->get_ID() ) { ?>
		<?php if ( ! self::can_edit( $service ) ) { ?>
			<span class="app-service-item app-service-readonly" title="<?php echo __( 'You are not the owner or admin. Changes will not be saved.', 'wp-base' ) ?>"><?php echo __( '(Read Only)', 'wp-base' ) ?></span>
		<?php }	else { ?>
				<span class="app-service-item app-created-at"><?php echo $info['created-at']; ?></span>
				<span class="app-service-item app-created-by"><?php echo $info['created-by']; ?></span>
		<?php } ?>
				<span class="app-service-item app-desc-page"><?php echo $info['desc-page']; ?></span>
	<?php }

		/* Owner */
		if ( current_user_can( WPB_ADMIN_CAP ) ) {
			echo wpb_wrap_field( 'owner', __('Owner', 'wp-base'), self::select_owner( $service ), __( 'Owner can edit the service. Only admin can change owners.', 'wp-base' ) );
		} else if ( $service->get_ID() ) {
			$owner = $service->get_owner();
			if ( $owner == get_current_user_id() ) {
				$text = __( 'You','wp-base' );
			} else if ( $owner ) {
				$text = __( 'Another service provider','wp-base' );
			} else {
				$text = $this->a->get_text('admin');
			}
			echo wpb_wrap_field( 'owner', __('Owner', 'wp-base'), '<p><code>'. esc_html( $text ) .'</code></p>' );
		}

		/* Internal */
		echo wpb_wrap_field( 'internal', __( 'Internal', 'wp-base' ),
				'<input type="checkbox" name="internal" '.checked( $service->is_internal(), true, false).' value="1"/>'.
				'<input type="hidden" name="internal_check" value="1"/>',
				__( 'Similar to WordPress private posts. Internal services will not be displayed in service selection menu element at the front end.', 'wp-base' ) );


?>
	</div>
<?php
		/* Description */
		echo wpb_wrap_field( 'description', __('Description', 'wp-base'), apply_filters( 'app_admin_add_inline_field',
		'<textarea rows="10" name="service_description">'. esc_textarea( ($service->get_description() ?: '') ) .'</textarea>'.
		'<input type="hidden" name="service_description_check" value="1"/>', $service->get_ID(), 'service_description', 'textarea' ),
		__( 'Optional description text for the service. Alternatively you can assign a Description Page.', 'wp-base' ),
		'style="float:left"' );

		/* Workers */
		if ( apply_filters( 'app_service_inline_edit_show_workers', true, $service ) && $this->a->get_nof_workers() ) {
			$workers	= $service->get_workers();
			$count		= count( $workers );

			if ( $count > 30 ) {
				$slice	= array_slice( $workers, 0, 30 );
				$more	= sprintf( __( ' and %d more','' ), ($count - 30) );
			} else {
				$slice	= $workers;
				$more	= '';
			}

			$text = $slice ? implode( ', ', array_map( array( BASE(), 'get_worker_name' ), $slice ) ) . $more : '';

			echo wpb_wrap_field( 'providers', __('Service Providers', 'wp-base'),
			'<textarea rows="6" readonly>'. esc_textarea( $text ) .'</textarea>',
			__( 'Readonly value showing providers giving this service. To edit, use Providers page instead.', 'wp-base' ),
			'style="float:left"' );
		}

	if ( apply_filters( 'app_service_inline_edit_show_image', true, $service ) ) { ?>
		<div class="app-service-avatar left"><?php echo $info['image-html']; ?></div>
	<?php } ?>

	<div class="app-service-id right"><?php echo $info['id']; ?></div>

</div>
<?php
		return ob_get_clean();
	}

	/**
	 * Details about service
	*/
	private static function service_info( $service ) {
		return array(
			'id'			=> $service->get_ID() ? '#'. $service->get_ID() : '',
			'service_id'	=> $service->get_ID(),
			'image-html'	=> self::image_html( $service ),
			'name'			=> $service->get_name(),
			'created-at'	=> sprintf( __( 'Created at: %s', 'wp-base'), $service->get_created_at() ),
			'created-by'	=> self::created_by( $service ),
			'desc-page'		=> $service->get_page() ? '<a href="'. get_permalink( $service->get_page() ) .'" target="_blank">' . __( 'Description Page', 'wp-base' ) .'</a>' : '',
		);
	}

	/**
	 * Featured image
	*/
	public static function image_html( $s ) {
		$ID			= $s->get_ID();
		$img_id		= $s->get_image_id() ?: 0;
		$pholder	= wpb_placeholder_img_src();
		$img_url	= $s->get_image_url() ?: '';
		$img_html	= $img_id
					  ? wp_get_attachment_image( $img_id, array(144,144), false, array( 'class' => 'app-main-avatar', 'srcset' => ' ', 'loading' => false, 'alt' => BASE()->get_service_name( $ID ) ) )
					  : '<img class="app-main-avatar" src="'.esc_attr($pholder).'" alt="Service" />';

		return wpb_wrap_field( 'image', __('Featured Image', 'wp-base'),
				'<div class="app-image-upload lp-border-bottom padding-bottom-45">
					<div class="image-preview">
						<span class="dashicons dashicons-trash app-remove-image'.(!$img_url ? ' has-pholder' : '').'" title="'.esc_attr( __( 'Delete', 'wp-base' ) ).'"></span>
						'. $img_html.'
						<input class="pholder_img" type="hidden" value="'.esc_attr($pholder).'">
					</div>
					<div class="image-description">
						<div class="upload-photo margin-top-25">
							<span class="file-input file-upload-btn btn-first-hover btn-file">
								<input data-content="'.BASE()->get_text('choose_image').'" class="app-upload-image-btn" data-author="'.(current_user_can( WPB_ADMIN_CAP ) ? 0 : get_current_user_id()).'" type="file">
							</span>
							<input class="criteria-image-url" type="hidden" name="service_image_url" value="'.esc_attr($img_url).'">
							<input class="criteria-image-id" type="hidden" name="service_image_id" value="'.esc_attr($img_id).'">
							<input type="hidden" name="service_image_check" value="1"/>
						</div>
					</div>
				</div>',
				__( 'Optional Featured Image for the service which will be used in service selection slider. Alternatively you can assign a Description Page.', 'wp-base' )
			);

	}

	/**
	 * Return created by in user readable format
	 * @param $s	object		WpB_Service object
	 * @return string
	 */
	public static function created_by( $s ) {
		$created_by	= '';
		if ( $maybe_created_by = $s->get_created_by() ) {
			if ( $uid = get_user_by( 'ID', $maybe_created_by ) ) {
				$created_by = $uid->display_name;
			}
		}

		return sprintf( __( 'Created by: %s', 'wp-base' ), ($created_by ?: BASE()->get_text( 'unknown' ) ) );
	}

	/**
	 * Save edited or new appointment on admin side
	 * @uses global variable $wpb_setting_saved set in wpb_notice function
	 * @return json
	 */
	public function inline_edit_save() {

		if ( ! check_ajax_referer( 'inline_edit', 'ajax_nonce', false ) ) {
			die( json_encode( array('error' => $this->a->get_text('unauthorised') ) ) );
		}

		if ( wpb_is_demo() ) {
			die( json_encode( array( 'error' => __('Changes cannot be saved in DEMO mode!', 'wp-base' ) ) ) );
		}

		$by_worker = ! empty( $_REQUEST['by_worker'] ) ? $_REQUEST['by_worker'] : false;

		if ( ! self::check_by_worker( $by_worker ) ) {
			die( json_encode( array('error' => $this->a->get_text('unauthorised') ) ) );
		}

		$service 		= new WpB_Service( wpb_clean( $_POST['service_id'] ) );
		$old_service	= clone $service;

		if ( $service->get_ID() && ! self::can_edit( $service ) ) {
			die( json_encode( array('error' => $this->a->get_text('unauthorised') ) ) );
		}

		if ( $service->get_ID() && $locked_by_id = self::check_lock( $service->get_ID() ) ) {
			$locked_by	= get_userdata( $locked_by_id );
			$name		= isset( $locked_by->display_name ) ? $locked_by->display_name : $locked_by_id;
			die( json_encode( array( 'error' => sprintf( __('Record has been taken over by %s. It could not be saved.', 'wp-base' ), $name ) ) ) );
		}

		if ( isset( $_POST['internal_check'] ) ) {
			$service->set_internal( ! empty( $_POST['internal'] ) ? 1 : 0 );
		}

		$service->set_name( ! empty( $_POST['service_name'] ) ? wpb_clean( $_POST['service_name'] ) : '' );

		if ( isset( $_POST['app_locations_check'] ) ) {
			$service->set_locations( ! empty( $_POST['app_locations'] ) ? explode( ',', wpb_clean( $_POST['app_locations'] ) ) : array() );
		}

		if ( isset( $_POST['app_categories_check'] ) ) {
			$service->set_categories( ! empty( $_POST['app_categories'] ) ? explode( ',', wpb_clean( $_POST['app_categories'] ) ) : array() );
		}

		$service->set_capacity( ! empty( $_POST['capacity'] ) ? preg_replace( '/[^0-9]/', '', $_POST['capacity'] ) :'' );

		$service->set_duration( preg_replace( '/[^0-9]/', '', $_POST['duration'] ) );

		if ( isset( $_POST['padding_check'] ) ) {
			$service->set_padding_before( ! empty( $_POST['padding_before'] ) ? wpb_clean( $_POST['padding_before'] ) : '' );
			$service->set_padding_after( ! empty( $_POST['padding_after'] ) ? wpb_clean( $_POST['padding_after'] ) : '' );
		}

		$service->set_price( ! empty( $_POST['price'] ) ? wpb_sanitize_price( $_POST['price'] ) :'' );

		if ( isset( $_POST['deposit_check'] ) ) {
			$service->set_deposit( ! empty( $_POST['deposit'] ) ? wpb_sanitize_price( $_POST['deposit'] ) :'' );
		}

		$service->set_page( ! empty( $_POST['description_page'] ) ? intval( $_POST['description_page'] ) : 0 );


		if ( ! $service->get_ID() ) {
			$max_sort = (int)$this->a->db->get_var( "SELECT MAX(sort_order) FROM " . $this->table );
			$service->set_sort_order( $max_sort + 1 );
		}

		$service = apply_filters( 'app_service_inline_edit_save_data', $service, $old_service, $this );

		$update_result = $insert_result = null;
		$changed = false;

		# Do save
		if ( $service->get_ID() ) {
			$update_result = $service->save();
			do_action( 'app_service_maybe_updated', $service->get_ID(), $update_result );
		} else	if ( $insert_result = $service->insert() ) {
			BASE('WH')->add_default( $service->get_ID(), 'service' );
			$service->add_created_by();
			$service->add_owner();
			$service->add_created_at();
			$service->add_managed_by();

			if ( $by_worker ) {
				$this->assign2owner( $service->get_ID() );
				do_action( 'app_service_added_by_worker', $service->get_ID(), $by_worker );
			}
		}

		global $wpb_setting_saved;

		if ( $wpb_setting_saved ) {
			$changed = true;
		}

		# Add a default name for empty name submission
		if ( $service->get_ID() && '' == trim( $service->get_name() ) ) {
			$service->set_name( sprintf( __( 'Service %d', 'wp-base' ), $service->get_ID() ) );
			if ( $service->save() ) {
				$changed = true;
			}
		}

		if ( isset( $_POST['owner_check'] ) ) {
			if ( empty( $_POST['owner'] ) ) {
				if ( $service->delete_owner() ) {
					$changed = true;
				}
			} else if ( $service->update_owner( wpb_clean( $_POST['owner'] ) ) ) {
				$changed = true;
			}
		}

		if ( isset( $_POST['service_description_check'] ) ) {
			if ( $service->update_description( ! empty( $_POST['service_description'] ) ? wp_kses_post( $_POST['service_description'] ) : '' ) ) {
				$changed = true;
			}
		}

		if ( isset( $_POST['service_image_check'] ) ) {
			if ( $service->update_image_url( ! empty( $_POST['service_image_url']) ? wpb_clean( trim( $_POST['service_image_url'] ) ) : '' ) ) {
				$changed = true;
			}
		}

		if ( isset( $_POST['service_image_check'] ) ) {
			if ( $service->update_image_id( ! empty( $_POST['service_image_id'] ) ? wpb_clean( trim( $_POST['service_image_id'] ) ) : '' ) ) {
				$changed = true;
			}
		}

		$changed = apply_filters( 'app_service_inline_edit_save', $changed, $service, $old_service, $this );

		if ( $update_result ) {
			do_action( 'app_service_inline_edit_updated', $service, $old_service, $this );
		} else if ( $insert_result ) {
			do_action( 'app_service_inline_edit_new_service', $service, $old_service, $this );
			do_action( 'app_new_service_added', $service->get_ID() );
		}

		wpb_flush_cache( true );

		$new_results = apply_filters( 'app_service_inline_edit_save_result', array(
				'id'				=> self::table_cell_ID( $service ),
				'result_service_id'	=> $service->get_ID(),
				'created_by'		=> self::created_by( $service ),
				'created_at'		=> sprintf( __( 'Created at: %s', 'wp-base'), $service->get_created_at() ),
				'internal'			=> $service->is_internal() ? __( 'Yes', 'wp-base' ) : __( 'No', 'wp-base' ),
				'owner'				=> $service->get_owner() ? BASE()->get_worker_name( $service->get_owner() ) : $this->a->get_text('admin'),
				'image'				=> self::table_cell_image( $service ),
				'service_name'		=> self::table_cell_service_inner( $service ),
				'location'			=> self::table_cell_locations( $service ),
				'category'			=> self::table_cell_categories( $service ),
				'capacity'			=> $service->get_capacity(),
				'duration'			=> wpb_format_duration( $service->get_duration() ),
				'padding_before'	=> wpb_format_duration( $service->get_padding_before() ),
				'padding_after'		=> wpb_format_duration( $service->get_padding_after() ),
				'price'				=> wpb_format_currency( $service->get_price(), false, true ),
				'deposit'			=> wpb_format_currency( $service->get_deposit(), false, true ),
				'description_page'	=> self::table_cell_page( $service ),
				'collapse'			=> $by_worker || 'yes' == wpb_setting( 'admin_edit_collapse' ) ? 1 : 0,
				'new_service'		=> $insert_result,
		), $service, $old_service );

		if ( $update_result === false || $insert_result === false ) {
			$result = array( 'result' => __('Record could not be saved!', 'wp-base' ) );
		} else if ( $insert_result ) {
			$result = array_merge( array( 'result' => __('New service successfully saved.', 'wp-base' ) ), $new_results );
		} else if ( $update_result || $changed ) {
			$result = array_merge( array( 'result' => __('Changes saved.', 'wp-base' ) ), $new_results );
		} else {
			$result = array( 'no_change' => __('You did not make any changes...', 'wp-base' ) );
		}

		if ( $update_result || $insert_result || $changed ) {
			$service->update_meta( '_edit_last', $this->a->_time .':'. get_current_user_id() );
			$service->delete_meta( '_edit_lock' );
			update_user_meta( get_current_user_id(), 'app_service_check_needed',  true );

			# Calculate min time
			wpb_flush_cache();
			$options = wpb_setting();
			$options['calc_min_time'] = $this->a->find_optimum_time_base();
			$this->a->update_options( $options );
		}

		$result = apply_filters( 'app_service_inline_edit_save_result_final', $result, $service, $old_service );

		die( json_encode( $result ) );

	}

	/**
	 * Assign service to worker who also submitted it
	 */
	private function assign2owner( $ID ) {
		$current_uid = get_current_user_id();

		$sp = $this->a->db->get_var( $this->a->db->prepare(
				"SELECT services_provided FROM " . $this->a->workers_table . " WHERE ID=%d",
				$current_uid
		) );

		$sp_arr = wpb_explode( $sp );
		$sp_arr[] = $ID;

		return $this->a->db->update( $this->a->workers_table,
					array( 'services_provided'	=> wpb_implode( $sp_arr ) ),
					array( 'ID' 				=> $current_uid )
		);
	}

	/**
	 * Start locking when a service selection is opened
	 * @since 3.6.0
	 * @return	json response
	 */
	public function start_lock(){
		$sid = ! empty( $_POST['service_id'] ) ? wpb_clean( $_POST['service_id'] ) : 0;

		if ( $sid ) {
			wpb_delete_service_meta( $sid, '_edit_lock' );
			$this->set_lock( $sid );
		}

		wp_send_json_success();
	}

	/**
	 * Set lock for the service record
	 * @param $sid		integer		Service ID
	 * @since 3.6.0
	 * @return	string
	 */
	private function set_lock( $sid ) {
		if ( ! $sid ) {
			return false;
		}

		$user_id = get_current_user_id();
		if ( 0 == $user_id ) {
			return false;
		}

		$now  = $this->a->_time;
		$lock = "$now:$user_id";

		wpb_update_service_meta( $sid, '_edit_lock', $lock );

		return array( $now, $user_id );
	}

	/**
	 * Check if service record is locked
	 * @param $sid		integer		Service ID
	 * @since 3.6.0
	 * @return int|false|null   ID of the user with lock. False if the post does not exist, post is not locked,
	 *                   		the user with lock does not exist. null if the post is locked by current user.
 	 */
	private static function check_lock( $sid ) {
		if ( ! $sid ) {
			return false;
		}

		$lock = wpb_get_service_meta( $sid, '_edit_lock' );

		if ( ! $lock ) {
			return false;
		}

		$lock = explode( ':', $lock );
		$time = $lock[0];
		if ( isset( $lock[1] ) ) {
			$user = $lock[1];
		} else if ( $maybe_elast =  wpb_get_service_meta( $sid, '_edit_last' ) ) {
			$maybe_elast = explode( ':', $maybe_elast );
			$user = isset( $maybe_elast[1] ) ? $maybe_elast[1] : 0;
		} else {
			$user = 0;
		}

		if ( ! get_userdata( $user ) ) {
			return false;
		}

		$time_window = apply_filters( 'wp_check_post_lock_window', 30 );
		$time_window = apply_filters( 'app_check_service_lock_window', $time_window );

		if ( $time && $time > BASE()->_time - $time_window && $user != get_current_user_id() ) {
			return $user;
		}

		if ( $user == get_current_user_id() ) {
			return null;
		}

		return false;
	}

	/**
	 * Check lock status on the service record and refresh the lock
	 *
	 * @since 3.6.0
	 *
	 * @param array  $response  The Heartbeat response.
	 * @param array  $data      The $_POST data sent.
	 * @param string $screen_id The screen id.
	 * @return array The Heartbeat response.
	 */
	public function refresh_lock( $response, $data, $screen_id ) {
		if ( ! array_key_exists( 'app-check-locked-service', $data ) ) {
			return $response;
		}

		$received = $data['app-check-locked-service'];
		$checked     = array();

		if ( empty ( $received ) || !is_array( $received ) ) {
			return $response;
		}

		foreach ( $received as $key ) {

			$sid = absint( substr( $key, 15 ) );

			$user_id = self::check_lock( $sid );
			$user    = get_userdata( $user_id );

			if ( null === $user_id ) {
				$editing = ! empty( $data['app-refresh-locked-service'] ) ? $data['app-refresh-locked-service'] : array();
				if ( in_array( 'app-tr-'. $sid, $editing ) ) {
					if ( $new_lock = $this->set_lock( $sid ) ) {
						$checked[$sid]['new_lock'] = implode( ':', $new_lock );
					}
				}
			} else if ( $user ) {
				$send = array(
					/* translators: %s: User's display name. */
					'text' => sprintf( __( '%s is editing', 'wp-base' ), $user->display_name ),
				);

				$avatar = get_avatar( $user->ID, 18 );
				if ( $avatar ) {
					if ( preg_match( "|src='([^']+)'|", $avatar, $matches ) ) {
						$send['avatar_src'] = $matches[1];
					}
				}

				$checked[$sid] = $send;
			}

			$response['app-check-locked-service'] = $checked;
		}

		return $response;
	}

	/**
	 * Delete lock when a record is closed
	 * @since 3.6.0
	 */
	public function delete_lock() {
		if ( ! check_ajax_referer( 'inline_edit', 'ajax_nonce', false ) ) {
			die( json_encode( array( 'error' => $this->a->get_text('unauthorised') ) ) );
		}

		$sid = ! empty( $_POST['service_id'] ) ? wpb_clean( $_POST['service_id'] ) : 0;
		$user_id = self::check_lock( $sid );
		if ( null === $user_id ) {
			wpb_delete_service_meta( $sid, '_edit_lock' );
		}

		wp_send_json_success();
	}

	/**
	 * Add script to footer;
	 */
	public function footer(){
?>
<script type="text/javascript">
jQuery(document).ready(function($){

    /**
     * Add/edit/save services
     */
    var WPB_Services = {

        init: function () {
            var me = this;
			me.tbl = $("table.app-services");
            me.doBind();
        },

        doBind: function () {
            var me = this;
            var table = $("table.app-services");
			var navbar = $(document).find("#app-navbar");
			var openNavbar = $(document).find("#app-open-navbar");

			$("#app-open-navbar button").click(function(e){
				e.preventDefault();
				openNavbar.hide();
				navbar.show( "fast" );
				me.saveNavbar( false );
			});

			$("#app-close-navbar button").click(function(e){
				e.preventDefault();
				navbar.hide( "fast" );
				openNavbar.show();
				me.saveNavbar( true );
			});

            if (jQuery.ui.sortable !== undefined) {
                me.doSortable();
            }

			$("#change_status_service").click(function (e) {
                me.deleteCheck(e);
            });

            $(".add-new-service").on("click", function () {
                me.addNew($(this));
            });

             table.on("click", ".app-inline-edit-service", function (e) {
                me.edit(e);
            });

			table.on("click", ".save", function (e) {
                me.save(e);
            });

            /* Cancel */
            table.on("click", ".cancel", function (e) {
                me.collapseRecord($(e.target));
            });

			/* Info Buttons */
            table.on("click", "a.info-button", function (e) {
                var $this = $(this);
                var cName = $this.data("boxname") ? $this.data("boxname") : "app-instructions";
                if ($this.hasClass("addon-info-button")) {
                    var par = $this.parent(".addon-set-info");
                    if ($this.hasClass("app-fade")) {
                        $this.removeClass("app-fade");
                        par.find(".clone").remove();
                    } else {
                        $this.addClass("app-fade");
                        $(document).find("." + cName).first().clone().addClass("clone").appendTo(par).show();
                    }

                } else {
                    $("." + cName).toggle('fast');
                }
            });

            /* Redraw multiselect */
            $(window).resize(function () {
                $.each($(".app_service_locations, .app_service_categories, .app_displayed_columns"), function (ignore, val) {
                    var $this = $(val);
                    if ($this.data().hasOwnProperty("echMultiselect")) {
                        $this.multiselect("refresh");
                    }
                });
            });

			if (!table.length) {
				return false;
			}

			table.on( "keyup", "input[name='service_name']", function(){
				var val = $(this).val();
				if (val) {
					$(this).parents(".inline-edit-row").find(".app-service-name").text(val);
				}
			});

         },

        /**
         * Save navbar position
         */
        saveNavbar: function (hidden) {
			var data = {
				wpb_ajax: true,
				action: "app_save_navbar_service",
				hidden: hidden ? 1 :0,
				ajax_nonce: _app_.iedit_nonce
			};
			$.post(_app_.ajax_url, data, function (r) {
				if (r && r.error) {
					window.alert(r.error);
				} else if (r) {
				} else {
					window.alert(_app_.con_error);
				}
			}, "json");
		},

        /**
         * Ask confirm before deleting service
         */
        deleteCheck: function (e) {
            e.preventDefault();

			var form = $("#app-bulk-service-change-form");
			var sel = form.find("select[name='app_new_status']").val();

			if ( sel != "delete") {
				form.submit();
				return;
			}

            var r = confirm(_app_.confirmDelete);
            if (r) {
				form.submit();
            }
            return false;
        },

        /**
         * Read all field values of a row (par)
         * @param par   object  Possibly a row in the table under which fields will be read
         */
        readData: function (par) {
            var table = par.parents("table.app-services");
            var locations = par.find(".add_location_multiple").val();
			var categories = par.find(".add_category_multiple").val();

            var postData = {
				wpb_ajax: true,
				ajax_nonce: _app_.iedit_nonce,
				internal: par.find("input[name='internal']").is(":checked") ? 1 : 0,
				internal_check: par.find("input[name='internal_check']").val(),
				owner: par.find("select[name='owner']").val(),
				owner_check: par.find("input[name='owner_check']").val(),
				service_description: par.find("textarea[name='service_description']").val(),
				service_description_check: par.find("input[name='service_description_check']").val(),
				service_image_url: par.find("input[name='service_image_url']").val(),
				service_image_id: par.find("input[name='service_image_id']").val(),
				service_image_check: par.find("input[name='service_image_check']").val(),
				service_id: par.find("input[name='service_id']").val(),
				service_name: par.find("input[name='service_name']").val(),
				app_locations: locations ? locations.join(",") : "",
				app_locations_check: par.find("input[name='app_locations_check']").val(),
				app_categories: categories ? categories.join(",") : "",
				app_categories_check: par.find("input[name='app_categories_check']").val(),
				capacity: par.find("input[name='capacity']").val(),
				duration: par.find("select[name='duration']").val(),
				padding_before: par.find("select[name='padding_before']").val(),
				padding_after: par.find("select[name='padding_after']").val(),
				padding_check: par.find("input[name='padding_check']").val(),
				price: par.find("input[name='price']").val(),
				deposit: par.find("input[name='deposit']").val(),
				deposit_check: par.find("input[name='deposit_check']").val(),
				description_page: par.find("select[name='description_page']").val(),
				description_page_check: par.find("input[name='description_page_check']").val(),
				by_worker: par.find("input[name='by_worker']").val(),
				sel_location: par.find("input[name='sel_location']").val(),
				app_lang: $.urlParam("app_lang")
            };

			/* Add addon values to post data */
			var postSpecialData = [];
			var $this;

			$.merge(par.find(".app-service-addon-settings").find("input, select"), par.find("input.app-multilang, textarea.app-multilang")).each( function(){
				$this = $(this);
				var isCheckbox = !!($this.prop("type") === "checkbox");

				if ( isCheckbox && !$this.is(":checked") ) {
					return true;
				}

				var sp_name = $this.attr("name");
				var sp_value = isCheckbox ? 1 : $this.val();
				postSpecialData[sp_name] = sp_value;
			});

			$.extend(postData, postSpecialData);

            return postData;
        },

        /**
         * Add a new service record/row
         */
        addNew: function ($this) {
            var me = this;
            $(".add-new-waiting").show();
            $.infoPanel("reading");
            var table = $("table.app-services").first();
            var data = {
                wpb_ajax: true,
                action: "app_inline_edit_service",
                add_new: $.readGet("add_new"),
                col_len: parseInt(table.find("tr:first th:visible").length),
				by_worker: table.data("by_worker"),
				sel_location: table.data("sel_location"),
                service_id: 0,
				app_lang: $.urlParam("app_lang"),
                ajax_nonce: _app_.iedit_nonce
            };
            $.post(_app_.ajax_url, data, function (r) {
                $(".add-new-waiting").hide();
                if (r && r.error) {
                    window.alert(r.error);
                } else if (r) {
                    table.prepend(r.result);
                    me.configureMS(r.id);
                    me.configQtip(r.id);
                } else {
                    window.alert(_app_.con_error);
                }
            }, "json");
        },

        /**
         * Edit a service
         */
        edit: function (e) {
            var me = this;
            var $this = $(e.target);
            var par = $this.parents(".app-tr");
            var table = par.parents("table.app-services");
            var service_id = par.find(".span_service_id").html();
            $.infoPanel("reading");
            $this.addClass("app-disabled-button");
            par.find(".waiting").css("visibility", "visible");
			$(document).trigger("service-settings-opened");
            var data = {
                wpb_ajax: true,
                action: "app_inline_edit_service",
                col_len: parseInt(table.find("tr:first th:visible").length),
				by_worker: table.data("by_worker"),
				sel_location: table.data("sel_location"),
                service_id: service_id,
				app_lang: $.urlParam("app_lang"),
                ajax_nonce: _app_.iedit_nonce
            };
            $.post(_app_.ajax_url, data, function (r) {
                par.find(".waiting").css("visibility", "hidden");
				$(document).trigger("service-settings-closed");
                if (r && r.error) {
                    window.alert(r.error);
                } else if (r) {
                    par.hide();
                    var iedit_row = r.result;
                    $(iedit_row).insertAfter(par);
					me.configureTabs();
                    me.configureMS(r.id);
                    me.configQtip(r.id);
                } else {
                    window.alert(_app_.con_error);
                }
            }, "json");

       },

        /**
         * Tabs
         */
	   configureTabs: function() {
		    var me = this;
			if ($(document).find(".service-tabs-container").length > 0) {
				$(".service-tabs-container").tabs({
					activate: function () {
						me.multiselectAll();
					}
				}).removeClass("ui-widget ui-widget-content");
			}
	   },

        /**
         * Configure multiselect widget
         */
        multiselectAll: function () {
            $.each($("select.app_ms:visible, select.app_ms_tab"), function () {
                var $this = $(this);
				var bWidth = $this.data("buttonwidth") || "100%";
				var sList = $this.data("selectedlist") || "6";
                if ($this.data().hasOwnProperty("echMultiselect")) {
					$this.multiselect("option", "menuWidth", bWidth);
					$this.multiselect("option", "selectedList", sList);
                    return true;
                }
                $this.multiselect({
                    noneSelectedText: $this.data("noneselectedtext") || "Select",
                    buttonWidth: bWidth,
					selectedList: sList,
                    classes: $this.attr("class") || "",
					open: function () {
						$(document).trigger("app-ms-opened", $this);
					},
					close: function () {
						$(document).trigger("app-ms-closed", $this);
					}
                });
            });
        },

        /**
         * Save a service
         */
        save: function (e) {
			e.preventDefault();
            var me = this;
            var $this = $(e.target);
            var sPar = $this.parents(".inline-edit-row");
            var postData = me.readData(sPar);

            $this.attr("disabled", true);
            $.infoPanel("saving");
            sPar.find(".waiting").show();

            postData.action = "app_inline_edit_save_service";

            $.post(_app_.ajax_url, postData, function (r) {
                $this.attr("disabled", false);
                sPar.find(".waiting").hide();

                if (!r) {
                    window.alert(_app_.con_error);
                    return false;
                }
                var emailMsg = r.emailMsg ? " " + r.emailMsg : "";
                if (r.error) {
                    sPar.find(".error")
                    .html("<span class='app-error'>" + r.error + emailMsg + "</span>")
                    .show().delay(10000).fadeOut("slow");
                } else if (r.no_change) {
                    sPar.find(".error")
                    .html("<span class='app-b'>" + r.no_change + emailMsg + "</span>")
                    .show().delay(10000).fadeOut("slow");
                } else if (r.result) {
					me.sendSuccess(r, sPar, emailMsg, $this);
                } else if (emailMsg) {
                    sPar.find(".error")
                    .html("<span class='app-success'>" + emailMsg + "</span>")
                    .show().delay(10000).fadeOut("slow");
				}
            }, "json");
        },

        /**
         * Send success message
         */
		sendSuccess: function(r, sPar, emailMsg, $this) {
			var me = this;
            var serviceID = sPar.find("input[name='service_id']").val();
			sPar.find(".error")
			.html("<span class='app-success'>" + r.result + emailMsg + "</span>")
			.show().delay(10000).fadeOut("slow");

			var prevRow = sPar.prev(".app-tr");
			if (r.result_service_id) {
				var nid = r.result_service_id;
				sPar.find("input[name='service_id']").val(nid);
				if (nid !== serviceID) {
					sPar.find(".app-service-id").html("#" + nid);
				}
				if (nid) {
					sPar.find("a.save").text(_app_.update_text);
				}
			}

			$.each(r, function (i, v) {
				if ( "service_name" == i ) {
					prevRow.find(".service-inner").replaceWith(v);
				} else if ("id" ==i || "description_page" == i || "image" === i) {
					prevRow.find(".column-" + i).html(v);
				} else {
					prevRow.find(".column-" + i).text(v);
				}
			});

			prevRow.addClass("ui-state-highlight");
			setTimeout(function () {
				prevRow.removeClass("ui-state-highlight");
			}, 10000);

			if (parseInt(r.collapse) > 0) {
				setTimeout(function () {
					me.collapseRecord($this, r);
				}, 500);
			}
		},

        /**
         * Initiates Multiselect for users
         */
        configureMS: function (id) {
            $(".add_location_multiple").multiselect({
                selectedList: 6,
                buttonWidth: "100%",
                classes: "app_service_locations app-ms-small"
            }).multiselectfilter();

            $(".add_category_multiple").multiselect({
                selectedList: 6,
                buttonWidth: "100%",
                classes: "app_service_categories app-ms-small"
            }).multiselectfilter();

        },

        /**
         * Collapses record after cancel/save
         */
        collapseRecord: function (el, r) {
            var row = el.parents(".inline-edit-row");
			var app_id = row.find("input[name='app_id']").val();
            $(".app-inline-edit-service").removeClass("app-disabled-button");
            row.fadeOut(700, function () {
                row.remove();
            });
            row.prev(".app-tr").removeClass("app-locked").show("slow");

			if (!app_id) {
				if (r && r.new_service) {
					window.location.href = window.location.href;
				} else {
					return;
				}
			}

        },

        /**
         * Initiates qtip
         */
        configQtip: function (id) {
            var edit_row = $(".inline-edit-row");
            edit_row.find("[title][title!='']").each(function (ignore, val) {
                var $this = $(val);
                var title = $this.attr("title");
                var ttip = title ? title.replace(/●/g, "<br />").replace("/|/", "<br/>") : "";

                $this.qtip({
                    content: {
                        text: ttip
                    },
                    hide: _app_.qtipHide,
                    position: _app_.qtipPos,
                    style: _app_.qtipYellow
                });
            });

        },

        /**
         * makes items sortable
         */
        doSortable: function () {
            var me = this;
            var sortService = me.tbl.sortable({
                items: "tbody tr",
				handle: ".app-sortable",
                forcePlaceholderSize: true,
                forceHelperSize: true,
                start: function (ignore, ui) {
					me.tbl.css("table-layout", "fixed");
                },
                stop: function (ignore, ui) {
					me.tbl.css("table-layout", "auto");
                },
                helper: me.fixHelper,
                update: me.sortUpdate
            });
            $(document).on("service-settings-opened", function () {
                sortService.sortable("disable");
                $(".app-service-tr").css("cursor", "pointer");
            });
            $(document).on("service-settings-closed", function () {
                sortService.sortable("enable");
                $(".app-service-tr").css("cursor", "move");
            });
		},

        /**
         * Keep sortable item size during sorting
         */
        fixHelper: function (e, tr) {
			var table = tr.parents("table");
			var $originals = tr.children();
			var $helper = tr.clone();
			$helper.children().each( function(i)	{
			  $(this).width($originals.eq(i).width());
			});

			table.find("thead tr").children().each( function (i, val) {
				$(val).width($originals.eq(i).width());
			});

			return $helper;
        },

        /**
         * Save sort
         */
        sortUpdate: function (e, ui) {
			var table = ui.item.parents("table");
			var span = table.find("tbody tr.app-tr .column-sort_order span");
			var sort_data = {};

			span.each(function(){
				var service_id = " " + String($(this).data("service_id"));
				var service_sort = String($(this).data("sort_order"));
				if (service_id === null || service_sort === null) {
					return true;
				}
				sort_data[service_id] = service_sort;
			});

			var data = {
				wpb_ajax: true,
				action: "app_save_sort_service",
				sort_data: JSON.stringify(sort_data),
				app_paged: table.data("app_paged"),
				ajax_nonce: _app_.iedit_nonce
			};
			$.post(_app_.ajax_url, data, function (r) {
				if (r && r.error) {
					window.alert(r.error);
				} else if (r) {
					if (r.data) {
						var span = '';
						var prevOrder = -1;
						$.each(r.data, function(i, val) {
							span = table.find("tr#app-tr-"+val.sid+" td.column-sort_order").find("span.app-sortable");
							prevOrder = span.data("sort_order");
							span.text(val.order);
							span.data("sort_order", val.order);

							if (prevOrder != val.order) {
								span.addClass("app-flash");
							} else {
								span.removeClass("app-flash");
							}

							setTimeout(function () {
								table.find("span.app-sortable").removeClass("app-flash");
							}, 5000);
						});
					}
				} else {
					window.alert(_app_.con_error);
				}
			}, "json");
        }

    };
    WPB_Services.init();

});
</script>
<?php
	}
}

	BASE('AdminServices')->add_hooks();
}