<?php
/**
 * WPB Admin Service Properties
 *
 * Display and setting for details of services
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.7.6.2
 */

if ( !class_exists( 'WpBAdmin_Service_Props' ) ) {

class WpBAdmin_Service_Props {

	/**
     * Nice name of the addon
     */
	public $public_name;

	/**
     * WP BASE Core + Front [+Admin] instance
     */
	protected $a = null;

	/**
     * Constructor
     */
	public function __construct() {
		$this->public_name = __('Service Properties','wp-base');
		$this->a = BASE();
	}

	/**
     * Add actions and filters
     */
	public function add_hooks() {
		add_action( 'pre_get_posts', array( $this, 'pre_get_posts' ) );

		add_action( 'app_admin_services_start_render', array( $this, 'start_render' ) );
	}

	/**
     * Add selectable actions and filters when service page started being rendered
     */
	public function start_render(){

		add_filter( 'user_has_cap', array( $this, 'has_caps' ), 10, 4 );

		wp_enqueue_media();
	}

	/**
     * Determines if this is a BP page
	 * @return	bool
     */
	public static function is_bp_page(){
		return has_action( 'app_bp_load_assets' ) || did_action( 'app_bp_load_assets' );
	}

	/**
     * Determines if user can upload image
	 * @uses user_has_cap filter hook @ class-wp-user.php
	 * @return	array
     */
	public function has_caps( $all_caps, $caps, $args, $user ) {
		if ( empty( $args[0] ) || 'edit_post' != $args[0] || empty( $args[1] ) || empty( $args[2] ) ) {
			return $all_caps;
		}

		$user_id = $args[1];

		# User must be a provider or admin
		if ( wpb_is_account_page() || self::is_bp_page() ) {
			if ( ! wpb_is_worker( $user_id ) ) {
				return $all_caps;
			}
		} else if ( wpb_is_admin_page() ) {
			if ( ! user_can( WPB_ADMIN_CAP ) ) {
				return $all_caps;
			}
		} else {
			return $all_caps;
		}

		// Must have ALL requested caps.
		foreach( $caps as $key => $cap ) {
			$all_caps[ $cap ] = true;
		}

		return $all_caps;
	}

	/**
     * Adds tab to services table under add more
     */
	public function add_tab( $tabs ) {
		$class_name = get_class();
		return array_merge( $tabs, array( $class_name => $this->public_name ) );
	}

	/**
     * Add html codes
     */
	public function add_html( $html, $n, $service ) {
		wpb_add_action_footer( $this );

		$ID			= ! empty( $service->ID ) ? $service->ID  : false;
		$s			= new WpB_Service( $ID );
		$desc		= $s->get_description() ?: '';
		$img_id		= $s->get_image_id() ?: 0;
		$pholder	= wpb_placeholder_img_src();
		$img_url	= $s->get_image_url() ?: '';
		$img_html	= $img_id
					  ? wp_get_attachment_image( $img_id, array(200,200), false, array( 'class' => 'app-main-image', 'srcset' => ' ', 'loading' => false, 'alt' => BASE()->get_service_name( $ID ) ) )
					  : '<img class="app-main-image" src="'.esc_attr($pholder).'" alt="Service" />';

		$html .= '<tr id="'.get_class().'_'.$n.'" class="app_admin_service_properties inline-edit-row"><td colspan="'.(BASE('AdminServices')->nof_columns()-2).'">';

		$html .= '<fieldset>';

		$html .= wpb_info_button( $this );

		$html .= '<div class="inline-edit-col">';

		$html .= '<div class="app-flex app-iedit-created-owner">';

		$user = false;

		if ( current_user_can( WPB_ADMIN_CAP ) ) {
			$html .= wpb_wrap_field( 'owner', __('Owner', 'wp-base'), self::select_owner( $s ) );
		} else {
			$html .= wpb_wrap_field( 'owner', __('Owner', 'wp-base'), '<p>'. esc_attr( $s->get_owner() ? BASE()->get_worker_name( $s->get_owner() ) : $this->a->get_text('admin') ) .'</p>' );
		}

		$html .= wpb_wrap_field( 'created_by', __('Created by', 'wp-base'),
				'<p>' . esc_html( self::created_by( $s ) ) .'</p>' );

		$html .= wpb_wrap_field(  'created_at', __('Created at', 'wp-base'),
				'<p>' . esc_html( ($s->get_created_at() ?: '') ) .'</p>' );

		$html .= '</div>';

		$html .= '<div class="app_iedit_image">';
		$html .= '<span class="title">'.__('Featured Image', 'wp-base'). '</span>';
		$html .= '<span class="input-text-wrap">';
		$html .=
		'<div class="app-image-upload lp-border-bottom padding-bottom-45">
			<div class="image-preview">
				<span class="dashicons dashicons-trash app-remove-image'.(!$img_url ? ' has-pholder' : '').'" title="'.esc_attr( __( 'Delete', 'wp-base' ) ).'"></span>
				'. $img_html.'
				<input class="pholder_img" type="hidden" value="'.esc_attr($pholder).'">
			</div>
			<div class="image-description">
				<div class="upload-photo margin-top-25">
					<span class="file-input file-upload-btn btn-first-hover btn-file">
						<input data-content="'.$this->a->get_text('choose_image').'" class="app-upload-image-btn" data-author="'.(current_user_can( WPB_ADMIN_CAP ) ? 0 : get_current_user_id()).'" type="file">
					</span>
					<input class="criteria-image-url" type="hidden" name="service_image_url['.$ID.']" value="'.esc_attr($img_url).'">
					<input class="criteria-image-id" type="hidden" name="service_image_id['.$ID.']" value="'.esc_attr($img_id).'">
				</div>
			</div>
		</div>';

		$html .= '</span>';
		$html .= '</div>';

		$html .= wpb_wrap_field( 'description', __('Description', 'wp-base'),
				'<textarea rows="10" name="service_description['.$ID.']">' . esc_textarea( $desc ) .'</textarea>' );

		$html .= '</div>';
		$html .= '</fieldset>';

		$html .= '</td></tr>';

		return $html;
	}

	/**
	 * Add Owner select element
	 * @param $s	object		WpB_Service object
	 * @return string
	 */
	public static function select_owner( $s ) {
		$ID	 = $s->get_ID();

		$h = '<select name="owner['.$ID.']" >';
		$h .= '<option value="">'. esc_attr( BASE()->get_text('admin') ) .'</option>';

		foreach( BASE()->get_workers( 'name' ) as $worker ) {
			$name = BASE()->get_worker_name( $worker->ID );
			$h .= '<option value="'.$worker->ID.'" '.selected( $worker->ID, $s->get_owner(), false ).'>'. esc_attr( $name )	.'</option>';
		}

		$h .= '</select>';
		$h .= '<input type="hidden" name="owner_check['.$ID.']" value="'.$ID.'">';

		return $h;
	}

	/**
	 * Add Managed by multi select element
	 * @param $s	object		WpB_Service object
	 * @return string
	 */
	public static function select_managed_by( $s ) {
		$ids = $s->get_managed_by();
		$ID	 = $s->get_ID();

		$h = '<select multiple name="managed_by['.$ID.'][]" class="select-usable-by app_ms_tab app-ms-small" data-buttonwidth="200" data-noneselectedtext="'.esc_attr( 'All Providers', 'wp-base' ).'">';

		foreach( BASE()->get_workers( 'name' ) as $worker ) {
			$name = BASE()->get_worker_name( $worker->ID );
			$h .= '<option value="'.$worker->ID.'" '.selected( in_array( $worker->ID, $ids ), true, false ).'>'. esc_attr( $name ) .'</option>';
		}

		$h .= '</select>';
		$h .= '<input type="hidden" name="managed_by_check['.$ID.']" value="'.$ID.'">';

		return $h;
	}

	/**
	 * List Managed by without select
	 * @param $s	object		WpB_Service object
	 * @return string
	 */
	public static function list_managed_by( $s ) {
		$arr = '';

		foreach( $s->get_managed_by() as $worker_id ) {
			$arr[] = BASE()->get_worker_name( $worker_id );
		}

		return implode( ', ', $arr );
	}

	/**
	 * Return created by in user readable format
	 * @param $s	object		WpB_Service object
	 * @return string
	 */
	public static function created_by( $s ) {
		$created_by	= '';
		if ( $maybe_created_by = $s->get_created_by() ) {
			if ( $uid = get_user_by( 'ID', $maybe_created_by ) ) {
				$created_by = $uid->display_name;
			}
		}

		return $created_by ?: BASE()->get_text( 'unknown' );
	}

	/**
	 * Send author ID to image uploader post query
	 * Add description to footer
	 */
	public function footer(){
		$id = get_current_user_id();
?>
<br>
<script type="text/javascript">
jQuery(document).ready(function($){
	$.ajaxPrefilter( function( options, originalOptions ) {
		options.data = $.param($.extend(originalOptions.data||{}, {
			author_id: <?php echo $id ?>
		}));
	});
});
</script>
<div class="<?php echo strtolower( get_class( $this ) ) ?>-desc" style="display:none">
	<ul class="app-info-list">
		<?php
			foreach ( WpBConstant::service_props_desc() as $key => $line ) {
				echo "<li>". $line . "</li>";
			}
		?>
	</ul>
</div>
<?php
	}

	/**
     * Filter attachments so that author can only view own attachments
	 * @uses pre_get_posts action hook
	 * @return	none ($wp_query is passed by ref)
     */
	public function pre_get_posts( $wp_query ) {
		if ( empty( $_POST['action'] ) || 'query-attachments' != $_POST['action'] ) {
			return;
		}

		if ( empty( $_POST['author_id'] ) || user_can( $_POST['author_id'], WPB_ADMIN_CAP ) ) {
			return;
		}

		if ( 'attachment' != $wp_query->query_vars['post_type'] ) {
			return;
		}

		$wp_query->query_vars['author'] = $_POST['author_id'];
	}

	/**
	 *	Called when a service is added on admin
	 */
	public function service_added( $ID ) {
		$this->add_meta( $ID );
	}

	/**
	 *	Called when a service is *maybe* updated on admin (may be just submitted)
	 */
	public function service_updated( $ID ) {
		$this->add_meta( $ID, true );
	}

	/**
	 *	Add metas when service added/updated
	 */
	public function add_meta( $ID, $update = false ) {
		$s		 = new WpB_Service( $ID );
		$changed = false;

		if ( isset( $_POST['owner_check'][ $ID ] ) ) {
			if ( empty( $_POST['owner'][ $ID ] ) ) {
				if ( $s->delete_owner() ) {
					$changed = true;
				}
			} else if ( $s->update_owner( wpb_clean( $_POST['owner'][ $ID ] ) ) ) {
				$changed = true;
			}
		}

		if ( isset( $_POST['managed_by_check'][ $ID ] ) ) {
			if ( empty( $_POST['managed_by'][ $ID ] ) ) {
				if ( $s->delete_managed_by() ) {
					$changed = true;
				}
			} else if ( $s->update_managed_by( wpb_clean( $_POST['managed_by'][ $ID ] ) ) ) {
				$changed = true;
			}
		}

		if ( isset( $_POST['service_description'][ $ID ] ) ) {
			if ( $s->update_description( wp_kses_post( $_POST['service_description'][ $ID ] ) ) ) {
				$changed = true;
			}
		}

		if ( isset( $_POST['service_image_url'][ $ID ] ) ) {
			if ( $s->update_image_url( trim( $_POST['service_image_url'][ $ID ] ) ) ) {
				$changed = true;
			}
		}

		if ( isset( $_POST['service_image_id'][ $ID ] ) ) {
			if ( $s->update_image_id( trim( $_POST['service_image_id'][ $ID ] ) ) ) {
				$changed = true;
			}
		}

		if ( $changed ) {
			wpb_notice( 'saved' );
		}
	}
}

	BASE('Admin_Service_Props')->add_hooks();
}