<?php
/**
 * WPB Dashboard
 *
 * Displays general status, reports and graphs
 *
 * Most parts are adapted from Hakan Ozevin
 *
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.5.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpBDashboard' ) ) {

class WpBDashboard {

	/**
     * Name of the preference
     */
	const meta = 'app_reports_preferred_range';

	/**
     * Screen ID of the reports page
     */
	private $reports_page;

	/**
     * WC BASE Core + Front [+Admin] instance
     */
	protected $a = null;

	/**
     * Constructor
     */
	public function __construct(){
		$this->a = BASE();
	}

	/**
     * Add hooks
     */
	public function add_hooks(){
		add_action( 'app_menu_before_all', array( $this, 'add_menu' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_action( 'app_save_settings_with_get', array( $this, 'parse_report_dates' ) );
	}

	/**
     * Add submenu page
     */
	public function add_menu(){
		$this->menu_page = add_menu_page('WP BASE', WPB_NAME, WPB_ADMIN_CAP,  'appointments', array($this,'render_page'),'dashicons-calendar', '25.672');

		$this->reports_page = wpb_add_submenu_page( 'appointments', __('Dashboard', 'wp-base' ), __('Dashboard', 'wp-base' ), array(WPB_ADMIN_CAP,'manage_reports'), "appointments", array($this,'render_page'));
	}

	/**
     * Load jQuery Flot
     */
	public function enqueue_scripts(){
		if ( $this->reports_page != wpb_get_current_screen_id() ) {
			return;
		}

		wp_enqueue_script( 'jquery-flot', WPB_PLUGIN_URL . '/js/admin/jquery.flot.min.js', array( 'jquery' ), WPB_VERSION );
	}

	/**
     * Build report dates from submitted filter
     */
	public function parse_report_dates(){
		if ( empty( $_GET['action_app'] ) || 'filter_reports' != $_GET['action_app'] ) {
			return;
		}

		if ( !empty( $_GET['range'] ) ) {
			update_user_meta( get_current_user_id(), self::meta, $_GET['range'] );
		}

		$dates = self::get_report_dates();

		foreach ( array( 'app_location_id', 'app_service_id', 'app_worker_id', 'by_created' ) as $lsw ) {
			if ( isset( $_GET[ $lsw ] ) ) {
				$dates[ $lsw ] = wpb_clean( $_GET[ $lsw ] );
			}
		}

		wp_redirect( add_query_arg( $dates, admin_url( 'admin.php?page=appointments' ) ) );
		exit;
	}

	/**
	 * Sets up the dates used to filter graph data
	 *
	 * Date sent via $_GET is read first and then modified (if needed) to match the
	 * selected date-range (if any)
	 *
	 * @return array
	*/
	public static function get_report_dates() {
		$dates = array();

		$current_time = BASE()->_time;

		if ( !empty( $_GET['range'] ) ) {
			$dates['range'] = $_GET['range'];
		} else if ( $pref = get_user_meta( get_current_user_id(), self::meta, true ) ) {
			$dates['range'] = $pref;
		} else {
			$dates['range'] = apply_filters( 'app_get_report_dates_default_range', 'this_week' );
		}

		$dates['year_start'] = isset( $_GET['year_start'] ) ? $_GET['year_start'] : date( 'Y', $current_time );
		$dates['year_end']   = isset( $_GET['year_end'] )? $_GET['year_end']: date( 'Y', $current_time );
		$dates['m_start']    = isset( $_GET['m_start'] ) ? $_GET['m_start'] : 1;
		$dates['m_end']      = isset( $_GET['m_end'] ) ? $_GET['m_end']   : 12;
		$dates['day_start']  = isset( $_GET['day_start'] ) ? $_GET['day_start']     : 1;
		$dates['day_end']    = isset( $_GET['day_end'] ) ? $_GET['day_end'] : wpb_cal_days_in_month( 'cal_gregorian', $dates['m_end'], $dates['year_end'] );

		$dates['time_diff']	= mktime( 0,0,0, $dates['m_end'], $dates['day_end'], $dates['year_end'] ) - mktime( 0,0,0, $dates['m_start'], $dates['day_start'], $dates['year_start'] );

		// Modify dates based on predefined ranges
		switch ( $dates['range'] ) :

			case 'this_month' :
				$dates['m_start']  = date( 'n', $current_time );
				$dates['m_end']    = date( 'n', $current_time );
				$dates['day_start']      = 1;
				$dates['year_start']     = date( 'Y' );
				$dates['year_end'] = date( 'Y' );
				$dates['day_end']  = wpb_cal_days_in_month( 'cal_gregorian', $dates['m_end'], $dates['year_start'] );
			break;

			case 'next_month' :
				$next_month		   = wpb_first_of_month( $current_time, 1 );
				$dates['m_start']  = date( 'n', $next_month );
				$dates['m_end']    = date( 'n', $next_month );
				$dates['day_start']      = 1;
				$dates['year_start']     = date( 'Y', $next_month );
				$dates['year_end'] = date( 'Y', $next_month );
				$dates['day_end']  = wpb_cal_days_in_month( 'cal_gregorian', $dates['m_end'], $dates['year_start'] );
			break;

			case 'last_month' :
				if ( date( 'n' ) == 1 ) {
					$dates['m_start']  		= 12;
					$dates['m_end']    		= 12;
					$dates['year_start']	= date( 'Y', $current_time ) - 1;
					$dates['year_end']		= date( 'Y', $current_time ) - 1;
				} else {
					$dates['m_start']  		= date( 'n' ) - 1;
					$dates['m_end']    		= date( 'n' ) - 1;
					$dates['year_start']	= date( 'Y', $current_time );
					$dates['year_end'] 		= $dates['year_start'];
				}
				$dates['day_start']     = 1;
				$dates['day_end'] 		= wpb_cal_days_in_month( 'cal_gregorian', $dates['m_end'], $dates['year_start'] );
			break;

			case 'today' :
				$dates['day_start']     = date( 'd', $current_time );
				$dates['m_start'] 		= date( 'n', $current_time );
				$dates['m_end']   		= date( 'n', $current_time );
				$dates['year_start']    = date( 'Y', $current_time );
				$dates['year_end'] 		= $dates['year_start'];
			break;

			case 'yesterday' :

				$year  = date( 'Y', $current_time );
				$month = date( 'n', $current_time );
				$day   = date( 'd', $current_time );

				if ( $month == 1 && $day == 1 ) {
					$year  -= 1;
					$month = 12;
					$day   = wpb_cal_days_in_month( 'cal_gregorian', $month, $year );
				} elseif ( $month > 1 && $day == 1 ) {
					$month -= 1;
					$day   = wpb_cal_days_in_month( 'cal_gregorian', $month, $year );
				} else {
					$day -= 1;
				}

				$dates['day_start']		= $day;
				$dates['m_start']   	= $month;
				$dates['m_end']     	= $month;
				$dates['year_start']	= $year;
				$dates['year_end']  	= $year;
				$dates['day_end']   	= $day;
			break;

			case 'tomorrow' :
				$dates['day_start']     = date( 'd', $current_time + DAY_IN_SECONDS );
				$dates['day_end'] 		= date( 'd', $current_time + 2*DAY_IN_SECONDS );
				$dates['m_start'] 		= date( 'n', $current_time + DAY_IN_SECONDS );
				$dates['m_end']   		= date( 'n', $current_time + DAY_IN_SECONDS );
				$dates['year_start']    = date( 'Y', $current_time + DAY_IN_SECONDS );
				$dates['year_end']		= $dates['year_start'];
			break;

			case 'this_week' :
			case 'last_week' :
			case 'next_week' :
				if ( $dates['range'] === 'this_week' ) {
					$base_time = date( 'Y-m-d h:i:s', $current_time );
				} else if ( $dates['range'] === 'last_week' ) {
					$base_time = date( 'Y-m-d h:i:s', $current_time - WEEK_IN_SECONDS );
				} else {
					$base_time = date( 'Y-m-d h:i:s', $current_time + WEEK_IN_SECONDS );
				}

				$start_end = get_weekstartend( $base_time, get_option( 'start_of_week' ) );

				$dates['day_start']		= date( 'd', $start_end['start'] );
				$dates['m_start']  		= date( 'n', $start_end['start'] );
				$dates['year_start']	= date( 'Y', $start_end['start'] );

				$dates['day_end']  = date( 'd', $start_end['end'] );
				$dates['m_end']    = date( 'n', $start_end['end'] );
				$dates['year_end'] = date( 'Y', $start_end['end'] );
			break;

			case 'last_30_days' :

				$date_start = strtotime( '-30 days' );

				$dates['day_start']		= date( 'd', $date_start );
				$dates['m_start']  		= date( 'n', $date_start );
				$dates['year_start']	= date( 'Y', $date_start );

				$dates['day_end']  = date( 'd', $current_time );
				$dates['m_end']    = date( 'n', $current_time );
				$dates['year_end'] = date( 'Y', $current_time );

			break;

			case 'this_year' :
				$dates['day_start']		= 1;
				$dates['m_start']		= 1;
				$dates['m_end']			= 12;
				$dates['year_start']	= date( 'Y', $current_time );
				$dates['year_end']		= $dates['year_start'];
			break;

			case 'last_year' :
				$dates['day_start']		= 1;
				$dates['m_start']		= 1;
				$dates['m_end']			= 12;
				$dates['year_start']    = date( 'Y', $current_time ) - 1;
				$dates['year_end']		= $dates['year_start'];
			break;

		endswitch;

		return apply_filters( 'app_report_dates', $dates );
	}

	/**
     * Create HTML for the page
     */
	public function render_page(){

?>
<div id="app-navbar" class="app-control-wrap metabox-holder app-dash-navbar">
	<div class="postbox app-controls">

	<?php self::graph_controls(); ?>

	</div>
</div>
<div class="wrap app-page app-dashboard app-clearfix">
	<?php echo self::draw_graph(); ?>
</div>
<?php
	}

	/**
     * Whether this a a vendor dashboard
	 * @return bool
     */
	public static function is_vendor_page(){
		return apply_filters( 'app_is_vendor_dashboard', false );
	}

	/**
     * Helper to create page
     */
	private static function draw_graph() {

		$dates = self::get_report_dates();

		switch ( $dates['range'] ) {
			case 'today' :
			case 'yesterday' :
			case 'tomorrow' :
				$day_by_day = true;
				break;
			case 'last_year' :
			case 'this_year' :
				$day_by_day = false;
				break;
			case 'last_quarter' :
			case 'this_quarter' :
				$day_by_day = true;
				break;
			case 'custom' :
				if ( $dates['m_start'] == 12 && $dates['m_end'] == 1 ) {
					$day_by_day = true;
				} elseif ( $dates['m_end'] - $dates['m_start'] >= 3 || ( $dates['year_end'] > $dates['year_start'] && ( $dates['m_start'] - $dates['m_end'] ) != 10 ) ) {
					$day_by_day = false;
				} else {
					$day_by_day = true;
				}
				break;
			default:
				$day_by_day = true;
				break;
		}

		$args = array(
			'range'			=> $dates['range'],
			'location'		=> !empty( $_GET['app_location_id'] ) ? $_GET['app_location_id'] : false,
			'service'		=> !empty( $_GET['app_service_id'] ) ? $_GET['app_service_id'] : false,
			'worker'		=> isset( $_GET['app_worker_id'] ) ? $_GET['app_worker_id'] : false,
			'day_by_day'	=> $day_by_day,
			'by_created'	=> !empty( $_GET['by_created'] ) ? $_GET['by_created'] : false,
			'time_diff'		=> $dates['time_diff'],
		);

		if ( self::is_vendor_page() ) {
			$args['worker'] = get_current_user_id();
		}

		$earnings_totals = 0.00; // Total earnings for time period shown
		$sales_totals    = 0;    // Total sales for time period shown

		if ( $dates['range'] == 'today' || $dates['range'] == 'yesterday' ) {
			// Hour by hour
			$hour  = 0;
			$month = $dates['m_start'];

			$i = 0;
			$j = 0;

			$start 			= $dates['year_start'] . '-' . $dates['m_start'] . '-' . $dates['day_start'];
			$end 			= $dates['year_end'] . '-' . $dates['m_end'] . '-' . $dates['day_end'];
			$args['start']	= $start;
			$args['end']	= $end;
			$sales			= wpb_get_sales( $args );
			$earnings 		= wpb_get_revenue( $args );

			while ( $hour <= 23 ) {
				$date = mktime( $hour, 0, 0, $month, $dates['day_start'], $dates['year_start'] ) * 1000;

				if ( isset( $earnings[ $i ] ) && $earnings[ $i ]['h'] == $hour ) {
					$earnings_data[] = array( $date, $earnings[ $i ]['total'] );
					$earnings_totals += $earnings[ $i ]['total'];
					$i++;
				} else {
					$earnings_data[] = array( $date, 0 );
				}

				if ( isset( $sales[ $j ] ) && $sales[ $j ]['h'] == $hour ) {
					$sales_data[] = array( $date, $sales[ $j ]['count'] );
					$sales_totals += $sales[ $j ]['count'];
					$j++;
				} else {
					$sales_data[] = array( $date, 0 );
				}

				$hour++;
			}
		} elseif ( $dates['range'] == 'this_week' || $dates['range'] == 'last_week' ) {
			$report_dates = array();
			$i = 0;
			while ( $i <= 6 ) {
				if ( ( $dates['day_start'] + $i ) <= $dates['day_end'] ) {
					$report_dates[ $i ] = array(
						'day'   => (string) $dates['day_start'] + $i,
						'month' => $dates['m_start'],
						'year'  => $dates['year_start'],
					);
				} else {
					$report_dates[ $i ] = array(
						'day'   => (string) $i,
						'month' => $dates['m_end'],
						'year'  => $dates['year_end'],
					);
				}

				$i++;
			}

			$start_date 	= $report_dates[0];
			$end_date 		= end( $report_dates );
			$args['start']  = $start_date['year'] . '-' . $start_date['month'] . '-' . $start_date['day'];
			$args['end']	= $end_date['year'] . '-' . $end_date['month'] . '-' . $end_date['day'];
			$sales			= wpb_get_sales( $args );
			$earnings 		= wpb_get_revenue( $args );

			$i = 0;
			$j = 0;
			foreach ( $report_dates as $report_date ) {
				$date = mktime( 0, 0, 0,  $report_date['month'], $report_date['day'], $report_date['year']  ) * 1000;

				if ( array_key_exists( $i, $sales ) && $report_date['day'] == $sales[ $i ]['d'] && $report_date['month'] == $sales[ $i ]['m'] && $report_date['year'] == $sales[ $i ]['y'] ) {
					$sales_data[] = array( $date, $sales[ $i ]['count'] );
					$sales_totals += $sales[ $i ]['count'];
					$i++;
				} else {
					$sales_data[] = array( $date, 0 );
				}

				if ( array_key_exists( $j, $earnings ) && $report_date['day'] == $earnings[ $j ]['d'] && $report_date['month'] == $earnings[ $j ]['m'] && $report_date['year'] == $earnings[ $j ]['y'] ) {
					$earnings_data[] = array( $date, $earnings[ $j ]['total'] );
					$earnings_totals += $earnings[ $j ]['total'];
					$j++;
				} else {
					$earnings_data[] = array( $date, 0 );
				}
			}

		} else {
			if ( wpb_cal_days_in_month( 'cal_gregorian', $dates['m_start'], $dates['year_start'] ) < $dates['day_start'] ) {
				$next_day = mktime( 0, 0, 0, $dates['m_start'] + 1, 1, $dates['year_start'] );
				$day = date( 'd', $next_day );
				$month = date( 'm', $next_day );
				$year = date( 'Y', $next_day );
				$date_start = $year . '-' . $month . '-' . $day;
			} else {
				$date_start = $dates['year_start'] . '-' . $dates['m_start'] . '-' . $dates['day_start'];
			}

			if ( wpb_cal_days_in_month( 'cal_gregorian', $dates['m_end'], $dates['year_start'] ) < $dates['day_end'] ) {
				$date_end = $dates['year_end'] . '-' . $dates['m_end'] . '-' . wpb_cal_days_in_month( 'cal_gregorian', $dates['m_end'], $dates['year_start'] );
			} else {
				$date_end = $dates['year_end'] . '-' . $dates['m_end'] . '-' . $dates['day_end'];
			}

			$args['start']  = $date_start;
			$args['end']	= $date_end;
			$sales			= wpb_get_sales( $args );
			$earnings 		= wpb_get_revenue( $args );

			$y = $dates['year_start'];
			$temp_data = array(
				'sales'    => array(),
				'earnings' => array(),
			);

			foreach ( $sales as $sale ) {
				if ( $day_by_day ) {
					$temp_data['sales'][ $sale['y'] ][ $sale['m'] ][ $sale['d'] ] = $sale['count'];
				} else {
					$temp_data['sales'][ $sale['y'] ][ $sale['m'] ] = $sale['count'];
				}
				$sales_totals += $sale['count'];
			}

			foreach ( $earnings as $earning ) {
				if ( $day_by_day ) {
					$temp_data['earnings'][ $earning['y'] ][ $earning['m'] ][ $earning['d'] ] = $earning['total'];
				} else {
					$temp_data['earnings'][ $earning['y'] ][ $earning['m'] ] = $earning['total'];
				}
				$earnings_totals += $earning['total'];
			}

			while ( $day_by_day && ( strtotime( $date_start ) <= strtotime( $date_end ) ) ) {
				$d = date( 'd', strtotime( $date_start ) );
				$m = date( 'm', strtotime( $date_start ) );
				$y = date( 'Y', strtotime( $date_start ) );

				if ( ! isset( $temp_data['sales'][ $y ][ $m ][ $d ] ) ) {
					$temp_data['sales'][ $y ][ $m ][ $d ] = 0;
				}

				if ( ! isset( $temp_data['earnings'][ $y ][ $m ][ $d ] ) ) {
					$temp_data['earnings'][ $y ][ $m ][ $d ] = 0;
				}

				$date_start = date( 'Y-m-d', strtotime( '+1 day', strtotime( $date_start ) ) );
			}

			while ( ! $day_by_day && ( strtotime( $date_start ) <= strtotime( $date_end ) ) ) {
				$m = date( 'm', strtotime( $date_start ) );
				$y = date( 'Y', strtotime( $date_start ) );

				if ( ! isset( $temp_data['sales'][ $y ][ $m ] ) ) {
					$temp_data['sales'][ $y ][ $m ] = 0;
				}

				if ( ! isset( $temp_data['earnings'][ $y ][ $m ] ) ) {
					$temp_data['earnings'][ $y ][ $m ] = 0;
				}

				$date_start = date( 'Y-m', strtotime( '+1 month', strtotime( $date_start ) ) );
			}

			$sales_data    = array();
			$earnings_data = array();

			// When using 3 months or smaller as the custom range, show each day individually on the graph
			if ( $day_by_day ) {
				foreach ( $temp_data['sales'] as $year => $months ) {
					foreach ( $months as $month => $days ) {
						foreach ( $days as $day => $count ) {
							$date         = mktime( 0, 0, 0, $month, $day, $year ) * 1000;
							$sales_data[] = array( $date, $count );
						}
					}
				}

				foreach ( $temp_data['earnings'] as $year => $months ) {
					foreach ( $months as $month => $days ) {
						foreach ( $days as $day => $total ) {
							$date            = mktime( 0, 0, 0, $month, $day, $year ) * 1000;
							$earnings_data[] = array( $date, $total );
						}
					}
				}

				// Sort dates in ascending order
				foreach ( $sales_data as $key => $value ) {
					$timestamps[ $key ] = $value[0];
				}
				if ( ! empty( $timestamps ) ) {
					array_multisort( $timestamps, SORT_ASC, $sales_data );
				}

				foreach ( $earnings_data as $key => $value ) {
					$earnings_timestamps[ $key ] = $value[0];
				}
				if ( ! empty( $earnings_timestamps ) ) {
					array_multisort( $earnings_timestamps, SORT_ASC, $earnings_data );
				}

			// When showing more than 3 months of results, group them by month, by the first (except for the last month, group on the last day of the month selected)
			} else {

				foreach ( $temp_data['sales'] as $year => $months ) {
					$month_keys = array_keys( $months );
					$last_month = end( $month_keys );

					if ( $day_by_day ) {
						foreach ( $months as $month => $days ) {
							$day_keys = array_keys( $days );
							$last_day = end( $day_keys );

							$month_keys = array_keys( $months );

							$consolidated_date = $month === end( $month_keys ) ? wpb_cal_days_in_month( 'cal_gregorian', $month, $year ) : 1;

							$sales        = array_sum( $days );
							$date         = mktime( 0, 0, 0, $month, $consolidated_date, $year ) * 1000;
							$sales_data[] = array( $date, $sales );
						}
					} else {
						foreach ( $months as $month => $count ) {
							$month_keys = array_keys( $months );
							$consolidated_date = $month === end( $month_keys ) ? wpb_cal_days_in_month( 'cal_gregorian', $month, $year ) : 1;

							$date = mktime( 0, 0, 0, $month, $consolidated_date, $year ) * 1000;
							$sales_data[] = array( $date, $count );
						}
					}
				}

				// Sort dates in ascending order
				foreach ( $sales_data as $key => $value ) {
					$timestamps[ $key ] = $value[0];
				}
				if ( ! empty( $timestamps ) ) {
					array_multisort( $timestamps, SORT_ASC, $sales_data );
				}

				foreach ( $temp_data['earnings'] as $year => $months ) {
					$month_keys = array_keys( $months );
					$last_month = end( $month_keys );

					if ( $day_by_day ) {
						foreach ( $months as $month => $days ) {
							$day_keys = array_keys( $days );
							$last_day = end( $day_keys );

							$month_keys = array_keys( $months );

							$consolidated_date = $month === end( $month_keys ) ? wpb_cal_days_in_month( 'cal_gregorian', $month, $year ) : 1;

							$earnings        = array_sum( $days );
							$date            = mktime( 0, 0, 0, $month, $consolidated_date, $year ) * 1000;
							$earnings_data[] = array( $date, $earnings );
						}
					} else {
						foreach ( $months as $month => $count ) {
							$month_keys = array_keys( $months );
							$consolidated_date = $month === end( $month_keys ) ? wpb_cal_days_in_month( 'cal_gregorian', $month, $year ) : 1;

							$date = mktime( 0, 0, 0, $month, $consolidated_date, $year ) * 1000;
							$earnings_data[] = array( $date, $count );
						}
					}
				}

				// Sort dates in ascending order
				foreach ( $earnings_data as $key => $value ) {
					$earnings_timestamps[ $key ] = $value[0];
				}
				if ( ! empty( $earnings_timestamps ) ) {
					array_multisort( $earnings_timestamps, SORT_ASC, $earnings_data );
				}
			}
		}

		$data = array(
			__( 'Bookings', 'wp-base' ) => $sales_data,
			__( 'Revenue', 'wp-base' ) => $earnings_data,
		);

		$worked		= wpb_billable_hours( $args );
		$usable		= wpb_usable_hours( $args );
		$rate		= $usable ? intval( $worked/$usable *100 ) .'%' : 'N/A';
		$revenue	= wpb_format_currency( $earnings_totals, false, true );
		$qarg		= array(
			'page'				=> 'app_bookings',
			'app_location_id'	=> ! empty( $_GET['app_location_id'] ) ? $_GET['app_location_id'] : false,
			'app_service_id'	=> ! empty( $_GET['app_service_id'] ) ? $_GET['app_service_id'] : false,
			'app_worker_id'		=> ! empty( $_GET['app_worker_id'] ) ? $_GET['app_worker_id'] : false,
		);

		$classes = array_map( 'sanitize_html_class', apply_filters( 'app_dashboard_wrap_class', array(), $args, $qarg ) );

		ob_start();

		do_action( 'app_reports_graph_before' );
?>
		<div id="app-dashboard-wrap" <?php echo ( $classes ? 'class="'. implode( ' ', $classes ) .'"' : '') ?>>

			<div class="app-report-box metabox-holder">

				<div class="postbox dash-upcoming">
					<h3><?php _e( 'Upcoming Bookings', 'wp-base' ) ?></h3>
					<div class="app-count">
						<?php if ( self::is_vendor_page() ) { ?>
							<?php echo wpb_nof_bookings( $args, array( 'confirmed','paid','running','test' ) ); ?>
						<?php } else { ?>
							<a href="<?php echo add_query_arg( $qarg, admin_url( 'admin.php?status=active') ) ?>">
								<?php echo wpb_nof_bookings( $args, array( 'confirmed','paid','running','test' ) ); ?>
							</a>
						<?php } ?>
						<?php do_action( 'app_dashboard_after_box', $args, 'upcoming' ) ?>
					</div>
				</div>

				<div class="postbox dash-pending">
				<h3><?php _e( 'Pending Bookings', 'wp-base' ) ?></h3>
					<div class="app-count">
						<?php if ( self::is_vendor_page() ) { ?>
							<?php echo wpb_nof_bookings( $args, array( 'pending', 'test' ) ); ?>
						<?php } else { ?>
							<a href="<?php echo add_query_arg( $qarg, admin_url( 'admin.php?status=pending') ) ?>">
								<?php echo wpb_nof_bookings( $args, array( 'pending', 'test' ) ); ?>
							</a>
						<?php } ?>
						<?php do_action( 'app_dashboard_after_box', $args, 'pending' ) ?>
					</div>
				</div>

				<?php do_action( 'app_dashboard_after_pending', $args, $qarg ) ?>

				<div class="postbox dash-completed">
					<h3><?php _e( 'Completed Bookings', 'wp-base' ) ?></h3>
					<div class="app-count">
						<?php if ( self::is_vendor_page() ) { ?>
							<?php echo wpb_nof_bookings( $args, array( 'completed', 'test' ) ); ?>
						<?php } else { ?>
							<a href="<?php echo add_query_arg( $qarg, admin_url( 'admin.php?status=completed') ) ?>">
								<?php echo wpb_nof_bookings( $args, array( 'completed', 'test' ) ); ?>
							</a>
						<?php } ?>
						<?php do_action( 'app_dashboard_after_box', $args, 'completed' ) ?>
					</div>
				</div>

				<div class="postbox dash-billable">
					<h3><?php _e( 'Billable Hours', 'wp-base' ) ?></h3>
					<div class="app-count">
						<?php echo $worked; ?>
						<?php do_action( 'app_dashboard_after_box', $args, 'billable' ) ?>
					</div>
				</div>

				<div class="postbox dash-usable">
					<h3><?php _e( 'Usable Hours', 'wp-base' ) ?></h3>
					<div class="app-count">
						<?php echo $usable; ?>
						<?php do_action( 'app_dashboard_after_box', $args, 'usable' ) ?>
					</div>
				</div>

				<div class="postbox dash-utilization">
					<h3><?php _e( 'Utilization Rate', 'wp-base' ) ?></h3>
					<div class="app-count">
						<?php echo $rate; ?>
						<?php do_action( 'app_dashboard_after_box', $args, 'usable' ) ?>
					</div>
				</div>

				<?php if ( self::is_vendor_page() && function_exists('wpb_is_vendor') && wpb_is_vendor( get_current_user_id() ) ) { ?>
					<?php $total = wpb_commissions( array( 'worker' => get_current_user_id() ) ); ?>
					
					<div class="postbox dash-commissions">
						<h3><?php echo BASE()->get_text('bp_commissions') .' ('. wpb_format_currency( false, false ) .')'; ?></h3>
						<div class="app-count <?php echo ( strlen( $total ) > 6 ? 'app-crowded' : '' ); ?>">
							<?php echo $total ?>
							<?php do_action( 'app_dashboard_after_box', $args, 'commissions' ) ?>
						</div>
					</div>

				<?php } else { ?>

					<div class="postbox dash-revenue">
						<h3><?php printf( __( 'Revenue (%s)', 'wp-base' ), wpb_format_currency( false, false ) ) ?></h3>
						<div class="app-count <?php echo ( strlen( $revenue ) > 6 ? 'app-crowded' : '' ); ?>">
							<?php echo $revenue ?>
							<?php do_action( 'app_dashboard_after_box', $args, 'revenue' ) ?>
						</div>
					</div>

				<?php } ?>

			</div>

			<div class="app-graph-box metabox-holder">
				<div class="postbox">
					<h3><span><?php printf( __('Bookings by %s','wp-base' ), (!empty( $_GET['by_created'] ) ? __('Creation Time', 'wp-base') : __('Start Time', 'wp-base') ) ); ?></span></h3>

					<div class="inside">
						<?php
						include_once( WPBASE_PLUGIN_DIR . '/includes/lib/class.graph.php' );
						$graph = new WPB_Graph( $data );
						$graph->set( 'x_mode', 'time' );
						$graph->set( 'multiple_y_axes', true );
						$graph->display();

						do_action( 'app_reports_graph_additional_stats' );
						?>

					</div>
				</div>
			</div>

			<div class="app-report-box app-popular-box metabox-holder">

				<div class="postbox">
					<h3><?php _e( 'Popular Services by Bookings', 'wp-base' ) ?></h3>
					<div class="app-popular inside">
						<ul>
						<?php
							if ( $popular = wpb_popular_services( $args ) ) {
								foreach ( $popular as $pop ) {
									echo '<li><span class="service-name">'. wpb_cut( BASE()->get_service_name( $pop['service'] ) ) . '</span>';
									echo '<span class="number">'. $pop['total_seats'] . '</span></li>';
								}
							} else {
								echo '<li>'. BASE()->get_text( 'no_result' ) . '</li>';
							}
						?>
						</ul>
					</div>
				</div>

				<div class="postbox">
					<h3><?php _e( 'Popular Services by Revenue', 'wp-base' ) ?></h3>
					<div class="app-popular inside">
						<ul>
						<?php
							if ( $popular = wpb_popular_services_by_revenue( $args ) ) {
								foreach ( $popular as $pop ) {
									echo '<li><span class="service-name">'. wpb_cut( BASE()->get_service_name( $pop['service'] ) ) . '</span>';
									echo '<span class="number">'. wpb_format_currency( $pop['total'] ) . '</span></li>';
								}
							} else {
								echo '<li>'. BASE()->get_text( 'no_result' ) . '</li>';
							}
						?>
						</ul>
					</div>
				</div>

			<?php if ( current_user_can( WPB_ADMIN_CAP ) && BASE()->get_nof_workers() ) : ?>
				<div class="postbox">
					<h3><?php _e( 'Popular Providers by Bookings', 'wp-base' ) ?></h3>
					<div class="app-popular inside">
						<ul>
						<?php
							if ( $popular = wpb_popular_workers( $args ) ) {
								foreach ( $popular as $pop ) {
									echo '<li><span class="service-name">'. BASE()->get_worker_name( $pop['worker'] ) . '</span>';
									echo '<span class="number">'. $pop['total_seats'] . '</span></li>';
								}
							} else {
								echo '<li>'. BASE()->get_text( 'no_result' ) . '</li>';
							}
						?>
						</ul>
					</div>
				</div>

				<div class="postbox">
					<h3><?php _e( 'Popular Providers by Revenue', 'wp-base' ) ?></h3>
					<div class="app-popular inside">
						<ul>
						<?php
							if ( $popular = wpb_popular_workers_by_revenue( $args ) ) {
								foreach ( $popular as $pop ) {
									echo '<li><span class="service-name">'. BASE()->get_worker_name( $pop['worker'] ) . '</span>';
									echo '<span class="number">'. wpb_format_currency( $pop['total'] ) . '</span></li>';
								}
							} else {
								echo '<li>'. BASE()->get_text( 'no_result' ) . '</li>';
							}
						?>
						</ul>
					</div>
				</div>
			<?php endif; ?>

				<div class="postbox">
					<h3><?php _e( 'Popular Clients by Bookings', 'wp-base' ) ?></h3>
					<div class="app-popular inside">
						<ul>
						<?php
							if ( $popular = wpb_popular_clients( $args ) ) {
								foreach ( $popular as $pop ) {
									echo '<li><span class="service-name">'. BASE('User')->get_name( $pop['user'] ) . '</span>';
									echo '<span class="number">'. $pop['total_seats'] . '</span></li>';
								}
							} else {
								echo '<li>'. BASE()->get_text( 'no_result' ) . '</li>';
							}
						?>
						</ul>
					</div>
				</div>

				<div class="postbox">
					<h3><?php _e( 'Popular Clients by Revenue', 'wp-base' ) ?></h3>
					<div class="app-popular inside">
						<ul>
						<?php
							if ( $popular = wpb_popular_clients_by_revenue( $args ) ) {
								foreach ( $popular as $pop ) {
									echo '<li><span class="service-name">'. BASE('User')->get_name( $pop['user'] ) . '</span>';
									echo '<span class="number">'. wpb_format_currency( $pop['total'] ) . '</span></li>';
								}
							} else {
								echo '<li>'. BASE()->get_text( 'no_result' ) . '</li>';
							}
						?>
						</ul>
					</div>
				</div>

			</div>
		</div>

		<?php do_action( 'app_reports_graph_after' );

		$output = ob_get_contents();
		ob_end_clean();

		return $output;
	}


	/**
	 * Show report graph date filters
	 *
	 * @since 1.3
	 * @return void
	*/
	public static function graph_controls() {
		global $wp_locale;

		$date_options = apply_filters( 'app_report_date_options', array(
			'yesterday'    => __( 'Yesterday', 'wp-base' ),
			'today'        => __( 'Today', 'wp-base' ),
			'tomorrow'	   => __( 'Tomorrow', 'wp-base' ),
			'last_week'    => __( 'Last Week', 'wp-base' ),
			'this_week'    => __( 'This Week', 'wp-base' ),
			'next_week'    => __( 'Next Week', 'wp-base' ),
			'last_30_days' => __( 'Last 30 Days', 'wp-base' ),
			'last_month'   => __( 'Last Month', 'wp-base' ),
			'this_month'   => __( 'This Month', 'wp-base' ),
			'next_month'   => __( 'Next Month', 'wp-base' ),
			'last_year'    => __( 'Last Year', 'wp-base' ),
			'this_year'    => __( 'This Year', 'wp-base' ),
			'custom'       => __( 'Custom', 'wp-base' )
		) );

		$dates   = self::get_report_dates();
		$display = $dates['range'] == 'custom' ? '' : 'style="display:none;"';

		if( empty( $dates['day_end'] ) ) {
			$dates['day_end'] = wpb_cal_days_in_month( 'cal_gregorian', date( 'n' ), date( 'Y' ) );
		}

		# Get sanitized filter params
		$filt = wpb_sanitize_search();

?>

<form id="app-graphs-filter" method="get" action="">
	<div class="tablenav top">
		<div class="app-actions actions">
		<?php if ( is_admin() ) { ?>
			<input type="hidden" name="page" value="appointments"/>
		<?php } else { ?>
			<input type="hidden" name="tab" value="dashboard"/>
		<?php } ?>
		<?php do_action( 'app_dashboard_form_fields', $dates ); ?>

			<div class="app-dash-title"><span><?php echo (self::is_vendor_page() ? BASE()->get_text('dashboard') : sprintf( __('%s Dashboard','wp-base'), WPB_NAME ) ); ?></span></div>

			<div class="app-reports-group">
				<div id="app-date-select">
					<select id="app-graphs-date-options" name="range">
						<option disabled><?php _e( 'Filter by date', 'wp-base' ) ?></option>
					<?php foreach ( $date_options as $key => $option ) : ?>
							<option value="<?php echo esc_attr( $key ); ?>"<?php selected( $key, $dates['range'] ); ?>><?php echo esc_html( $option ); ?></option>
						<?php endforeach; ?>
					</select>
				</div>

				<div id="app-date-range-options" <?php echo $display; ?>>
					<span><?php _e( 'From', 'wp-base' ); ?>&nbsp;</span>
					<select id="app-graphs-month-start" name="m_start">
						<?php for ( $i = 1; $i <= 12; $i++ ) : ?>
							<option value="<?php echo absint( $i ); ?>" <?php selected( $i, $dates['m_start'] ); ?>><?php echo $wp_locale->get_month_abbrev( $wp_locale->get_month( $i ) ); ?></option>
						<?php endfor; ?>
					</select>
					<select id="app-graphs-day-start" name="day_start">
						<?php for ( $i = 1; $i <= 31; $i++ ) : ?>
							<option value="<?php echo absint( $i ); ?>" <?php selected( $i, $dates['day_start'] ); ?>><?php echo $i; ?></option>
						<?php endfor; ?>
					</select>
					<select id="app-graphs-year-start" name="year_start">
						<?php for ( $i = 2007; $i <= date( 'Y' ); $i++ ) : ?>
							<option value="<?php echo absint( $i ); ?>" <?php selected( $i, $dates['year_start'] ); ?>><?php echo $i; ?></option>
						<?php endfor; ?>
					</select>
					<span><?php _e( 'To', 'wp-base' ); ?>&nbsp;</span>
					<select id="app-graphs-month-end" name="m_end">
						<?php for ( $i = 1; $i <= 12; $i++ ) : ?>
							<option value="<?php echo absint( $i ); ?>" <?php selected( $i, $dates['m_end'] ); ?>><?php echo $wp_locale->get_month_abbrev( $wp_locale->get_month( $i ) ); ?></option>
						<?php endfor; ?>
					</select>
					<select id="app-graphs-day-end" name="day_end">
						<?php for ( $i = 1; $i <= 31; $i++ ) : ?>
							<option value="<?php echo absint( $i ); ?>" <?php selected( $i, $dates['day_end'] ); ?>><?php echo $i; ?></option>
						<?php endfor; ?>
					</select>
					<select id="app-graphs-year-end" name="year_end">
						<?php for ( $i = 2007; $i <= date( 'Y' ); $i++ ) : ?>
						<option value="<?php echo absint( $i ); ?>" <?php selected( $i, $dates['year_end'] ); ?>><?php echo $i; ?></option>
						<?php endfor; ?>
					</select>
				</div>
			</div>

			<div class="app-reports-group">
			<?php
			if ( ! self::is_vendor_page() && $locations = BASE()->get_locations( 'name' ) ) {
				$add_class = $filt['location_id'] ? 'class="app-option-selected"' : '';
			?>
				<select name="app_location_id" <?php echo $add_class ?>>
					<option value=""><?php _e('Filter by location','wp-base'); ?></option>
				<?php
					foreach ( $locations as $location ) {
						$selected = $filt['location_id'] == $location->ID ? " selected='selected' " : '';
						echo '<option '.$selected.' value="'.$location->ID.'">'. BASE()->get_location_name( $location->ID ) .'</option>';
					}
				?>
				</select>
				<?php }

				$add_class = $filt['service_id'] ? 'class="app-option-selected"' : '';
				?>
				<select name="app_service_id" <?php echo $add_class ?>>
					<option value=""><?php _e('Filter by service','wp-base'); ?></option>
					<?php
					$services = self::is_vendor_page() ? BASE()->get_services_by_worker( 'name' ) : BASE()->get_services( 'name' );

					foreach ( (array)$services as $service ) {
						if ( BASE()->is_package( $service->ID ) )
							continue;

						$selected = $filt['service_id'] == $service->ID ? " selected='selected' " : '';
						echo '<option '.$selected.' value="'.$service->ID.'">'. BASE()->get_service_name( $service->ID ) .'</option>';
					}
					?>
				</select>
				<?php

			if ( ! self::is_vendor_page() ) {
				$workers = BASE()->get_workers( 'name' );
				$add_class = $filt['worker_id'] || "0" === (string)$filt['worker_id'] ? 'class="app-option-selected"' : '';
				?>
				<select name="app_worker_id" <?php echo $add_class ?>>
					<option value=""><?php _e('Filter by provider','wp-base'); ?></option>
				<?php if ( $workers ) {  ?>
					<option value="0" <?php selected( $filt['worker_id'], 0 ) ?>><?php _e('Unassigned','wp-base'); ?></option>
				<?php
					foreach ( $workers as $worker ) {
						$selected = $filt['worker_id'] == $worker->ID ? " selected='selected' " : '';
						echo '<option '.$selected.' value="'.$worker->ID.'">'. BASE()->get_worker_name( $worker->ID ) .'</option>';
					}
				}
				?>
				</select>
			<?php } ?>
			</div>

			<div class="app-graph-filter-options graph-option-section">
				<input type="checkbox" id="by_created" <?php checked( true, !empty( $_GET['by_created'] ), true ); ?> value="1" name="by_created" />
				<label for="exclude_taxes"><?php _e( 'By Creation Time', 'wp-base' ); ?></label>
			</div>

			<div class="app-graph-filter-submit graph-option-section">
				<input type="hidden" name="action_app" value="filter_reports" />
				<input type="submit" class="button-secondary" value="<?php _e( 'Filter', 'wp-base' ); ?>"/>
			</div>

		</div>
	</div>
</form>
<?php
	}

}
	BASE('Dashboard')->add_hooks();
}