<?php

namespace Zprint\Admin;

use Zprint\Model\Location;
use Zprint\Admin;
use const Zprint\PLUGIN_ROOT_FILE;
use const Zprint\PLUGIN_VERSION;

class OrdersTable
{
    private const COLUMN_ACTIONS_KEY = 'zprint_actions';
    private const BULK_PRINT_KEY = 'zprint_print';

    public function __construct()
    {
        add_filter(
            'manage_shop_order_posts_columns',
            [$this, 'addColumns'],
            99
        );
        add_filter(
            'bulk_actions-edit-shop_order',
            [$this, 'addBulkActions'],
            1
        );
        add_action(
            'manage_shop_order_posts_custom_column',
            [$this, 'renderPrintColumn'],
            99,
            2
        );
        add_filter(
            'handle_bulk_actions-edit-shop_order',
            [$this, 'handleBulkPrintAction'],
            10,
            3
        );
        if (isset($_GET['bulk_print'])) {
            $this->renderBulkPrintDialog();
        }
        add_action('admin_enqueue_scripts', [$this, 'enqueueAssets'], 99);
    }

    public function addColumns(array $columns): array
    {
        $columns[self::COLUMN_ACTIONS_KEY] = __(
            'Print',
            'Print-Google-Cloud-Print-GCP-WooCommerce'
        );

        return $columns;
    }

    public function addBulkActions(array $actions): array
    {
        $actions[self::BULK_PRINT_KEY] = __(
            'Print',
            'Print-Google-Cloud-Print-GCP-WooCommerce'
        );

        return $actions;
    }

    public function renderPrintColumn(string $column, int $post_id)
    {
        if (self::COLUMN_ACTIONS_KEY !== $column) {
            return;
        }

        $locations = Location::getAllFormatted();

        if (empty($locations)) {
            return;
        }

        $dialog_id = Admin\PrintDialog::render($locations, $post_id, 'post');
        $btn_label = __('Print', 'Print-Google-Cloud-Print-GCP-WooCommerce');
        $attrs =
            ' class="zprint-print-btn thickbox button wc-action-button help_tip far fa-print"';
        $attrs .= ' href="#TB_inline&inlineId=' . $dialog_id . '"';
        $attrs .= ' aria-label="' . esc_attr($btn_label) . '"';
        $attrs .= ' data-tip="' . esc_attr($btn_label) . '"';

        echo '<p><a' . $attrs . '></a></p>';
    }

    public function handleBulkPrintAction(
        string $redirect_to,
        string $action,
        array $post_ids
    ): string {
        if (self::BULK_PRINT_KEY !== $action) {
            return $redirect_to;
        }

        return add_query_arg(
            'bulk_print',
            wp_json_encode($post_ids),
            $redirect_to
        );
    }

    private function renderBulkPrintDialog()
    {
        global $pagenow;

        if (
            'edit.php' !== $pagenow ||
            empty($_GET['post_type']) ||
            'shop_order' !== $_GET['post_type']
        ) {
            return;
        }

        $locations = Location::getAllFormatted();
        $post_ids = json_decode($_GET['bulk_print']);

        if (empty($locations) || !is_array($post_ids)) {
            return;
        }

        PrintDialog::render($locations, 0, 'post', $post_ids);
    }

    public function enqueueAssets()
    {
        global $pagenow;

        if ('edit.php' === $pagenow && 'shop_order' === $_GET['post_type']) {
            add_thickbox();
            wp_enqueue_style(
                'zprint-orders-table-fa',
                plugins_url('assets/fa/app.css', PLUGIN_ROOT_FILE),
                [],
                PLUGIN_VERSION
            );
        }
    }
}
