<?php

/**
 * Create a shortcode to one or all reviews
 * @since 0.1
 */
if ( !function_exists( 'grfwp_reviews_shortcode' ) ) {
function grfwp_reviews_shortcode( $atts ) {

	// We must disable the automatic append to content filter while we're
	// rendering the shortcode to prevent it from displaying twice
	global $grfwp_controller;
	remove_action( 'the_content', array( $grfwp_controller, 'append_to_content' ) );

	$output = grfwp_print_reviews(
		shortcode_atts(
			array(
				'review' 	=> null,
				'category' 	=> null,
				'random'	=> false,
				'limit'		=> null,
				'cycle'		=> false,
				'excerpt'	=> false,
				'paged'		=> isset( $_REQUEST['page'] ) ? intval( $_REQUEST['page'] ) : 1
			),
			$atts
		)
	);

	// Restore the_content filter
	add_action( 'the_content', array( $grfwp_controller, 'append_to_content' ) );

	return $output;
}
add_shortcode( 'good-reviews', 'grfwp_reviews_shortcode' );
add_shortcode( 'five-star-reviews', 'grfwp_reviews_shortcode' );
} // endif;

/**
 * Create a shortcode to allow reviews to be submitted
 * @since 2.0.0
 */
if ( !function_exists( 'grfwp_submit_reviews_shortcode' ) ) {
function grfwp_submit_reviews_shortcode( $atts ) {

	$output = grfwp_print_submit_review_form();

	return $output;
}
add_shortcode( 'submit-good-review', 'grfwp_submit_reviews_shortcode' );
add_shortcode( 'five-star-submit-review', 'grfwp_submit_reviews_shortcode' );
} // endif;

/*
 * Print reviews
 * @since 0.1
 */
if ( !function_exists( 'grfwp_print_reviews' ) ) {
function grfwp_print_reviews( $args = array() ) {
	global $grfwp_controller;

	$output = apply_filters( 'grfwp_print_reviews_output', false, $args );
	if ( $output !== false ) {
		return $output;
	}

	$output = '';
	
	$grfwp_controller->get_query_args( $args );

	$reviews = new WP_Query( $grfwp_controller->args );

	if ( $reviews->have_posts() ) :

		// Enqueue the frontend stylesheet
		if ( apply_filters( 'grfwp-load-frontend-assets', true ) ) {
			wp_enqueue_style( 'dashicons' );
			wp_enqueue_style( 'gr-reviews' );

			if ( $grfwp_controller->settings->get_setting('grfwp-styling-layout') == 'default' ) {
				wp_enqueue_style( 'grfwp-default-css', GRFWP_PLUGIN_URL . '/assets/css/default.css' );
			}

			if ( $grfwp_controller->settings->get_setting('grfwp-styling-layout') == 'thumbnail' ) {
				wp_enqueue_style( 'grfwp-thumbnail-css', GRFWP_PLUGIN_URL . '/assets/css/thumbnail.css' );
			}

			if ( $grfwp_controller->settings->get_setting('grfwp-styling-layout') == 'image' ) {
				wp_enqueue_style( 'grfwp-image-css', GRFWP_PLUGIN_URL . '/assets/css/image.css' );
				wp_enqueue_script( 'grfwp-image-js', GRFWP_PLUGIN_URL . '/assets/js/image.js', array( 'jquery' ), GRFWP_VERSION, true );
			}

		}

		// Set up the classes for styling
		$classes = array();
		$classes[] = 'gr-reviews';
		$classes[] = empty( $grfwp_controller->args['p'] ) ? 'gr-reviews-all' : 'gr-reviews-single';
		$classes[] = apply_filters( 'grfwp_microformat_css_class', 'hentry' );


		// Enqueue the frontend script if required
		if ( !empty( $grfwp_controller->args['cycle'] ) && $reviews->found_posts > 1 ) {

			$id = count( $grfwp_controller->ids );
			array_push( $grfwp_controller->ids, $id );

			wp_enqueue_script( 'gr-reviews' );

			$cycle_data = array(
				'ids' 	=> $grfwp_controller->ids,
				'delay' => apply_filters( 'grfwp_review_cycle_delay', 8000 )
			);

			$grfwp_controller->add_front_end_php_data( 'gr-reviews', 'grfwp_cycle', $cycle_data );

			$classes[] = 'gr-reviews-cycle';
		}

		// Get information about this site to use in the itemReviewed schema.
		$grfwp_controller->get_reviewed_item();

		ob_start();

		// Custom styling
		echo grfwp_add_custom_styling();

		foreach ( $reviews->posts as $review ) {

			echo grfwp_print_review( $review );
		}

		if ( $reviews->max_num_pages > 1 ) {

			$grfwp_controller->add_front_end_php_data( 'gr-pagination', 'grfwp_reviews_shortcode_args', $args );

			wp_enqueue_script( 'gr-pagination' );

			?>

			<div id='grfwp-pagination-controls'>

				<div id='grfwp-pagination-load-more' data-next_page='2'>
					<?php echo esc_html( $grfwp_controller->settings->get_setting( 'label-show-more' ) ); ?>
				</div>

			</div>

		<?php

		}

		$output .= ob_get_clean();

	endif;

	// Reset the WP_Query() loop
	wp_reset_postdata();

	return $output;

}
} // endif;

if ( !function_exists( 'grfwp_print_review' ) ) { 
function grfwp_print_review( $review ) {
	global $grfwp_controller;

	$review = apply_filters('grfwp_print_review_obj', $review);

	$grfwp_controller->cpts->get_post_metadata( $review->ID );
	$post_meta = $grfwp_controller->cpts->post_metadata;
	$post_meta['img'] = get_the_post_thumbnail( $review->ID, apply_filters( 'grfwp_the_post_thumbnail_size', 'thumbnail' ) );

	// Set
	if ( !empty( $grfwp_controller->args['excerpt'] ) ) {
		$post_meta['review_date'] = '';
		$post_meta['review_url'] = '';
	}

	$post_meta = apply_filters( 'grfwp_post_meta', $post_meta );

	// Store css classes to adjust layout
	$classes = grfwp_get_review_css_classes( $post_meta );

	// Get information about this site to use in the itemReviewed schema.
	$grfwp_controller->get_reviewed_item();

	//return the output instead of printing it
	ob_start();

	?>

	<div <?php echo $grfwp_controller->format_classes( $classes ); ?> itemscope itemtype="http://schema.org/Review">
		<?php grfwp_the_schema_item_reviewed( $grfwp_controller->reviewed ); ?>

		<div class="gr-content">

			<?php if ( !empty( $post_meta['img'] ) ) : ?>
				<div class="gr-author-img">
					<?php echo wp_kses_post( $post_meta['img'] ); ?>
				</div>
			<?php endif; ?>

			<?php if ( grfwp_has_rating() ) : ?>

				<div class="gr-review-rating gr-rating-<?php echo esc_attr( $post_meta['rating_display'] ); ?>" itemprop="reviewRating" itemscope itemtype="http://schema.org/Rating">
					<meta itemprop="worstRating" content="1">

					<?php if ( $post_meta['rating_display'] == 'numbers' ) : ?>
					<span itemprop="ratingValue"><?php echo esc_attr( $post_meta['rating'] ); ?></span> /
					<span itemprop="bestRating"><?php echo esc_attr( $post_meta['rating_max'] ); ?></span>

					<?php elseif ( $post_meta['rating_display'] == 'stars' ) : ?>
					<meta itemprop="ratingValue" content="<?php echo esc_attr( $post_meta['rating'] ); ?>">
					<meta itemprop="bestRating" content="<?php echo esc_attr( $post_meta['rating_max'] ); ?>">
					<span class="screen-reader-text"><?php echo esc_attr( $post_meta['rating'] ) . '/' . esc_attr( $post_meta['rating_max'] ); ?></span>
					<?php echo str_repeat( grfwp_the_star( true ), $post_meta['rating'] ); ?>
					<?php echo str_repeat( grfwp_the_star( false ), max( ( $post_meta['rating_max'] - $post_meta['rating'] ), 0 ) ); ?>

					<?php endif; ?>
				</div>

			<?php endif; ?>

			<div class="gr-author" itemprop="author" itemscope itemtype="http://schema.org/Person">
				<div class="gr-author-text">
					<span class="gr-author-icon"></span>
					<span class="gr-author-name" itemprop="name"><?php echo esc_html( $review->post_title ) ?></span>

					<?php if ( $post_meta['reviewer_org'] ) : ?>
						<div class="grfwp-clear"></div>
						<span class="gr-organization-icon"></span>
						<div class="gr-author-affiliation" itemprop="affiliation" itemscope itemtype="http://schema.org/Organization">

							<?php if ( $post_meta['reviewer_url'] ) : ?>
							<a class="gr-author-url" itemprop="url" href="<?php echo esc_attr( $post_meta['reviewer_url'] ); ?>">
								<span itemprop="name"><?php echo esc_html( $post_meta['reviewer_org'] ); ?></span>
							</a>
							<?php else : ?>
							<span class="gr-author-org" itemprop="name"><?php echo esc_html( $post_meta['reviewer_org'] ); ?></span>
							<?php endif; ?>

						</div>
					<?php endif; ?>

					<?php if ( $post_meta['review_date'] ) : ?>
						<div class="grfwp-clear"></div>
						<span class="gr-review-date-icon"></span>
						<span class="gr-review-date" itemprop="datePublished"><?php echo esc_html( $post_meta['review_date'] ); ?></span>
					<?php endif; ?>

				</div>
			</div> <!-- gr-author -->

			<div class="gr-review-body" itemprop="reviewBody">
			<?php 
				if ( !empty( $grfwp_controller->args['excerpt'] ) ) {
					echo wp_kses_post( $review->post_excerpt );
				}
				else {
					echo wp_kses_post( $review->post_content );
				}
			?>
			</div>

			<div class="gr-custom-fields">
				<?php if ( $grfwp_controller->settings->get_setting('grfwp-cf-status') ) : ?>

				  	<?php 
				  	$fields = @json_decode( $grfwp_controller->settings->get_setting('grfwp-custom-fields') ) ?? array();
				  	foreach ( $fields as $field ) {

				  		if ( array_key_exists( sanitize_title( $field->cf_field_name ), $post_meta ) ) {
				  		?>
				  			<div class="gr-review-custom-field">
				  				<span class="gr-review-custom-field-label"><?php echo esc_html( $field->cf_field_name ); ?></span><span class="gr-review-custom-field-colon">:</span> <span class="gr-review-custom-field-content"><?php echo esc_html( $post_meta[ sanitize_title( $field->cf_field_name ) ] ); ?></span>
				  			</div>
				  		<?php
				  		}
				  	}
				endif; ?>
			</div> <!-- gr-custom-fields -->

			<?php
			if ( empty( $grfwp_controller->args['excerpt'] ) ) :
				if ( $post_meta['review_url'] ) : ?>
					<a class="gr-review-url" itemprop="url" href="<?php echo esc_attr( $post_meta['review_url'] ); ?>">
						<?php echo __( 'Read More', 'good-reviews-wp'); ?>
					</a>
				<?php endif;
			else :
			?>
				<a class="gr-review-url" itemprop="sameAs" href="<?php echo get_the_permalink( $review->ID ); ?>">
					<?php echo __( 'Read More', 'good-reviews-wp'); ?>
				</a>
			<?php endif; ?>

		</div>

	</div> <!-- main container for each review -->

	<?php 

	$output = ob_get_clean();

	return $output;
}
}

/**
 * Display the submit review form 
 * @since 2.0.0
 */
if ( !function_exists( 'grfwp_print_submit_review_form' ) ) {
function grfwp_print_submit_review_form( $args = array() ) {
	global $grfwp_controller;
	
	if ( isset($_POST['grfwp_submit_review']) ) {
		$submit_review_response = grfwp_handle_submitted_review();
	}
	else { $submit_review_response = false; }

	// Capture output to return in one string
	ob_start();

	?>
	<div class="grfwp-submit-review">
		<form method="post">
			<?php wp_nonce_field( 'grfwp_submit_review', 'grfwp_submit_review_nonce' ); ?>

			<?php if ( isset($args['fdm_menu_item_id']) ) { ?>
				<input type="hidden" name="fdm_menu_item_id" value="<?php echo esc_html( $args['fdm_menu_item_id'] ); ?>" />
			<?php } ?>

			<?php if ( $submit_review_response ) { ?>
				<h5><?php echo $submit_review_response['message']; ?></h5>
			<?php } ?>

			<label for="grfwp_rating"><?php echo esc_html( $grfwp_controller->settings->get_setting( 'label-rating' ) ); ?></label>
			<input type="number" name="grfwp_rating" value="<?php echo esc_attr( isset($_POST['grfwp_rating']) ? intval( $_POST['grfwp_rating'] ) : '' ); ?>" min='1' max='5' />

			<label for="grfwp_reviewer"><?php echo esc_html( $grfwp_controller->settings->get_setting( 'label-review-author' ) ); ?></label>
			<input type="text" name="grfwp_reviewer" value="<?php echo esc_attr( isset($_POST['grfwp_reviewer']) ? sanitize_text_field( $_POST['grfwp_reviewer'] ) : '' ); ?>" />

			<label for="grfwp_review"><?php echo esc_html( $grfwp_controller->settings->get_setting( 'label-review' ) ); ?></label>
			<textarea name="grfwp_review">
				<?php echo esc_attr( isset($_POST['grfwp_review']) ? sanitize_textarea_field( $_POST['grfwp_review'] ) : '' ); ?>
			</textarea>

			<?php if ( $grfwp_controller->settings->get_setting('grfwp-cf-status') ) { ?>
				<?php 
				$fields = @json_decode( $grfwp_controller->settings->get_setting('grfwp-custom-fields') ) ?? array();
				foreach ( $fields as $field ) {

					$field_name = sanitize_title( $field->cf_field_name );

					$required = $field->cf_required == 'yes' ? 'required' : '';
					?>

					<label for='grfwp_review-<?php echo $field_name; ?>'><?php echo esc_html( $field->cf_field_name ); ?></label>

					<?php 
					switch ( $field->cf_type ) {

						case 'number':
	  						echo "<input type='number' " . $required . " id='grfwp_review-" . $field_name . "' name='" . $field_name . "' value='" . ( isset( $_POST[ $field_name ] ) ? esc_attr( sanitize_text_field( $_POST[ $field_name ] ) ) : '' ) . "' >";
	  						break;

						case 'text_box':
	  						echo "<input type='text' " . $required . " id='grfwp_review-" . $field_name . "' name='" . $field_name . "' value='" . ( isset( $_POST[ $field_name ] ) ? esc_attr( sanitize_text_field( $_POST[ $field_name ] ) ) : '' ) . "' />";
	  						break;

	  					case 'text_area':
	  						echo "<textarea " . $required . " id='grfwp_review-" . $field_name . "' name='" . $field_name . "'> " . ( isset( $_POST[ $field_name ] ) ? esc_attr( sanitize_text_field( $_POST[ $field_name ] ) ) : '' ) . "</textarea>";
	  						break;

	  					case 'dropdown':
	  						$options = explode( ',', $field->cf_options );
	  						echo "<select " . $required . " id='grfwp_review-" . $field_name . "' name='" . $field_name . "'>";
	  							foreach ( $options as $value ) {
	  								echo "<option value='" . $value . "'>" . $value . "</option>";
	  							}
	  						echo '</select>';
	  						break;

	  					case 'date':
	  						echo "<input " . $required . " type='date' id='grfwp_review-" . $field_name . "' name='" . $field_name . "' " . ( isset( $_POST[ $field_name ] ) ?  esc_attr( sanitize_text_field( $_POST[ $field_name ] ) ) : '' ) . "' class='datepicker'>";
	  						break;
	  	
	  					default:
	  						# code...
	  						break;
					} 
					?>

				<?php  } ?>

			<?php } ?>

			<?php do_action( 'grfwp-submit-review-form' ); ?>

			<input type="submit" name="grfwp_submit_review" value="<?php echo esc_attr( $grfwp_controller->settings->get_setting( 'label-submit-review' ) ); ?>" />
		</form>
	</div>

	<?php

	$output = ob_get_clean();

	return $output;
}
} //endif;



/**
 * Handle submitted reviews
 * @since 2.0.0
 */
if ( !function_exists( 'grfwp_handle_submitted_review' ) ) {
function grfwp_handle_submitted_review() {
	global $grfwp_controller;
	
	if ( ! isset( $_POST['grfwp_submit_review_nonce'] ) || ! wp_verify_nonce( sanitize_key( $_POST['grfwp_submit_review_nonce'] ), 'grfwp_submit_review' ) ) { 
		
		$response = array(
			'message' => 'There was a problem with the review validation. Please try to submit your review again.',
			'status'  => 'fail'
		);

		return $response;
	}

	if ( ! isset( $_POST['grfwp_rating'] ) || ! is_numeric($_POST['grfwp_rating']) ) { 
		
		$response = array(
			'message' => 'Review rating should be a number.',
			'status'  => 'fail'
		);

		return $response;
	}

	if ( $grfwp_controller->settings->get_setting( 'grfwp-cf-status' ) ) {

		$fields = @json_decode( $grfwp_controller->settings->get_setting( 'grfwp-custom-fields' ) ) ?? array();
		foreach ( $fields as $field ) {
			if ( $field->cf_required && empty( $_POST[ sanitize_title( $field->cf_field_name ) ] ) ) {
				$response = array(
						'message' => $field->cf_field_name . ' is required',
						'status'  => 'fail'
				);

				return $response;
			}
			if ( 'review_line' == $field->cf_type && ( $_POST[ sanitize_title( $field->cf_field_name ) ] > 5 ) ) {
				$response = array(
						'message' => $field->cf_field_name . ' is required',
						'status'  => 'fail'
				);

				return $response;
			}
		}
	}

	$post_data = array(
		'post_title' => sanitize_text_field( $_POST['grfwp_reviewer'] ),
		'post_content' => sanitize_textarea_field( $_POST['grfwp_review'] ),
		'post_status' => 'publish',
		'post_type' => 'grfwp-review'
	);

	$post_id = wp_insert_post( apply_filters( 'grfwp-submit-review-post', $post_data ) );

	if ( $post_id ) {

		$review_meta = array(
			'rating' => sanitize_text_field( $_POST['grfwp_rating'] ),
			'rating_display' => 'stars',
			'rating_max' => 5
		);

		if ( $grfwp_controller->settings->get_setting( 'grfwp-cf-status' ) ) {

			$fields = @json_decode( $grfwp_controller->settings->get_setting( 'grfwp-custom-fields' ) ) ?? array();
			foreach ( $fields as $field ) {
				
				$review_meta[ sanitize_title( $field->cf_field_name ) ] = isset( $_POST[ sanitize_title( $field->cf_field_name ) ] ) ? sanitize_text_field( $_POST[ sanitize_title( $field->cf_field_name ) ] ) : '';

				unset( $_POST[ sanitize_title( $field->cf_field_name ) ] );
			}
		}

		update_post_meta( $post_id, 'grfwp', apply_filters( 'grfwp-submit-review-post-meta', $review_meta, $post_id ) );

		unset( $_POST['grfwp_reviewer'] );
		unset( $_POST['grfwp_review'] );
		unset( $_POST['grfwp_rating'] );

		$response = array(
			'message' => 'Review has been sucessfully submitted!',
			'status'  => 'success'
		);

		return $response;
	}

	$response = array(
		'message' => 'Review failed to submit. Please try again.',
		'status'  => 'fail'
	);

	return $response;
}
} //endif;



/**
 * Print the schema.org output for the item being reviewed
 * @since 0.0.1
 */
if ( !function_exists( 'grfwp_the_schema_item_reviewed' ) ) {
function grfwp_the_schema_item_reviewed( $item ) {
	?>
	<div itemprop="itemReviewed" itemscope itemtype="http://schema.org/<?php echo esc_attr( $item['schema'] ); ?>">
		<meta itemprop="name" content="<?php echo esc_attr( $item['name'] ); ?>">
		<meta itemprop="description" content="<?php echo esc_attr( $item['description'] ); ?>">
		<meta itemprop="url" content="<?php echo esc_attr( $item['url'] ); ?>">
	</div>
	<?php
}
} //endif;

/**
 * Specify CSS classes that should be added to each review for easier styling
 * @since 0.0.1
 */
if ( !function_exists( 'grfwp_get_review_css_classes' ) ) {
function grfwp_get_review_css_classes( $post_meta ) {

	$classes = array( 'gr-review' );

	$classes[] = $post_meta['img'] ? 'gr-item-has-image' : 'gr-item-no-image';
	$classes[] = ( $post_meta['rating_display'] && $post_meta['rating'] ) ? 'gr-review-has-rating gr-review-display-' . $post_meta['rating_display'] : '';

	$classes = apply_filters( 'grfwp_review_css_classes', $classes );

	return $classes;
}
} //endif;

/**
 * Wrapper function to check if a review has a rating
 * @since 0.0.1
 */
if ( !function_exists( 'grfwp_has_rating' ) ) {
function grfwp_has_rating() {
	global $grfwp_controller;
	return $grfwp_controller->cpts->has_rating();
}
} //endif;

/**
 * Output HTML markup for a rating star
 * @var $state = boolean; true is filled, false is empty
 * @since 0.0.1
 */
if ( !function_exists( 'grfwp_the_star' ) ) {
function grfwp_the_star( $state ) {
	if ( $state ) {
		return apply_filters( 'grfwp_star_html_filled', '<span class="dashicons dashicons-star-filled"></span>' );
	} else {
		return apply_filters( 'grfwp_star_html_empty', '<span class="dashicons dashicons-star-empty"></span>' );
	}
}
} //endif;



if ( !function_exists( 'grfwp_add_custom_styling' ) ) {
	function grfwp_add_custom_styling() {
		global $grfwp_controller;
		$styling = '<style>';
			if ( $grfwp_controller->settings->get_setting('grfwp-styling-layout') == 'image' && $grfwp_controller->settings->get_setting('grfwp-styling-image-hover') == 0 ) { $styling .=  '.gr-author { display: none !important; }'; }
			if ( $grfwp_controller->settings->get_setting('grfwp-styling-stars-color') != '' ) { $styling .=  '.gr-review .dashicons-star-filled, .gr-review .dashicons-star-empty { color: ' . $grfwp_controller->settings->get_setting('grfwp-styling-stars-color') . ' !important; }'; }
			if ( $grfwp_controller->settings->get_setting('grfwp-styling-stars-color') != '' ) { $styling .=  '.gr-rating-numbers { color: ' . $grfwp_controller->settings->get_setting('grfwp-styling-stars-color') . ' !important; }'; }
			if ( $grfwp_controller->settings->get_setting('grfwp-styling-rating-font-size') != '' ) { $styling .=  '.gr-rating-numbers { font-size: ' . $grfwp_controller->settings->get_setting('grfwp-styling-rating-font-size') . ' !important; }'; }
			if ( $grfwp_controller->settings->get_setting('grfwp-styling-review-text-font-size') != '' ) { $styling .=  '.gr-review-body { font-size: ' . $grfwp_controller->settings->get_setting('grfwp-styling-review-text-font-size') . ' !important; }'; }
			if ( $grfwp_controller->settings->get_setting('grfwp-styling-review-text-color') != '' ) { $styling .=  '.gr-review-body { color: ' . $grfwp_controller->settings->get_setting('grfwp-styling-review-text-color') . ' !important; }'; }
			if ( $grfwp_controller->settings->get_setting('grfwp-styling-author-font-size') != '' ) { $styling .=  '.gr-author-name { font-size: ' . $grfwp_controller->settings->get_setting('grfwp-styling-author-font-size') . ' !important; }'; }
			if ( $grfwp_controller->settings->get_setting('grfwp-styling-author-color') != '' ) { $styling .=  '.gr-author-name { color: ' . $grfwp_controller->settings->get_setting('grfwp-styling-author-color') . ' !important; }'; }
			if ( $grfwp_controller->settings->get_setting('grfwp-styling-organization-font-size') != '' ) { $styling .=  '.gr-author-url { font-size: ' . $grfwp_controller->settings->get_setting('grfwp-styling-organization-font-size') . ' !important; }'; }
			if ( $grfwp_controller->settings->get_setting('grfwp-styling-organization-color') != '' ) { $styling .=  '.gr-author-url { color: ' . $grfwp_controller->settings->get_setting('grfwp-styling-organization-color') . ' !important; }'; }
			if ( $grfwp_controller->settings->get_setting('grfwp-styling-read-more-text-color') != '' ) { $styling .=  '.gr-review-url { color: ' . $grfwp_controller->settings->get_setting('grfwp-styling-read-more-text-color') . ' !important; }'; }
			if ( $grfwp_controller->settings->get_setting('grfwp-styling-read-more-background-color') != '' ) { $styling .=  '.gr-review-url { background: ' . $grfwp_controller->settings->get_setting('grfwp-styling-read-more-background-color') . ' !important; }'; }
			if ( $grfwp_controller->settings->get_setting('grfwp-styling-show-more-background-color') != '' ) { $styling .=  '#grfwp-pagination-load-more { background: ' . $grfwp_controller->settings->get_setting('grfwp-styling-show-more-background-color') . ' !important; }'; }
			if ( $grfwp_controller->settings->get_setting('grfwp-styling-show-more-background-hover-color') != '' ) { $styling .=  '#grfwp-pagination-load-more:hover, #grfwp-pagination-load-more:active { background: ' . $grfwp_controller->settings->get_setting('grfwp-styling-show-more-background-hover-color') . ' !important; }'; }
			if ( $grfwp_controller->settings->get_setting('grfwp-styling-show-more-border-color') != '' ) { $styling .=  '#grfwp-pagination-load-more { border-color: ' . $grfwp_controller->settings->get_setting('grfwp-styling-show-more-border-color') . ' !important; }'; }
			if ( $grfwp_controller->settings->get_setting('grfwp-styling-show-more-text-color') != '' ) { $styling .=  '#grfwp-pagination-load-more { color: ' . $grfwp_controller->settings->get_setting('grfwp-styling-show-more-text-color') . ' !important; }'; }
			if ( $grfwp_controller->settings->get_setting('grfwp-styling-icon-color') != '' ) { $styling .=  '.gr-author-icon:before, .gr-organization-icon:before, .gr-review-date-icon:before { color: ' . $grfwp_controller->settings->get_setting('grfwp-styling-icon-color') . ' !important; }'; }
		$styling .=   '</style>';
		return $styling;
	}
}


if ( ! function_exists( 'fsp_add_frontend_ajax_url' ) ) {
function fsp_add_frontend_ajax_url() { ?>
    
    <script type="text/javascript">
        var ajaxurl = '<?php echo admin_url('admin-ajax.php'); ?>';
    </script>
<?php }
}